import { Component, ChangeDetectionStrategy, output, input, OnInit, ViewEncapsulation, inject } from '@angular/core';
import { tap, map, catchError, switchMap, shareReplay, retry } from 'rxjs/operators';
import { of, BehaviorSubject, combineLatest, zip } from 'rxjs';
import { MessageService } from '../../_shared/services/message.service';
import * as util from '../../_shared/utils/util'
import { ContactInfoService, MainContractContactDetail, RequiredData } from './contact-info.service';
import { CustomFormBuilder } from '../../_shared/services/custom-form-builder.service';
import { ContactContractHelper } from './contract-contact-helper';
import { InfoType } from '../main-contract.service';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../../app.config';
import { HesitateDirective } from '../../_shared/directives/hesitate-directive';

@Component({
  selector: 'app-contact-info',
  templateUrl: './contact-info.component.html',
  styleUrl: './contact-info.component.scss',
  changeDetection: ChangeDetectionStrategy.OnPush,
  encapsulation: ViewEncapsulation.None,
  imports: [FAST_PAGE_COMMON, FAST_KENDO_COMMON, HesitateDirective]
})
export class ContactInfoComponent implements OnInit {
  contractId = input.required<number>();
  entityId = input.required<number>();
  productId = input.required<number>();
  entityType = input.required<InfoType>();
  saveClicked = output();
  closeClicked = output();

  private service = inject(ContactInfoService);
  private messageService = inject(MessageService);
  private fb = inject(CustomFormBuilder);

  constructor() {

  }
  util = util;
  icons = util.icons;
  localRequiredData: RequiredData;
  hasModifyPermission = false;
  ContractHelper = new ContactContractHelper();
  contactInfoDetailForm = this.getContactInfoDetailForm();
  refreshRequiredData$ = new BehaviorSubject(null);
  loading$ = new BehaviorSubject<boolean>(true);
  refreshContact$ = new BehaviorSubject<void>(null);

  ngOnInit(): void {
    this.refreshContact$.next();
  }

  getContactInfoDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<MainContractContactDetail> = fb.group({
      commercialAddressLine1: fb.ctr(null),
      commercialAddressLine2: fb.ctr(null),
      commercialCountryId: fb.ctr(1),
      commercialAttn: fb.ctr(null),
      commercialEmailAddress: fb.ctr(null),
      confirmationsAddressLine1: fb.ctr(null),
      confirmationsAddressLine2: fb.ctr(null),
      confirmationsCountryId: fb.ctr(1),
      confirmationsAttn: fb.ctr(null),
      confirmationsEmailAddress: fb.ctr(null),
      creditAddressLine1: fb.ctr(null),
      creditAddressLine2: fb.ctr(null),
      creditCountryId: fb.ctr(1),
      creditAttn: fb.ctr(null),
      creditEmailAddress: fb.ctr(null),
      noticesAddressLine1: fb.ctr(null),
      noticesAddressLine2: fb.ctr(null),
      noticesCountryId: fb.ctr(1),
      noticesAttn: fb.ctr(null),
      noticesEmailAddress: fb.ctr(null),
      schedulingAddressLine1: fb.ctr(null),
      schedulingAddressLine2: fb.ctr(null),
      schedulingCountryId: fb.ctr(1),
      schedulingAttn: fb.ctr(null),
      schedulingEmailAddress: fb.ctr(null),
      commercialCity: fb.ctr(null),
      commercialTelephoneNum: fb.ctr(null),
      commercialZip: fb.ctr(null),
      commercialStateId: fb.ctr(null),
      confirmationsCity: fb.ctr(null),
      commercialFaxNum: fb.ctr(null),
      confirmationsFaxNum: fb.ctr(null),
      confirmationsStateId: fb.ctr(null),
      creditCity: fb.ctr(null),
      confirmationsZip: fb.ctr(null),
      confirmationsTelephoneNum: fb.ctr(null),
      creditFaxNum: fb.ctr(null),
      creditStateId: fb.ctr(null),
      creditTelephoneNum: fb.ctr(null),
      creditZip: fb.ctr(null),
      doEmail: fb.ctr(false),
      doFax: fb.ctr(false),
      combineEmail: fb.ctr(false),
      noticesCity: fb.ctr(null),
      noticesFaxNum: fb.ctr(null),
      noticesStateId: fb.ctr(null),
      noticesTelephoneNum: fb.ctr(null),
      noticesZip: fb.ctr(null),
      schedulingCity: fb.ctr(null),
      schedulingFaxNum: fb.ctr(null),
      schedulingStateId: fb.ctr(null),
      schedulingTelephoneNum: fb.ctr(null),
      schedulingZip: fb.ctr(null)
    });
    return fg;
  }

  pasteInOtherFields() {
    this.ContractHelper.PasteFromCommercial(this.contactInfoDetailForm);
  }

  copyToScheduling() {
    this.ContractHelper.CopyCommercialToScheduling(this.contactInfoDetailForm);
  }

  copyToNotices() {
    this.ContractHelper.CopyCommercialToNotices(this.contactInfoDetailForm);
  }

  copyToCredit() {
    this.ContractHelper.CopyCommercialToCredit(this.contactInfoDetailForm);
  }

  copyToConfirmations() {
    this.ContractHelper.CopyCommercialToConfirmations(this.contactInfoDetailForm);
  }

  saveContactInfo() {
    this.saveClicked.emit();
  }

  closeContactInfo() {
    this.closeClicked.emit();
  }

  getCommercialInfo(): MainContractContactDetail {
    const commercialItems = this.contactInfoDetailForm.value as MainContractContactDetail;
    return commercialItems;
  }

  requiredData$ = this.refreshRequiredData$.pipe(
    tap(() => this.loading$.next(true)),
    switchMap(refreshType => {
      return combineLatest([this.service.requiredData$, of(refreshType)]);
    }),
    map(([requiredData]) => {
      this.localRequiredData = requiredData;
      return requiredData;
    }),
    tap((requiredData) => {
      this.hasModifyPermission = requiredData.hasModifyPermission;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  contact$ = this.refreshContact$.pipe(
    tap(() => {
      this.loading$.next(true);
    }),
    switchMap(() => {
      const getInternalSide: boolean = this.entityType() === InfoType.Internal;
      return this.service.getContactInfo(this.contractId(), getInternalSide, this.entityId(), this.productId());
    }),
    map(detail => {
      this.contactInfoDetailForm.setValue(detail);
      return detail;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  loadingComplete$ = zip(this.contact$, this.requiredData$).pipe(
    tap(() => {
      this.loading$.next(false);
    })
  )

  filterStates$ = new BehaviorSubject<string>(null)
  states$ = this.filterStates$.pipe(util.filterIdNames(this.loading$, this.requiredData$, 'states'));

  filterCountries$ = new BehaviorSubject<string>(null)
  countries$ = this.filterCountries$.pipe(util.filterIdNames(this.loading$, this.requiredData$, 'countries'));
}
