import { Component, ChangeDetectionStrategy, output, input, OnInit, ViewChild, ViewEncapsulation, ChangeDetectorRef, computed, signal, inject } from '@angular/core';
import { tap, map, catchError, switchMap, filter, shareReplay, retry } from 'rxjs/operators';
import { of, BehaviorSubject, combineLatest, zip } from 'rxjs';
import { MessageService } from '../../_shared/services/message.service';
import * as util from '../../_shared/utils/util'
import { AccountingInfoService, MainContractAccountingDetail, RequiredData, PayOptions } from './accounting-info.service';
import { CustomFormBuilder } from '../../_shared/services/custom-form-builder.service';
import { ExpansionPanelComponent, KENDO_EXPANSIONPANEL } from '@progress/kendo-angular-layout';
import { MainContractContactDetail } from '../contact-info/contact-info.service';
import { InfoType } from '../main-contract.service';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../../app.config';
import { KENDO_MASKEDTEXTBOX } from '@progress/kendo-angular-inputs';

@Component({
  selector: 'app-accounting-info',
  templateUrl: './accounting-info.component.html',
  styleUrl: './accounting-info.component.scss',
  changeDetection: ChangeDetectionStrategy.OnPush,
  encapsulation: ViewEncapsulation.None,
  imports: [FAST_PAGE_COMMON, FAST_KENDO_COMMON, KENDO_EXPANSIONPANEL, KENDO_MASKEDTEXTBOX]
})
export class AccountingInfoComponent implements OnInit {
  @ViewChild('wirePanel') wirePanelElem: ExpansionPanelComponent;
  @ViewChild('achPanel') achPanelElem: ExpansionPanelComponent;
  @ViewChild('checkPanel') checkPanelElem: ExpansionPanelComponent;

  contractId = input.required<number>();
  entityId = input.required<number>();
  productId = input.required<number>();
  entityType = input.required<InfoType>();
  saveClicked = output();
  closeClicked = output();
  copyFromCommercialClicked = output();
  hasNaesb = computed(() => { return this.productId() === 3 });

  private service = inject(AccountingInfoService);
  private messageService = inject(MessageService);
  private fb = inject(CustomFormBuilder);
  private ref = inject(ChangeDetectorRef);

  constructor() {

  }

  util = util;
  icons = util.icons;
  localRequiredData: RequiredData;
  hasModifyPermission = false;
  accountingInfoDetailForm = this.getAccountingInfoDetailForm();
  refreshRequiredData$ = new BehaviorSubject(null);
  loading$ = new BehaviorSubject<boolean>(true);
  refreshAccounting$ = new BehaviorSubject<void>(null);
  hasWireInfo = signal(false);
  hasAchInfo = signal(false);
  hasChecksInfo = signal(false);

  ngOnInit(): void {
    this.refreshAccounting$.next();
  }

  getAccountingInfoDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<MainContractAccountingDetail> = fb.group({
      isExhibitCChecked: fb.ctr(false),
      achAbaNum: fb.ctr(null),
      achAccountNum: fb.ctr(null),
      achBank: fb.ctr(null),
      achOtherDetails: fb.ctr(null),
      checksAddressLine1: fb.ctr(null),
      checksAddressLine2: fb.ctr(null),
      checksCountryId: fb.ctr(1),
      checksAttn: fb.ctr(null),
      invoiceAddressLine1: fb.ctr(null),
      invoiceAddressLine2: fb.ctr(null),
      invoiceCountryId: fb.ctr(1),
      invoiceAttn: fb.ctr(null),
      invoiceEmailAddress: fb.ctr(null),
      wireAbaNum: fb.ctr(null),
      wireAccountNum: fb.ctr(null),
      checksCity: fb.ctr(null),
      checksStateId: fb.ctr(null),
      checksZip: fb.ctr(null),
      doEmail: fb.ctr(false),
      doFax: fb.ctr(false),
      invoiceCity: fb.ctr(null),
      invoiceFaxNum: fb.ctr(null),
      invoiceStateId: fb.ctr(null),
      invoiceTelephoneNum: fb.ctr(null),
      invoiceZip: fb.ctr(null),
      wireBank: fb.ctr(null),
      wireOtherDetails: fb.ctr(null),
      isAchChecked: fb.ctr(true),
      isChecksChecked: fb.ctr(false),
      isWireChecked: fb.ctr(true)
    });
    return fg;
  }

  saveAccountingInfo() {
    this.saveClicked.emit();
  }

  getAccountingInfo(): MainContractAccountingDetail {
    const accountingItems = this.accountingInfoDetailForm.value as MainContractAccountingDetail;
    return accountingItems;
  }

  closeAccountingInfo() {
    this.closeClicked.emit();
  }

  copyFromCommercial(): void {
    this.copyFromCommercialClicked.emit();
  }

  pasteCommercial(contactDetail: MainContractContactDetail): void {
    this.accountingInfoDetailForm.patchValue({ invoiceAddressLine1: contactDetail.commercialAddressLine1 });
    this.accountingInfoDetailForm.patchValue({ invoiceAddressLine2: contactDetail.commercialAddressLine2 });
    this.accountingInfoDetailForm.patchValue({ invoiceCountryId: contactDetail.commercialCountryId });
    this.accountingInfoDetailForm.patchValue({ invoiceCity: contactDetail.commercialCity });
    this.accountingInfoDetailForm.patchValue({ invoiceStateId: contactDetail.commercialStateId });
    this.accountingInfoDetailForm.patchValue({ invoiceZip: contactDetail.commercialZip });
    this.accountingInfoDetailForm.patchValue({ invoiceAttn: contactDetail.commercialAttn });
    this.accountingInfoDetailForm.patchValue({ invoiceTelephoneNum: contactDetail.commercialTelephoneNum });
    this.accountingInfoDetailForm.patchValue({ invoiceFaxNum: contactDetail.commercialFaxNum });
    this.accountingInfoDetailForm.patchValue({ invoiceEmailAddress: contactDetail.commercialEmailAddress });
    this.accountingInfoDetailForm.markAsDirty();
  }

  updateExhibitC(addendumExhibitC: boolean): void {
    this.accountingInfoDetailForm.patchValue({ isExhibitCChecked: addendumExhibitC });
  }

  requiredData$ = this.refreshRequiredData$.pipe(
    tap(() => {
      this.loading$.next(true);
    }),
    switchMap(refreshType => {
      return combineLatest([this.service.requiredData$, of(refreshType)]);
    }),
    map(([requiredData]) => {
      this.localRequiredData = requiredData;
      return requiredData;
    }),
    tap((requiredData) => {
      this.hasModifyPermission = requiredData.hasModifyPermission;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  payInfoChanged$ = this.accountingInfoDetailForm.valueChanges.pipe(
    tap(detail => {
      const wireText = [detail.wireBank, detail.wireAbaNum, detail.wireAccountNum].join('');
      this.hasWireInfo.set(!util.isNullOrWhitespace(wireText));

      const achText = [detail.achBank, detail.achAbaNum, detail.achAccountNum].join('');
      this.hasAchInfo.set(!util.isNullOrWhitespace(achText));

      const checksText = [detail.checksAttn, detail.checksAddressLine1, detail.checksAddressLine2, detail.checksCity, detail.checksStateId, detail.checksZip].join('');
      this.hasChecksInfo.set(!util.isNullOrWhitespace(checksText));
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(1)
  )

  isWireCheckedChanged$ = this.accountingInfoDetailForm.get('isWireChecked').valueChanges.pipe(
    filter(() => {
      return !this.loading$.value;
    }),
    tap(() => {
      const checkedOptions = this.getCheckedPayOptions();
      this.setPanelExpandState(this.wirePanelElem, checkedOptions.isWireChecked);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(1)
  )

  isAchCheckedChanged$ = this.accountingInfoDetailForm.get('isAchChecked').valueChanges.pipe(
    filter(() => {
      return !this.loading$.value;
    }),
    tap(() => {
      const checkedOptions = this.getCheckedPayOptions();
      this.setPanelExpandState(this.achPanelElem, checkedOptions.isAchChecked);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(1)
  )

  isChecksCheckedChanged$ = this.accountingInfoDetailForm.get('isChecksChecked').valueChanges.pipe(
    filter(() => {
      return !this.loading$.value;
    }),
    tap(() => {
      const checkedOptions = this.getCheckedPayOptions();
      this.setPanelExpandState(this.checkPanelElem, checkedOptions.isChecksChecked);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(1)
  )

  getCheckedPayOptions = (): PayOptions => {
    return {
      isWireChecked: this.accountingInfoDetailForm.get('isWireChecked').value,
      isAchChecked: this.accountingInfoDetailForm.get('isAchChecked').value,
      isChecksChecked: this.accountingInfoDetailForm.get('isChecksChecked').value
    }
  }

  accounting$ = this.refreshAccounting$.pipe(
    tap(() => {
      this.loading$.next(true);
    }),
    switchMap(() => {
      const getInternalSide: boolean = this.entityType() === InfoType.Internal;
      return this.service.getAccountingInfo(this.contractId(), getInternalSide, this.entityId(), this.productId());
    }),
    map(detail => {
      this.accountingInfoDetailForm.setValue(detail);
      return detail;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  loadingComplete$ = zip(this.accounting$, this.requiredData$).pipe(
    tap(() => {
      this.loading$.next(false);
    })
  )

  filterStates$ = new BehaviorSubject<string>(null)
  states$ = this.filterStates$.pipe(util.filterIdNames(this.loading$, this.requiredData$, 'states'));

  filterCountries$ = new BehaviorSubject<string>(null)
  countries$ = this.filterCountries$.pipe(util.filterIdNames(this.loading$, this.requiredData$, 'countries'));

  setPanelExpandState(expansionPanel: ExpansionPanelComponent, isExpanding: boolean): void {
    const isExhibitCChecked = this.accountingInfoDetailForm.get('isExhibitCChecked').value;
    if (!isExhibitCChecked) {
      setTimeout(() => {
        expansionPanel.expanded = isExpanding;
        //detectChanges helps with the panel not completely expanding/collapsing until you click away
        //it also helps with misclicks on checkboxes since the panel may move slightly if you don't have detectChanges
        this.ref.detectChanges();
      }, 100);
    }
  }
}
