import { Component, ChangeDetectionStrategy, ChangeDetectorRef, ViewEncapsulation, HostListener, OnInit, inject } from '@angular/core';
import { Title } from '@angular/platform-browser';
import { tap, catchError, switchMap, shareReplay, retry } from 'rxjs/operators';
import { of, BehaviorSubject, Subject, Observable } from 'rxjs';
import { GasTrackService, RequiredData } from './gastrack.service';
import { MessageService } from '../_shared/services/message.service';
import { AbstractControl, UntypedFormBuilder, UntypedFormGroup, Validators } from '@angular/forms';
import { ToastService } from '../_shared/services/fast-toast.service';
import * as util from '../_shared/utils/util';
import dayjs from 'dayjs';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../app.config';

function ValidateDate(c: AbstractControl): { [key: string]: boolean } | null {
  if (c.parent) {
    const startDate: Date = c.parent.get('startDate').value;
    const endDate: Date = c.parent.get('endDate').value;
    if (endDate < startDate) {
      return { 'InvalidDate': true };
    }
    return null;
  } else return null;
}

@Component({
  selector: 'app-gastrack',
  templateUrl: './gastrack.component.html',
  encapsulation: ViewEncapsulation.None,
  changeDetection: ChangeDetectionStrategy.OnPush,
  imports: [FAST_PAGE_COMMON, FAST_KENDO_COMMON]
})
export class GasTrackComponent implements OnInit {
  @HostListener('window:resize') onResize() {
    //this function is empty but for some reason it helps the window to resize faster
  };

  exporting$ = new BehaviorSubject<boolean>(false)
  exportClicked$ = new Subject()
  refreshRequiredData$ = new BehaviorSubject(null)

  util = util;
  // icons = util.icons;
  Form: UntypedFormGroup;
  hasGasTrackStandardPermission = false;
  localRequiredData: RequiredData;

  requiredData$: Observable<RequiredData>;
  exportAction$: Observable<{ fileBlob: Blob; fileName: string }>;

  private messageService = inject(MessageService);
  private titleService = inject(Title);
  private service = inject(GasTrackService);
  private fb = inject(UntypedFormBuilder);
  private ref = inject(ChangeDetectorRef);
  private notify = inject(ToastService);

  constructor() {
    this.requiredData$ = this.refreshRequiredData$.pipe(
      switchMap(() => {
        return this.service.requiredData$
      }),
      tap(requiredData => {
        this.localRequiredData = requiredData;
        this.hasGasTrackStandardPermission = requiredData.hasGasTrackStandardPermission;
        if (this.hasGasTrackStandardPermission) {
          this.Form.enable();
          // util.focusAndSelectInputTarget();
        }
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService);
      }), retry(10)
    )

    this.exportAction$ = this.exportClicked$.pipe(
      tap(() => {
        this.exporting$.next(true);
      }),
      switchMap(() => {
        util.clearSelection();
        const startDate = this.Form.get('startDate').value;
        const endDate = this.Form.get('endDate').value;
        return this.service.export(startDate, endDate);
      }),
      tap(res => {
        util.openOrSaveFile(res.fileBlob, res.fileName);
        this.exporting$.next(false);
      }),
      shareReplay(1),
      catchError(err => {
        this.exporting$.next(false);
        return util.handleError(err, this.messageService);
      }), retry(10)
    )
  }

  ngOnInit(): void {
    this.Form = this.fb.group({
      startDate: [dayjs().startOf('month').toDate(), [Validators.required, ValidateDate]],
      endDate: [dayjs().endOf('month').startOf('day').toDate(), [Validators.required, ValidateDate]]
    });
    this.Form.disable();
  }

  export(): void {
    if (this.Form.valid)
      this.exportClicked$.next(null);
    else
      this.notify.error("validation failed");
  }

  title$ = of('GasTrack').pipe(
    tap((title) => util.trySetTitle(this.titleService, title))
  )

  startDateValueChange() {
    const startDateControl = this.Form.get('startDate');
    if (startDateControl.valid) {
      this.Form.patchValue({ endDate: dayjs(startDateControl.value).endOf('month').toDate() });
    }
  }
}
