import { inject, Injectable } from '@angular/core';
import { HttpClient, HttpHeaders } from '@angular/common/http';
import { OdataGridHelper } from '../_shared/utils/odata-grid-helper';
import { Observable } from 'rxjs';
import { State } from '@progress/kendo-data-query';
import { GridDataResult } from '@progress/kendo-angular-grid';
import { DealFilter } from '../deal-filter/deal-filter.service';
import * as util from '../_shared/utils/util';
import * as models from './models';
import { map } from 'rxjs/operators';
import dayjs from 'dayjs';
import { convertDatesToDateOnlyStrings } from '../_shared/utils/util';

export enum MergeType {
  Confirms,
  Tickets
}

export interface SaveDealResult {
  dealId: number;
  dealNum: string;
}

export interface DistributeResponse {
  hasErrors: boolean;
  message: string;
}

export enum SaveType {
  New = 1,
  Normal = 2
}

@Injectable({
  providedIn: 'root'
})
export class DealService extends OdataGridHelper {
  private baseUrl = `${window.location.origin}/api/Deal`;
  http = inject(HttpClient);

  requiredData$ = this.http.get<models.RequiredData>(`${this.baseUrl}/GetRequiredData`)

  getDeals(state: State, filterId: number): Observable<GridDataResult> {
    const extraParamsQueryStr = `&filterId=${filterId.toString()}`
    return this.fetch(`GetDealItems`, state, extraParamsQueryStr, false);
  }

  getDealDetail(dealId: number): Observable<models.DealDetail> {
    const url = `${this.baseUrl}/GetDealDetail?dealId=${dealId}`;
    return this.http.get<models.DealDetail>(url);
  }

  getWaspNums(counterpartyId: number, startDate: Date, addNewWaspNum: boolean): Observable<string[]> {
    const startDateStr = dayjs(startDate).format("YYYY-MM-DD");
    const url = `${this.baseUrl}/GetWaspNums?counterpartyId=${counterpartyId}&startDate=${startDateStr}&addNewWaspNum=${addNewWaspNum}`;
    return this.http.get<string[]>(url);
  }

  getContacts(): Observable<models.ContactInfo[]> {
    const url = `${this.baseUrl}/GetContacts`;
    return this.http.get<models.ContactInfo[]>(url);
  }

  getWaspVolAndPrice(dealNum: string, waspNum: string, productId: number, numOfContracts: number, fixedPrice: number, brokerId: number, isBuy: boolean, accountingMonth: Date, startDate: Date): Observable<models.WaspVolPriceResult> {
    const accountingMonthStr = accountingMonth ? dayjs(accountingMonth).format("YYYY-MM-DD") : '';
    const startDateStr = startDate ? dayjs(startDate).format("YYYY-MM-DD") : '';
    const productIdStr = productId ?? '';
    const numOfContractsStr = numOfContracts ?? '';
    const fixedPriceStr = fixedPrice ?? '';
    const brokerIdStr = brokerId ?? '';
    const url = `${this.baseUrl}/GetWaspVolAndPrice?dealNum=${dealNum}&waspNum=${waspNum}&productId=${productIdStr}&numOfContracts=${numOfContractsStr}&fixedPrice=${fixedPriceStr}&brokerId=${brokerIdStr}&isBuy=${isBuy}&accountingMonth=${accountingMonthStr}&startDate=${startDateStr}`;
    return this.http.get<models.WaspVolPriceResult>(url);
  }

  saveDealDetail(dealDetail: models.DealDetail, saveType: SaveType): Observable<SaveDealResult> {
    const url = `${this.baseUrl}/SaveDealDetail?saveType=${saveType}`;
    dealDetail = convertDatesToDateOnlyStrings(dealDetail, ['accountingMonth', 'startDate', 'endDate', 'tradingDate'], true);
    return this.http.put<SaveDealResult>(url, dealDetail);
  }

  saveSourceDelivery(dealDetail: models.DealDetail): Observable<SaveDealResult> {
    const url = `${this.baseUrl}/SaveSourceDelivery`;
    dealDetail = convertDatesToDateOnlyStrings(dealDetail, ['accountingMonth', 'startDate', 'endDate', 'tradingDate'], true);
    return this.http.put<SaveDealResult>(url, dealDetail);
  }

  uniqueDealCheck(dealDetail: models.DealDetail, saveType: SaveType): Observable<string> {
    const url = `${this.baseUrl}/UniqueDealCheck?saveType=${saveType}`;
    dealDetail = convertDatesToDateOnlyStrings(dealDetail, ['accountingMonth', 'startDate', 'endDate', 'tradingDate'], true);
    return this.http.put(url, dealDetail, { observe: 'response', responseType: 'text' }).pipe(
      map(result => {
        return result.body;
      })
    );
  }

  deleteDeal(id: number) {
    const url = `${this.baseUrl}/DeleteDeal/${id}`;
    return this.http.delete(url);
  }

  exportDeals(state: State, filterId: number, fileNameOriginal: string) {
    const extraParamsQueryStr = `&filterId=${filterId.toString()}`
    return this.fetchExport(`GetDealItems`, state, extraParamsQueryStr, fileNameOriginal);
  }

  saveGridSettings(state: State, filterId: number): Observable<DealFilter> {
    const headers = new HttpHeaders({ 'Content-Type': 'application/json' });
    const url = `${this.baseUrl}/SaveGridSettings?filterId=${filterId}`;
    //stringify twice so that we can send a JSON string inside JSON content
    const stateJson: string = JSON.stringify(JSON.stringify(state));
    return this.http.put<DealFilter>(url, stateJson, { headers });
  }

  downloadConfirm(ticketNum: string): Observable<{ fileBlob: Blob; fileName: string }> {
    const url = `${this.baseUrl}/DownloadConfirm?ticketNum=${ticketNum}`;
    return util.downloadFile(this.http, url);
  }

  downloadTicket(ticketNum: string): Observable<{ fileBlob: Blob; fileName: string }> {
    const url = `${this.baseUrl}/DownloadTicket?ticketNum=${ticketNum}`;
    return util.downloadFile(this.http, url);
  }

  mergeFiles(ticketNums: string[], mergeType: MergeType): Observable<{ fileBlob: Blob; fileName: string }> {
    const typeStr = mergeType === MergeType.Confirms ? 'confirms' : 'tickets';
    const url = `${this.baseUrl}/MergeFiles?type=${typeStr}`;
    return util.downloadFileWithBody(this.http, url, ticketNums);
  }

  distribute(ticketNums: string[]): Observable<DistributeResponse> {
    const url = `${this.baseUrl}/Distribute`;
    return this.http.put<DistributeResponse>(url, ticketNums);
  }

}
