import { Component, ChangeDetectionStrategy, ElementRef, ChangeDetectorRef, ViewEncapsulation, ViewChild, HostListener, AfterViewInit, ViewChildren, TemplateRef, inject } from '@angular/core';
import { Title } from '@angular/platform-browser';
import { tap, take, catchError, switchMap, shareReplay, filter, map, debounceTime, delay, retry } from 'rxjs/operators';
import { of, BehaviorSubject, Subject, combineLatest, Subscription, zip } from 'rxjs';
import { DealService, MergeType, SaveType } from './deal.service';
import { DealFilterService, DealFilter, DisplayInfo } from '../deal-filter/deal-filter.service';
import { MessageService, promptAction, PromptSettings } from '../_shared/services/message.service';
import { Validators, AbstractControl, FormArray } from '@angular/forms';
import { CustomFormBuilder } from '../_shared/services/custom-form-builder.service';
import { DialogService, DialogCloseResult, DialogSettings, WindowComponent } from '@progress/kendo-angular-dialog';
import { WindowState } from '../_shared/elements/fast-window.component';
import { ToastService } from '../_shared/services/fast-toast.service';
import * as util from '../_shared/utils/util';
import { CompositeFilterDescriptor, State } from '@progress/kendo-data-query';
import { TooltipDirective } from '@progress/kendo-angular-tooltip';
import { SelectableSettings, GridComponent, RowClassArgs } from '@progress/kendo-angular-grid';
import { ActivatedRoute } from '@angular/router';
import * as models from './models';
import { BookInfo } from './models/BookInfo';
import { ComboBoxComponent, KENDO_DROPDOWNLIST } from '@progress/kendo-angular-dropdowns';
import { formatNumber, formatDate } from '@progress/kendo-intl';
import dayjs from 'dayjs';
import { ContextMenuSelectEvent, KENDO_CONTEXTMENU } from '@progress/kendo-angular-menu';
import { PipeInfo } from './models/PipeInfo';
import { HttpParams } from '@angular/common/http';
import { PreventableEvent } from '@progress/kendo-angular-dropdowns';
import { CommonService } from '../_shared/services/common.service';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../app.config';
import { HesitateDirective } from '../_shared/directives/hesitate-directive';
import { KENDO_POPUP } from '@progress/kendo-angular-popup';
import { KENDO_DATEINPUT } from '@progress/kendo-angular-dateinputs';
import { DealFilterComponent } from '../deal-filter/deal-filter.component';
import { Product } from './models/Product';
import { FastGridComponent } from '../_shared/elements/fast-grid.component';

enum Mode {
  Deal,
  Distribute
}

interface volumeTooltipItem {
  label: string;
  valueText: string;
}

enum contextMenuEnum {
  CopyUp,
  CopyDown
}

interface ContextMenuItem {
  enum: contextMenuEnum;
  text: string;
  cssClass: string;
}

function ValidateMasterContract(instance: DealComponent) {
  return (c: AbstractControl): { [key: string]: boolean } | null => {
    if (c.parent) {
      const contractCtrl = c.parent.get('contractNumber');
      const transactionTypeId = c.parent.get('transactionTypeId').value;

      if (instance.localRequiredData && transactionTypeId) {
        const physicalTransTypeIds = [1, 9, 11] //Physical Gas, Physical NGL, Physical Crude Oil
        const isPhysical = physicalTransTypeIds.includes(transactionTypeId);

        //only check master contract for physicals
        if (isPhysical) {
          if (contractCtrl.value) {
            const masterContractInfo = instance.localRequiredData.contracts.find(item => item.contractNumber === contractCtrl.value);
            //if inactive contract
            if (masterContractInfo && !masterContractInfo.isActive)
              return { 'contractNumber': true };
          }
          else //missing contract
            return { 'contractNumber': true };
        }
      }
    }
    return null;
  }
}

@Component({
  selector: 'app-deal',
  templateUrl: './deal.component.html',
  styleUrls: ['./deal.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush,
  encapsulation: ViewEncapsulation.None,
  imports: [FAST_PAGE_COMMON, FAST_KENDO_COMMON, KENDO_DROPDOWNLIST, KENDO_POPUP, KENDO_DATEINPUT, KENDO_CONTEXTMENU, DealFilterComponent, HesitateDirective]
})
export class DealComponent implements AfterViewInit {
  @ViewChildren(TooltipDirective) allTips: TooltipDirective[];
  @ViewChild('tooltipGrid') tooltipGrid: TooltipDirective;
  @ViewChild('tooltipInputItems') tooltipInputItems: TooltipDirective;
  @ViewChild('modeToggleTarget') modeToggleElem: ElementRef;
  // @ViewChild("copyDealNumElem") copyDealNumElem: ElementRef;
  @ViewChild("pipelineComboElem") pipelineComboElem: ComboBoxComponent;
  // @ViewChild("grid", { read: ElementRef }) kendoGridEl: ElementRef;
  @ViewChild("grid") fastGrid: FastGridComponent;
  @ViewChild(GridComponent) grid: GridComponent;
  @ViewChild("detailWindow") detailWindow: WindowComponent;
  @HostListener('window:resize') onResize() {
    //this function is empty but for some reason it helps the window to resize faster
  };
  // detailPointsForm = this.getDetailPointsForm();
  detailPointsForm: util.FormModel<models.SourceDeliveryPointItem>
  exporting = false;
  formEditor: util.FormModel<models.DealDetail>;
  formEditorVariableVolumes: FormArray;
  filterEditorOpened = false;
  hasDealModifyPermission = false;
  hasDealForAccountingPermission = false;
  hasDealSaveSourceDelivery = false;
  mode: Mode = Mode.Deal;
  dealFilters: DealFilter[];
  distributionViewName = "Distribution View";
  localAllColumnDisplayInfos: Record<string, DisplayInfo> = {};
  visibleColumnInfos: DisplayInfo[];
  localSelectedFilterId: number;
  localRequiredData: models.RequiredData;
  localDealDetail: models.DealDetail;
  savedDealDetail: models.DealDetail;
  localWaspNums: string[];
  state: State = {};
  hasDealNumParam = false;
  loadMode: Mode = null;
  gridScrollTopPos: number;
  gridScrollLeftPos: number;
  isVariableVolumeEditorOpened = false;
  savedDealVolumes: models.DealDetailVolume[] = null;
  minDate: Date = new Date(1900, 0, 1);
  previousStartDate: Date;
  previousEndDate: Date;
  addNewWaspNum = false;
  util = util;
  accountingDealPurposes = [21, 22, 23, 24];
  units = [{ id: 2, name: "$/gallon" }, { id: 5, name: "$/month" }];

  mySelection: string[] = [];
  selectableSettings: SelectableSettings = {
    checkboxOnly: false,
    mode: 'single',
    enabled: true
  }

  contextMenuCopyItems: ContextMenuItem[] = [
    { enum: contextMenuEnum.CopyUp, text: 'Copy Up', cssClass: 'variableVolumeContextMenuItem' },
    { enum: contextMenuEnum.CopyDown, text: 'Copy Down', cssClass: 'variableVolumeContextMenuItem' }
  ];

  loading$ = new BehaviorSubject<boolean>(true)
  gridLoading$ = new BehaviorSubject<boolean>(true)
  editorLoading$ = new BehaviorSubject<boolean>(true) //this needs to be observable for filterIdNames
  windowLoading$ = new BehaviorSubject<boolean>(true)
  editorOpened$ = new BehaviorSubject<boolean>(false)
  isMergingFiles$ = new BehaviorSubject<boolean>(false)
  isDistributing$ = new BehaviorSubject<boolean>(false)
  waspNumsLoading$ = new BehaviorSubject<boolean>(false)
  waspVolPriceLoading$ = new BehaviorSubject<boolean>(false)
  detailPointsOpened$ = new BehaviorSubject<boolean>(false)
  isUpdatePointsOpened$ = new BehaviorSubject<boolean>(false)

  localEditorLoading = true;
  editorLoadingWatcher$ = this.editorLoading$.pipe(
    tap(isLoading => {
      this.localEditorLoading = isLoading
    })
  )

  refreshDeals$ = new BehaviorSubject<string>(null)
  refreshDetail$ = new BehaviorSubject<number>(null)
  refreshRequiredData$ = new Subject<number>()
  refreshWaspVolPrice$ = new Subject<number>()
  refreshWaspNums$ = new Subject<number>()
  refreshProducts$ = new Subject<number>()
  refreshContractsForEntities$ = new Subject<[number, number]>()
  refreshContacts$ = new Subject()
  counterpartyChanged$ = new Subject<number>()
  refreshConfirmButton$ = new BehaviorSubject<number>(0)
  traderChanged$ = new Subject<number>()
  pipelineChanged$ = new Subject<number>()
  pipelineSourceDeliveryChanged$ = new Subject<number>()
  productChanged$ = new Subject<number>()
  brokerChanged$ = new Subject<number>()
  startDateChanged$ = new Subject<Date>()

  filterSelected$ = new Subject<number>()
  exportClicked$ = new Subject()
  saveGridSettings$ = new Subject()

  saveDeal$ = new Subject<[models.DealDetail, SaveType]>()
  saveSourceDelivery$ = new Subject<models.DealDetail>()
  deleteDeal$ = new Subject()
  mergeFiles$ = new Subject<MergeType>()
  distribute$ = new Subject()

  uniqueDealCheck$ = new Subject<[models.DealDetail, SaveType]>()

  downloadConfirm$ = new Subject<string>()
  downloadTicket$ = new Subject<string>()

  filterRegions$ = new BehaviorSubject<string>(null)
  filterStrategies$ = new BehaviorSubject<string>(null)
  filterPortfolios$ = new BehaviorSubject<string>(null)
  filterBooks$ = new BehaviorSubject<string>(null)
  filterCounterparties$ = new BehaviorSubject<string>(null)
  filterInternalEntities$ = new BehaviorSubject<string>(null)
  filterContacts$ = new BehaviorSubject<string>(null)
  filterTraders$ = new BehaviorSubject<string>(null)
  filterDealStatuses$ = new BehaviorSubject<string>(null)
  filterDealPurposes$ = new BehaviorSubject<string>(null)
  filterDealTypes$ = new BehaviorSubject<string>(null)
  filterPriceIndexes$ = new BehaviorSubject<string>(null)
  filterPriceIndexesC2$ = new BehaviorSubject<string>(null)
  filterPriceIndexesC3$ = new BehaviorSubject<string>(null)
  filterPriceIndexesIc4$ = new BehaviorSubject<string>(null)
  filterPriceIndexesNc4$ = new BehaviorSubject<string>(null)
  filterPriceIndexesC5p$ = new BehaviorSubject<string>(null)
  filterPipelines$ = new BehaviorSubject<string>(null)
  filterPoints$ = new BehaviorSubject<string>(null)
  filterPipelineSourceDeliveries$ = new BehaviorSubject<string>(null)
  filterPointSourceDeliveries$ = new BehaviorSubject<string>(null)
  filterContracts$ = new BehaviorSubject<string>(null)
  filterVolumeTypes$ = new BehaviorSubject<string>(null)
  filterFuelCalculationTypes$ = new BehaviorSubject<string>(null)
  filterForceMajeures$ = new BehaviorSubject<string>(null)
  filterWaspNums$ = new BehaviorSubject<string>(null)
  filterProducts$ = new BehaviorSubject<string>(null)
  filterBrokers$ = new BehaviorSubject<string>(null)
  filterBrokerAccounts$ = new BehaviorSubject<string>(null)
  filterPaidOnTypes$ = new BehaviorSubject<string>(null)
  filterDeliveryMethods$ = new BehaviorSubject<string>(null)

  private messageService = inject(MessageService);
  private titleService = inject(Title);
  private dealService = inject(DealService);
  private dealFilterService = inject(DealFilterService);
  private dialogService = inject(DialogService);
  private notify = inject(ToastService);
  private activatedRoute = inject(ActivatedRoute);
  private commonService = inject(CommonService);
  private fb = inject(CustomFormBuilder);
  private ref = inject(ChangeDetectorRef);

  constructor() {
    this.detailPointsForm = this.getDetailPointsForm();
    this.formEditor = this.fb.group({
      id: this.fb.ctr(0, Validators.required),
      dealNum: this.fb.ctr(null),
      accountingMonth: this.fb.ctr(null),
      basis: this.fb.ctr(null),
      bookId: this.fb.ctr(null),
      brokerAccountId: this.fb.ctr(null),
      brokerId: this.fb.ctr(null),
      isBuy: this.fb.ctr(null, Validators.required),
      comments: this.fb.ctr(null),
      contactId: this.fb.ctr(null),
      numOfContracts: this.fb.ctr(null, Validators.required),
      costBasisInc: this.fb.ctr(null),
      costFuelCalculationTypeId: this.fb.ctr(null),
      costPremInc: this.fb.ctr(null),
      counterpartyId: this.fb.ctr(null, Validators.required),
      dealPurposeId: this.fb.ctr(null, Validators.required),
      dealStatusId: this.fb.ctr(null, Validators.required),
      endDate: this.fb.ctr(null, Validators.required),
      fixedPrice: this.fb.ctr(null),
      isFixedPrice: this.fb.ctr(null),
      fmlanguage: this.fb.ctr(null),
      hedgeFee: this.fb.ctr(null),
      hypotheticalId: this.fb.ctr(null, Validators.required),
      internalEntityId: this.fb.ctr(null, Validators.required),
      internalMemo: this.fb.ctr(null),
      isVariableVolume: this.fb.ctr(null, Validators.required),
      contractNumber: this.fb.ctr(null, ValidateMasterContract(this)),
      dealTypeId: this.fb.ctr(null),
      pipelineId: this.fb.ctr(null, Validators.required),
      pipelineSourceDeliveryId: this.fb.ctr(null, Validators.required),
      pointId: this.fb.ctr(null, Validators.required),
      portfolioId: this.fb.ctr(null),
      premiumOrDiscount: this.fb.ctr(null),
      priceIndexId: this.fb.ctr(null),
      priceIndexId2: this.fb.ctr(null),
      regionId: this.fb.ctr(null),
      startDate: this.fb.ctr(null, Validators.required),
      strategyId: this.fb.ctr(null),
      traderId: this.fb.ctr(null, Validators.required),
      tradingDate: this.fb.ctr(null),
      transactionTypeId: this.fb.ctr(null, Validators.required),
      updateNotes: this.fb.ctr(null),
      volume: this.fb.ctr(null, Validators.required),
      volumeTypeId: this.fb.ctr(null, Validators.required),
      waspNum: this.fb.ctr(null),
      dealVolumes: this.fb.arr([]),
      pointSourceDeliveryIds: this.fb.ctr([]),
      created: this.fb.ctr(null),
      modified: this.fb.ctr(null),
      productId: this.fb.ctr(util.Product.NaturalGas),
      volumeC2: this.fb.ctr(null),
      volumeC3: this.fb.ctr(null),
      volumeIc4: this.fb.ctr(null),
      volumeNc4: this.fb.ctr(null),
      volumeC5p: this.fb.ctr(null),
      fixedPriceC2: this.fb.ctr(null),
      fixedPriceC3: this.fb.ctr(null),
      fixedPriceIc4: this.fb.ctr(null),
      fixedPriceNc4: this.fb.ctr(null),
      fixedPriceC5p: this.fb.ctr(null),
      priceIndexIdC2: this.fb.ctr(null),
      priceIndexIdC3: this.fb.ctr(null),
      priceIndexIdIc4: this.fb.ctr(null),
      priceIndexIdNc4: this.fb.ctr(null),
      priceIndexIdC5p: this.fb.ctr(null),
      basisC2: this.fb.ctr(null),
      basisC3: this.fb.ctr(null),
      basisIc4: this.fb.ctr(null),
      basisNc4: this.fb.ctr(null),
      basisC5p: this.fb.ctr(null),
      premDiscC2: this.fb.ctr(null),
      premDiscC3: this.fb.ctr(null),
      premDiscIc4: this.fb.ctr(null),
      premDiscNc4: this.fb.ctr(null),
      premDiscC5p: this.fb.ctr(null),
      inSpecMarketingFee: this.fb.ctr(null),
      inSpecMarketingFeeTypeId: this.fb.ctr(null),
      outSpecMarketingFee: this.fb.ctr(null),
      outSpecMarketingFeeTypeId: this.fb.ctr(null),
      superiorFee: this.fb.ctr(null),
      superiorFeeTypeId: this.fb.ctr(null),
      deliveryModeId: this.fb.ctr(null),
      sourceDeliveryPointItems: this.fb.ctr([], Validators.required),
      paidOnId: this.fb.ctr(null),
      isNetback: this.fb.ctr(null),
      deductTransport: this.fb.ctr(false)
    });
    this.formEditorVariableVolumes = this.formEditor.controls['dealVolumes'] as FormArray;
    this.formEditor.disable();
    this.formEditorVariableVolumes.disable();
  }

  getDetailPointsForm() {
    const fb = this.fb;
    const fg: util.FormModel<models.SourceDeliveryPointItem> = fb.group({
      pointName: fb.ctr(null),
      pointId: fb.ctr(null, Validators.required),
      pointVolume: fb.ctr(null)
    });
    return fg;
  }

  ngAfterViewInit(): void {
    this.activatedRoute.queryParams.subscribe(x => {
      //this reads the query params to decide whether to load into deal or distribute mode
      this.loadMode = x['distribution'] ? Mode.Distribute : Mode.Deal;

      const params = new HttpParams({ fromObject: x });
      const filter: CompositeFilterDescriptor = { filters: [], logic: 'and' };
      const filters = filter.filters;

      const dealNum = params.get('dealNum');

      if (dealNum) {
        this.hasDealNumParam = true;
        filters.push({ field: 'DealNum', operator: 'eq', value: dealNum });
        this.state.filter = filter;
      }

      this.refreshRequiredData$.next(null);
    });
  }

  deal<K extends keyof models.DealDetail>(propName: K): models.DealDetail[K] {
    return this.formEditor.get(propName).value;
  }

  title$ = of('Deal').pipe(
    tap((title) => util.trySetTitle(this.titleService, title))
  )

  getNewVariableVolumeEditorItem(dealVol: models.DealDetailVolume) {
    return this.fb.group({
      startDate: [new Date(dealVol.startDate)],
      physicalVolume: [dealVol.physicalVolume]
    });
  }

  saveGridScrollPos(): void {
    const gridContent = this.getGridContentElement();
    this.gridScrollTopPos = gridContent ? gridContent.scrollTop : 0;
    this.gridScrollLeftPos = gridContent ? gridContent.scrollLeft : 0;
  }

  goToSavedGridScrollPos(): void {
    setTimeout(() => {
      const gridContent = this.getGridContentElement();
      if (gridContent)
        gridContent.scrollTo({ top: this.gridScrollTopPos, left: this.gridScrollLeftPos });
    });
  }

  getGridContentElement(): Element {
    const el = this.fastGrid?.kendoGridElement?.nativeElement as HTMLElement;
    const gridContent = el?.getElementsByClassName("k-grid-content").item(0);
    return gridContent;
  }

  preventOpen(event: PreventableEvent) {
    event.preventDefault();
  }

  get selectionText(): string {
    let text = "";
    if (this.mySelection.length === 1)
      text = "1 deal selected"
    else if (this.mySelection.length > 1)
      text = `${this.mySelection.length} deal selected`

    return text;
  }

  requiredData$ = this.refreshRequiredData$.pipe(
    switchMap(savedFilterId => {
      return combineLatest([of(savedFilterId), this.dealService.requiredData$]);
    }),
    map(([savedFilterId, requiredData]) => {
      this.hasDealModifyPermission = requiredData.hasDealModifyPermission;
      this.hasDealForAccountingPermission = requiredData.hasDealForAccountingPermission;
      this.hasDealSaveSourceDelivery = requiredData.hasDealSaveSourceDelivery;
      this.localRequiredData = requiredData;
      this.dealFilters = requiredData.filters;
      requiredData.allColumnDisplayInfos.forEach(col => {
        this.localAllColumnDisplayInfos[col.propName] = col;
      });
      if (this.loadMode !== null && this.mode !== this.loadMode) {
        this.toggleMode(null);
      }
      else {
        const distributionViewId = this.dealFilters.find(x => x.name === this.distributionViewName).id;
        if (savedFilterId && savedFilterId === distributionViewId && this.mode === Mode.Deal)
          this.toggleMode(savedFilterId);
        else if (savedFilterId && savedFilterId !== distributionViewId && this.mode === Mode.Distribute)
          this.toggleMode(savedFilterId);
        else if (savedFilterId) {
          this.filterSelected$.next(savedFilterId);
          this.loading$.next(false);
        }
        else {
          const selectedFilterId: number = requiredData.filters.find(x => x.isSelected)?.id ?? null;
          this.filterSelected$.next(selectedFilterId);
          this.loading$.next(false);
        }
      }

      this.loadMode = null;
      return requiredData;
    }),
    // tap(() => {
    //   util.focusInputTarget();
    // }),
    shareReplay(1),
    catchError(err => {
      this.loading$.next(false);
      this.gridLoading$.next(false);
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  dealDetail$ = this.refreshDetail$.pipe(
    filter(id => id !== null),
    switchMap(id => {
      if (id === 0)
        return of(this.getNewDealDetail());
      else
        return this.dealService.getDealDetail(id);
    }),
    map(result => {
      this.formEditorVariableVolumes.clear();
      const dealDetail: models.DealDetail = result;
      this.localDealDetail = dealDetail;
      this.savedDealDetail = { ...dealDetail };
      util.convertToDates(dealDetail);
      util.convertToDates(dealDetail.dealVolumes);
      this.savedDealVolumes = dealDetail.dealVolumes ? [...dealDetail.dealVolumes] : [];
      this.previousStartDate = new Date(dealDetail.startDate);
      this.previousEndDate = new Date(dealDetail.endDate);
      this.refreshConfirmButton$.next(dealDetail.dealPurposeId);
      this.setEditorItem(dealDetail);
      return dealDetail
    }),
    tap(() => {
      //trigger dynamic changes
      setTimeout(() => {
        this.counterpartyChanged$.next(this.deal('counterpartyId'));
        this.traderChanged$.next(this.deal('traderId'));
        this.pipelineChanged$.next(this.deal('pipelineId'));
        this.pipelineSourceDeliveryChanged$.next(this.deal('pipelineSourceDeliveryId'));
        this.productChanged$.next(this.deal('productId'));
        this.brokerChanged$.next(this.deal('brokerId'));
        this.startDateChanged$.next(this.deal('startDate'));
        this.refreshContractsForEntities$.next([this.deal('counterpartyId'), this.deal('internalEntityId')]);
        this.refreshWaspNums$.next(null);
        this.filterProducts$.next(null);
      });
    }),
    shareReplay(1),
    catchError(err => {
      this.closeEditor(false);
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  getNewDealDetail(): models.DealDetail {
    const userName = localStorage.getItem('userName');
    const startDate = dayjs().add(1, 'day').startOf('day').toDate();
    const endDate = dayjs(startDate).endOf('month').startOf('day').toDate();

    const deal: models.DealDetail = {
      id: 0,
      dealNum: null,
      isBuy: true,
      startDate: startDate,
      endDate: endDate,
      tradingDate: dayjs().startOf('day').toDate(),
      dealPurposeId: 8, //General
      internalEntityId: 315, //SUPERIOR NATURAL GAS CORPORATION
      dealTypeId: 2, //Baseload
      traderId: this.localRequiredData.traders.find(x => x.userName === userName)?.traderId,
      volumeTypeId: 0,
      costFuelCalculationTypeId: 2, //Price
      isFixedPrice: true,
      isVariableVolume: false,
      dealStatusId: 2, //Completed
      accountingMonth: dayjs().startOf('month').toDate(),
      hypotheticalId: 0,
      transactionTypeId: 1,
      portfolioId: 1, //Unassigned
      regionId: 1, //Unassigned
      strategyId: 1, //Unassigned
      basis: null,
      bookId: null,
      brokerId: null,
      brokerAccountId: null,
      comments: null,
      contactId: null,
      numOfContracts: null,
      costBasisInc: null,
      costPremInc: null,
      counterpartyId: null,
      fixedPrice: null,
      fmlanguage: null,
      hedgeFee: null,
      internalMemo: null,
      contractNumber: null,
      pipelineId: null,
      pipelineSourceDeliveryId: null,
      pointId: null,
      premiumOrDiscount: null,
      priceIndexId: null,
      priceIndexId2: null,
      updateNotes: null,
      volume: null,
      waspNum: null,
      pointSourceDeliveryIds: [],
      created: null,
      modified: null,
      dealVolumes: [],
      productId: 1,
      volumeC2: null,
      volumeC3: null,
      volumeIc4: null,
      volumeNc4: null,
      volumeC5p: null,
      fixedPriceC2: null,
      fixedPriceC3: null,
      fixedPriceIc4: null,
      fixedPriceNc4: null,
      fixedPriceC5p: null,
      priceIndexIdC2: null,
      priceIndexIdC3: null,
      priceIndexIdIc4: null,
      priceIndexIdNc4: null,
      priceIndexIdC5p: null,
      basisC2: null,
      basisC3: null,
      basisIc4: null,
      basisNc4: null,
      basisC5p: null,
      premDiscC2: null,
      premDiscC3: null,
      premDiscIc4: null,
      premDiscNc4: null,
      premDiscC5p: null,
      inSpecMarketingFee: null,
      inSpecMarketingFeeTypeId: 2,
      outSpecMarketingFee: null,
      outSpecMarketingFeeTypeId: 2,
      superiorFee: null,
      superiorFeeTypeId: 2,
      deliveryModeId: 4,
      sourceDeliveryPointItems: [],
      paidOnId: 1,
      isNetback: false,
      deductTransport: false
    };
    return deal;
  }

  dealRequiredData$ = combineLatest([this.requiredData$, this.dealDetail$]).pipe(
    map(([, deal]) => {
      const data = { ...this.localRequiredData };
      data.regions = data.regions.filter(item => item.isActive || item.id === deal.regionId);
      data.strategies = data.strategies.filter(item => item.isActive || item.id === deal.strategyId);
      data.portfolios = data.portfolios.filter(item => item.isActive || item.id === deal.portfolioId);
      data.counterparties = data.counterparties.filter(item => item.isActive || item.counterpartyId === deal.counterpartyId);
      data.internalEntities = data.internalEntities.filter(item => item.isActive || item.id === deal.internalEntityId);
      data.traders = data.traders.filter(item => item.isActive || item.traderId === deal.traderId);
      data.points = data.points.filter(item => item.isActive || item.pointId === deal.pointId);
      data.pointSourceDeliveries = data.pointSourceDeliveries.filter(item => item.isActive || (deal.pointSourceDeliveryIds && deal.pointSourceDeliveryIds.some(pointSourceDeliveryId => item.pointId === pointSourceDeliveryId)));
      data.contracts = data.contracts.filter(item => item.isActive || item.contractNumber === deal.contractNumber);
      return data;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  priceUnit$ = '';

  booksForTrader$ = this.traderChanged$.pipe(
    map(newTraderId => {
      let booksForTrader: BookInfo[] = null;
      if (newTraderId) {
        booksForTrader = this.localRequiredData.books.filter(bookInfo => {
          const isActiveMappedItem = bookInfo.isActive && bookInfo.traderId === newTraderId;
          return isActiveMappedItem;
        });
      }

      //check if saved item exists and if not then add it
      const savedDeal = this.savedDealDetail;
      const isSavedTrader = newTraderId === savedDeal.traderId;
      if (isSavedTrader && booksForTrader && booksForTrader.findIndex(x => x.bookId === savedDeal.bookId) === -1) {
        const savedItem = this.localRequiredData.books.find(x => x.bookId === savedDeal.bookId);
        if (savedItem)
          booksForTrader.push(savedItem);
      }

      if (!booksForTrader || booksForTrader.length === 0)
        booksForTrader = [this.localRequiredData.books.find(bookInfo => bookInfo.bookId === 1)];

      return booksForTrader;
    }),
    tap((booksForTrader) => {
      const bookId = this.deal('bookId');
      if (booksForTrader && booksForTrader.findIndex(bookInfo => bookInfo.bookId === bookId) === -1)
        this.formEditor.patchValue({ bookId: 1 });
    }),
    shareReplay(1)
  )

  newWaspNums$ = this.refreshWaspNums$.pipe(
    delay(0), //delay for dropdown selection changes to finish
    filter(() => {
      const deal: models.DealDetail = this.formEditor.getRawValue();
      return deal.counterpartyId && deal.startDate && deal.transactionTypeId === 3 && deal.dealPurposeId === 9;
    }),
    tap(() => {
      this.waspNumsLoading$.next(true);
      this.waspVolPriceLoading$.next(true);
    }),
    switchMap(() => {
      const deal: models.DealDetail = this.formEditor.getRawValue();
      return this.dealService.getWaspNums(deal.counterpartyId, deal.startDate, this.addNewWaspNum);
    }),
    map(waspNums => {
      this.localWaspNums = waspNums;
      return waspNums;
    }),
    tap(() => {
      const hasSavedWaspNum = this.localWaspNums.findIndex(x => x === this.savedDealDetail.waspNum) !== -1;
      if (this.addNewWaspNum || !hasSavedWaspNum) {
        this.addNewWaspNum = false;
        const monthYear: string = formatDate(this.deal('startDate'), 'MMMyy');
        const waspNums = this.localWaspNums.filter(x => x.includes(`-${monthYear}-`));
        const lastWaspNum = waspNums[waspNums.length - 1];
        this.formEditor.patchValue({ waspNum: lastWaspNum });
      }
      else //hasSavedWaspNum is true
        this.formEditor.patchValue({ waspNum: this.savedDealDetail.waspNum });

      this.refreshWaspVolPrice$.next(null);
      this.waspNumsLoading$.next(false);
    }),
    shareReplay(1)
  )

  contactsResult$ = this.refreshContacts$.pipe(
    tap(() => this.editorLoading$.next(true)),
    delay(0),
    switchMap(() => {
      return this.dealService.getContacts();
    }),
    map(contacts => {
      this.localRequiredData.contacts = contacts;
      this.counterpartyChanged$.next(this.deal('counterpartyId'));
      return contacts;
    }),
    tap(() => this.editorLoading$.next(false)),
    shareReplay(1),
    catchError(err => {
      this.editorLoading$.next(false);
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  waspVolPriceResult$ = this.refreshWaspVolPrice$.pipe(
    delay(0), //delay for dropdown selection changes to finish
    filter(() => {
      const deal: models.DealDetail = this.formEditor.getRawValue();
      return deal.transactionTypeId === 3 && deal.dealPurposeId === 9;
    }),
    tap(() => {
      this.waspVolPriceLoading$.next(true)
    }),
    switchMap(() => {
      const deal: models.DealDetail = this.formEditor.getRawValue();
      return this.dealService.getWaspVolAndPrice(deal.dealNum, deal.waspNum, deal.productId, deal.numOfContracts, deal.fixedPrice, deal.brokerId, deal.isBuy, deal.accountingMonth, deal.startDate);
    }),
    tap((result) => {
      this.formEditor.patchValue({ hedgeFee: result.hedgeFee });
      this.waspVolPriceLoading$.next(false);
    }),
    shareReplay(1)
  )

  isConfirmButtonEnabled$ = this.refreshConfirmButton$.pipe(
    map(dealPurposeId => {
      if (dealPurposeId == null || dealPurposeId == undefined)
        return true;

      if (this.accountingDealPurposes.includes(dealPurposeId))
        return false;
      else
        return true;
    }),
    tap(() => {
      this.filterCounterparties$.next(null);
    }),
    shareReplay(1)
  )

  contactsForCounterparty$ = this.counterpartyChanged$.pipe(
    map(newCounterpartyId => {
      let contactsForCounterparty: models.ContactInfo[] = null;
      if (newCounterpartyId) {
        contactsForCounterparty = this.localRequiredData.contacts.filter(contactInfo => {
          const isMappedItem = contactInfo.counterpartyId === newCounterpartyId;
          return isMappedItem;
        });
      }

      //check if saved item exists and if not then add it
      const savedDeal = this.savedDealDetail;
      const isSavedCounterparty = newCounterpartyId === savedDeal.counterpartyId;
      if (isSavedCounterparty && contactsForCounterparty && contactsForCounterparty.findIndex(x => x.id === savedDeal.contactId) === -1) {
        const savedItem = this.localRequiredData.contacts.find(x => x.id === savedDeal.contactId);
        if (savedItem)
          contactsForCounterparty.push(savedItem);
      }

      return contactsForCounterparty;
    }),
    tap(contactsForCounterparty => {
      const contactId = this.deal('contactId');
      if (contactsForCounterparty && contactsForCounterparty.findIndex(contactInfo => contactInfo.id === contactId) === -1)
        this.formEditor.patchValue({ contactId: null });

      if (!this.localEditorLoading && contactsForCounterparty && contactsForCounterparty.length === 1)
        this.formEditor.patchValue({ contactId: contactsForCounterparty[0].id });
    }),
    shareReplay(1)
  )

  PointMatchesTransactionType(pointInfo: models.PointInfo) {
    const transactionTypeId = this.deal('transactionTypeId');
    // TODO: lets switch the numbers with the enumns from util
    if (transactionTypeId === util.TransactionType.PhysicalGas && pointInfo.isGasPoint)
      return true;
    else if (transactionTypeId === util.TransactionType.PhysicalNGL && pointInfo.isNglPoint)
      return true;
    else if (transactionTypeId === util.TransactionType.PhysicalCrudeOil && pointInfo.isCrudePoint)
      return true;
    else
      return false
  }

  pointsForPipeline$ = this.pipelineChanged$.pipe(
    map(newPipeId => {
      let pointsForPipeline: models.PointInfo[] = null;
      if (newPipeId) {
        pointsForPipeline = this.localRequiredData.points.filter(pointInfo => {
          const isActiveMappedItem = pointInfo.isActive && pointInfo.pipeId === newPipeId;
          const isTransactionTypeMatch = this.PointMatchesTransactionType(pointInfo);
          return isActiveMappedItem && isTransactionTypeMatch;
        });
      }

      return pointsForPipeline;
    }),
    tap(pointsForPipeline => {
      const pointId = this.deal('pointId');
      if (!pointsForPipeline || pointsForPipeline.findIndex(pointInfo => pointInfo.pointId === pointId) === -1)
        this.formEditor.patchValue({ pointId: null });

      if (!this.localEditorLoading && pointsForPipeline && pointsForPipeline.length === 1)
        this.formEditor.patchValue({ pointId: pointsForPipeline[0].pointId });
    }),
    shareReplay(1)
  )

  pointSourceDeliveriesForPipeline$ = this.pipelineSourceDeliveryChanged$.pipe(
    map(newPipeId => {
      const isValidTransactionType = this.deal('transactionTypeId') === util.TransactionType.PhysicalGas || this.deal('transactionTypeId') === util.TransactionType.PhysicalCrudeOil;
      if (!isValidTransactionType)
        return null;

      let pointsForPipeline: models.PointInfo[] = null;
      if (newPipeId) {
        pointsForPipeline = this.localRequiredData.pointSourceDeliveries.filter(pointInfo => {
          const isActiveMappedItem = pointInfo.isActive && pointInfo.pipeId === newPipeId;
          const isTransactionTypeMatch = this.PointMatchesTransactionType(pointInfo);
          return isActiveMappedItem && isTransactionTypeMatch;
        });
      }

      return pointsForPipeline;
    }),
    tap(pointsForPipeline => {
      const pointIds: number[] = this.deal('pointSourceDeliveryIds');
      if (!pointsForPipeline || !pointIds || !pointsForPipeline.some(pointInfo => pointIds.some(pointId => pointId === pointInfo.pointId)))
        this.formEditor.patchValue({ pointSourceDeliveryIds: null });

      if (!this.localEditorLoading && pointsForPipeline && pointsForPipeline.length === 1)
        this.formEditor.patchValue({ pointSourceDeliveryIds: [pointsForPipeline[0].pointId] });
    }),
    shareReplay(1)
  )

  accountsForBroker$ = this.brokerChanged$.pipe(
    map(newBrokerId => {
      let accountsForBroker: models.BrokerAccountInfo[] = null;
      if (newBrokerId) {
        accountsForBroker = this.localRequiredData.brokerAccounts.filter(accountInfo => {
          const isMappedItem = accountInfo.brokerId === newBrokerId;
          return isMappedItem;
        });
      }

      return accountsForBroker;
    }),
    tap(accountsForBroker => {
      const accountId = this.deal('brokerAccountId');
      if (!accountsForBroker || accountsForBroker.findIndex(accountInfo => accountInfo.brokerAccountId === accountId) === -1)
        this.formEditor.patchValue({ brokerAccountId: null });

      if (!this.localEditorLoading && accountsForBroker && accountsForBroker.length === 1)
        this.formEditor.patchValue({ brokerAccountId: accountsForBroker[0].brokerAccountId });
    }),
    shareReplay(1)
  )

  contractsForEntities$ = this.refreshContractsForEntities$.pipe(
    delay(0), //delay for dropdown selection changes to finish
    map(([newCounterpartyId, newInternalEntityId]) => {
      let contractsForEntities: models.ContractInfo[] = null;
      if (newCounterpartyId && newInternalEntityId) {
        contractsForEntities = this.localRequiredData.contracts.filter(contractInfo => {
          const isActiveMappedItem = contractInfo.isActive && contractInfo.counterpartyId === newCounterpartyId && contractInfo.internalEntityId === newInternalEntityId;
          return isActiveMappedItem;
        });
      }

      //check if saved item exists and if not then add it
      const savedDeal = this.savedDealDetail;;
      const isSavedEntities = newCounterpartyId === savedDeal.counterpartyId && newInternalEntityId === savedDeal.internalEntityId;
      if (isSavedEntities && contractsForEntities && contractsForEntities.findIndex(x => x.contractNumber === savedDeal.contractNumber) === -1) {
        const savedItem = this.localRequiredData.contracts.find(x => x.contractNumber === savedDeal.contractNumber);
        if (savedItem)
          contractsForEntities.push(savedItem);
      }

      return contractsForEntities;
    }),
    tap(contractsForEntities => {
      const contractNumber = this.deal('contractNumber');
      if (contractsForEntities && contractsForEntities.findIndex(contractInfo => contractInfo.contractNumber === contractNumber) === -1)
        this.formEditor.patchValue({ contractNumber: null });

      if (!this.localEditorLoading && contractsForEntities && contractsForEntities.length === 1)
        this.formEditor.patchValue({ contractNumber: contractsForEntities[0].contractNumber });
    }),
    shareReplay(1)
  )

  regions$ = this.filterRegions$.pipe(util.filterIdNames(this.editorLoading$, this.dealRequiredData$, 'regions'));
  strategies$ = this.filterStrategies$.pipe(util.filterIdNames(this.editorLoading$, this.dealRequiredData$, 'strategies'));
  portfolios$ = this.filterPortfolios$.pipe(util.filterIdNames(this.editorLoading$, this.dealRequiredData$, 'portfolios'));
  books$ = this.filterBooks$.pipe(util.filterSpecials(this.editorLoading$, this.booksForTrader$, null, 'bookName'));

  contacts$ = this.filterContacts$.pipe(util.filterSpecials(this.editorLoading$, this.contactsForCounterparty$, null, 'name'));
  traders$ = this.filterTraders$.pipe(util.filterSpecials(this.editorLoading$, this.dealRequiredData$, 'traders', 'traderName'));
  dealStatuses$ = this.filterDealStatuses$.pipe(util.filterIdNames(this.editorLoading$, this.dealRequiredData$, 'dealStatuses'));
  dealPurposes$ = this.filterDealPurposes$.pipe(util.filterIdNames(this.editorLoading$, this.dealRequiredData$, 'dealPurposes'));
  dealTypes$ = this.filterDealTypes$.pipe(util.filterIdNames(this.editorLoading$, this.dealRequiredData$, 'dealTypes'));

  counterparties$ = this.filterCounterparties$.pipe(util.filterSpecials(this.editorLoading$, this.dealRequiredData$.pipe(
    map((data) => {
      const currentTransactionTypeId = this.deal('transactionTypeId');
      const currentDealPurposeId = this.deal('dealPurposeId');
      const currentCounterpartyId = this.deal('counterpartyId');
      const isAccountingDeal = this.accountingDealPurposes.includes(currentDealPurposeId);
      const counterparties = data.counterparties.filter((counterparty) => {
        const isGasTransactionType = currentTransactionTypeId === util.TransactionType.PhysicalGas && counterparty.isGasEntity;
        const isCrudeTransactionType = currentTransactionTypeId === util.TransactionType.PhysicalCrudeOil && counterparty.isCrudeEntity;
        const isFinancialTransactionType = currentTransactionTypeId === util.TransactionType.Futures && counterparty.isFinancialEntity;
        const noProductCriteria = ![1, 3, 11].includes(currentTransactionTypeId);
        const matchesProductCriteria = isGasTransactionType || isCrudeTransactionType || isFinancialTransactionType || noProductCriteria;

        let isRelationshipValid = null;
        if (isAccountingDeal)
          isRelationshipValid = counterparty.isTradingEntity || counterparty.isTransportEntity;
        else
          isRelationshipValid = counterparty.isTradingEntity;


        const isCurrentSelectedCounterparty = currentCounterpartyId === counterparty.counterpartyId;
        return ((matchesProductCriteria && isRelationshipValid) || isCurrentSelectedCounterparty);
      });

      return counterparties;
    })
  ), null, 'counterpartyName'));

  internalEntities$ = this.filterInternalEntities$.pipe(util.filterIdNames(this.editorLoading$, this.dealRequiredData$, 'internalEntities'));

  pipelines$ = this.filterPipelines$.pipe(util.filterSpecials(this.editorLoading$, this.dealRequiredData$.pipe(
    map(data => data.pipelines.filter(pipe => (this.deal('transactionTypeId') === util.TransactionType.PhysicalGas && pipe.isGasPipe) || (this.deal('transactionTypeId') === util.TransactionType.PhysicalCrudeOil && pipe.isCrudePipe))),
  ), null, 'pipeName'));

  points$ = this.filterPoints$.pipe(util.filterSpecials(this.editorLoading$, this.pointsForPipeline$, null, 'pointName'));

  pipelineSourceDeliveries$ = this.filterPipelineSourceDeliveries$.pipe(util.filterSpecials(this.editorLoading$, this.dealRequiredData$.pipe(
    map(data => data.pipelineSourceDeliveries.filter(pipe => (this.deal('transactionTypeId') === 1 && pipe.isGasPipe) || (this.deal('transactionTypeId') === util.TransactionType.PhysicalCrudeOil && pipe.isCrudePipe))),
  ), null, 'pipeName'));

  pointSourceDeliveries$ = this.filterPointSourceDeliveries$.pipe(util.filterSpecials(this.editorLoading$, this.pointSourceDeliveriesForPipeline$, null, 'pointName'));
  contracts$ = this.filterContracts$.pipe(util.filterSpecials(this.editorLoading$, this.contractsForEntities$, null, 'contractNumber'));
  volumeTypes$ = this.filterVolumeTypes$.pipe(util.filterIdNames(this.editorLoading$, this.dealRequiredData$, 'volumeTypes'));
  fuelCalculationTypes$ = this.filterFuelCalculationTypes$.pipe(util.filterIdNames(this.editorLoading$, this.dealRequiredData$, 'fuelCalculationTypes'));
  forceMajeures$ = this.filterForceMajeures$.pipe(util.filterIdNames(this.editorLoading$, this.dealRequiredData$, 'forceMajeures'));

  products$ = this.filterProducts$.pipe(util.filterIdNames(this.editorLoading$, this.dealRequiredData$.pipe(
    map(data => {
      const deal: models.DealDetail = this.formEditor.getRawValue();
      const products = data.products.filter(product => {
        const isNglProduct = [Product.Ethane, Product.Propane, Product.IsoButane, Product.NormalButane, Product.NaturalGasoline, Product.YGrade].includes(product.id);
        const isCrudeProduct = [Product.CrudeOil, Product.Condensate, Product.Retrograde].includes(product.id);
        const isNatGasProduct = [Product.NaturalGas, Product.LiquifiedNaturalGas].includes(product.id);

        switch (deal.transactionTypeId) {
          case util.TransactionType.PhysicalGas: //Physical Natural Gas
            return isNatGasProduct;
          case util.TransactionType.PhysicalCrudeOil: //Physical Crude
            return isCrudeProduct;
          case util.TransactionType.PhysicalNGL:  //Physical NGLs
            return isNglProduct;
          case util.TransactionType.Futures: //Futures
            return true;
          case util.TransactionType.SwingSwaps: //Swing Swaps
            return true;
        }
      });
      return products;
    }),
  ), null));

  brokers$ = this.filterBrokers$.pipe(util.filterIdNames(this.editorLoading$, this.dealRequiredData$, 'brokers'));
  brokerAccounts$ = this.filterBrokerAccounts$.pipe(util.filterSpecials(this.editorLoading$, this.accountsForBroker$, null, 'brokerAccountName'));
  waspNums$ = this.filterWaspNums$.pipe(util.filterNames(this.editorLoading$, this.newWaspNums$, null));
  deliveryMethods$ = this.filterDeliveryMethods$.pipe(util.filterIdNames(this.editorLoading$, this.dealRequiredData$, 'deliveryMethods'));
  paidOnTypes$ = this.filterPaidOnTypes$.pipe(util.filterIdNames(this.editorLoading$, this.dealRequiredData$, 'paidOnTypes'));
  priceIndexes$ = this.filterPriceIndexes$.pipe(util.filterSpecials(this.editorLoading$, this.dealRequiredData$.pipe(
    map(data => data.priceIndexes.filter(index => index.productId === Product.NaturalGas || Product.LiquifiedNaturalGas || index.productId === null)),
  ), null, 'name'));
  priceIndexesC2$ = this.filterPriceIndexesC2$.pipe(util.filterSpecials(this.editorLoading$, this.dealRequiredData$.pipe(
    map(data => data.priceIndexes.filter(index => index.productId === Product.Ethane || index.productId === null)),
  ), null, 'name'));
  priceIndexesC3$ = this.filterPriceIndexesC3$.pipe(util.filterSpecials(this.editorLoading$, this.dealRequiredData$.pipe(
    map(data => data.priceIndexes.filter(index => index.productId === Product.Propane || index.productId === null)),
  ), null, 'name'));
  priceIndexesIc4$ = this.filterPriceIndexesIc4$.pipe(util.filterSpecials(this.editorLoading$, this.dealRequiredData$.pipe(
    map(data => data.priceIndexes.filter(index => index.productId === Product.IsoButane || index.productId === null)),
  ), null, 'name'));
  priceIndexesNc4$ = this.filterPriceIndexesNc4$.pipe(util.filterSpecials(this.editorLoading$, this.dealRequiredData$.pipe(
    map(data => data.priceIndexes.filter(index => index.productId === Product.NormalButane || index.productId === null)),
  ), null, 'name'));
  priceIndexesC5p$ = this.filterPriceIndexesC5p$.pipe(util.filterSpecials(this.editorLoading$, this.dealRequiredData$.pipe(
    map(data => data.priceIndexes.filter(index => index.productId === Product.NaturalGasoline || index.productId === null)),
  ), null, 'name'));
  priceIndexesCrude$ = this.filterPriceIndexes$.pipe(util.filterSpecials(this.editorLoading$, this.dealRequiredData$.pipe(
    map(data => data.priceIndexes.filter(index => index.productId === Product.CrudeOil || index.productId === Product.Condensate || index.productId === Product.Retrograde || index.productId === null)),
  ), null, 'name'));

  deals$ = this.refreshDeals$.pipe(
    filter(() => this.localSelectedFilterId ? true : false),
    debounceTime(100),
    tap(() => {
      this.gridLoading$.next(true);
    }),
    switchMap(() => {
      return this.dealService.getDeals(this.state, this.localSelectedFilterId);
    }),
    tap((x) => {
      this.gridLoading$.next(false);
      this.goToSavedGridScrollPos();
      if (this.hasDealNumParam && x.data.length > 0) {
        this.openEditor(x.data[0].Id);
        this.hasDealNumParam = false;
      }
    }),
    shareReplay(1),
    catchError(err => {
      this.gridLoading$.next(false);
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  exportAction$ = this.exportClicked$.pipe(
    switchMap(() => {
      this.exporting = true
      return this.dealService.exportDeals(this.state, this.localSelectedFilterId, 'Deals.xlsx');
    }),
    tap(res => {
      util.openOrSaveFile(res.fileBlob, res.fileName);
      this.exporting = false;
    }),
    shareReplay(1),
    catchError(err => {
      this.exporting = false;
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  saveGridSettingsAction$ = this.saveGridSettings$.pipe(
    filter(() => this.localSelectedFilterId !== null),
    switchMap(() => {
      return this.dealService.saveGridSettings(this.state, this.localSelectedFilterId);
    }),
    tap((savedDealFilter) => {
      this.dealFilters.find(x => x.id === savedDealFilter.id).state = savedDealFilter.state;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    })
  )

  saveDealResult$ = this.saveDeal$.pipe(
    switchMap(([itemToSave, saveType]) => {
      this.isVariableVolumeEditorOpened = false;
      this.formEditor.disable();
      this.formEditorVariableVolumes.disable();
      return this.dealService.saveDealDetail(itemToSave, saveType);
    }),
    tap(saveResult => {
      this.notify.success('save successful');
      this.closeEditor(false);
      this.refreshDeals$.next(null);
      this.mySelection = [saveResult.dealNum];
    }),
    shareReplay(1),
    catchError(err => {
      this.editorFinishedLoading();
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  saveSourceDeliveryResult$ = this.saveSourceDelivery$.pipe(
    switchMap(itemToSave => {
      this.editorLoading$.next(true);
      this.isVariableVolumeEditorOpened = false;
      this.formEditor.disable();
      this.formEditorVariableVolumes.disable();
      return this.dealService.saveSourceDelivery(itemToSave);
    }),
    tap(() => {
      this.formEditor.get('pipelineSourceDeliveryId').markAsPristine();
      this.formEditor.get('pointSourceDeliveryIds').markAsPristine();
      this.formEditor.get('sourceDeliveryPointItems').markAsPristine();
      this.notify.success('save successful');
      this.editorFinishedLoading();
      this.refreshDeals$.next(null);
    }),
    shareReplay(1),
    catchError(err => {
      this.editorFinishedLoading();
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  deleteDealResult$ = this.deleteDeal$.pipe(
    switchMap(() => {
      this.isVariableVolumeEditorOpened = false;
      const itemToDelete: models.DealDetail = this.formEditor.getRawValue();
      return this.dealService.deleteDeal(itemToDelete.id);
    }),
    tap(() => {
      this.notify.success('delete successful');
      this.editorFinishedLoading();
      this.closeEditor(false);
      this.refreshDeals$.next(null);
    }),
    shareReplay(1),
    catchError(err => {
      this.editorFinishedLoading();
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  manualTriggersCompleting$ = zip(
    this.refreshConfirmButton$,
    this.counterpartyChanged$,
    this.traderChanged$,
    this.pipelineChanged$,
    this.pipelineSourceDeliveryChanged$,
    this.productChanged$,
    this.brokerChanged$,
    this.startDateChanged$,
    this.contractsForEntities$,
    this.pointSourceDeliveriesForPipeline$,
    this.accountsForBroker$,
    this.booksForTrader$,
    this.contactsForCounterparty$,
    this.pointsForPipeline$
  ).pipe(
    tap(() => {
      this.editorFinishedLoading();
    })
  )

  fillColumnPropertyNames(filterId: number): void {
    if (filterId) {
      const filter = this.dealFilters.find(x => x.id === filterId);

      this.visibleColumnInfos = [];
      if (filter.columns && filter.columns.length > 0) {
        const columns = filter.columns.split(',');
        columns.forEach(col => {
          const colSplit = col.split(':');
          const colInfo = this.localAllColumnDisplayInfos[colSplit[0]];
          if (colSplit.length === 2) //if has width
            colInfo.width = parseInt(colSplit[1]);
          this.visibleColumnInfos.push(colInfo);
        });
      } else {
        const colInfo = this.localAllColumnDisplayInfos['DealNum'];
        this.visibleColumnInfos.push(colInfo);
      }
    }
  }

  uniqueDealResult$ = this.uniqueDealCheck$.pipe(
    switchMap(([itemToSave, saveType]) => {
      return combineLatest([this.dealService.uniqueDealCheck(itemToSave, saveType), of(itemToSave), of(saveType)]);
    }),
    tap(([similarTicketNums, itemToSave, saveType]) => {
      if (similarTicketNums && similarTicketNums.length > 0) {
        // const uniqueMessageSettings: DialogSettings = {
        //   title: "Similar deal approval",
        //   content: `A deal with similar properties already exists: ${similarTicketNums}.  Are you sure you want to save?`,
        //   actions: [{ text: 'No' }, { text: 'Yes', primary: true }],
        //   width: 400,
        //   cssClass: 'utilPrompt'
        // }

        const ps: PromptSettings = {
          title: "Similar deal approval",
          content: `A deal with similar properties already exists: ${similarTicketNums}.  Are you sure you want to save?`,
          type: "Yes-No"
        }

        this.messageService.prompt(ps).then(result => {
          if (result === promptAction.Yes)
            this.saveDeal$.next([itemToSave, saveType]);
          else
            this.editorLoading$.next(false);
        });

        //   this.dialogService.open(uniqueMessageSettings).result.pipe(take(1)).subscribe(result => {
        //     if (util.getDialogAction(result) === util.dialogAction.Yes)
        //       this.saveDeal$.next([itemToSave, saveType]);
        //     else
        //       this.editorLoading$.next(false);
        //   });
      }
      else
        this.saveDeal$.next([itemToSave, saveType]);
    }),
    shareReplay(1),
    catchError(err => {
      this.editorFinishedLoading();
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  setFilterSelectedResult$ = this.filterSelected$.pipe(
    filter(filterId => filterId ? true : false),
    switchMap(filterId => {
      this.localSelectedFilterId = filterId;
      const selectedFilter = this.localRequiredData.filters.find(x => x.id === filterId);
      const parsedState: State = util.parseWithDate(selectedFilter.state)

      if (!this.hasDealNumParam)
        this.state.filter = parsedState.filter;

      this.state.sort = parsedState.sort;
      this.state.take = 60;
      this.state.skip = null;
      const distributionViewId = this.dealFilters.find(x => x.name === this.distributionViewName)?.id;
      const isDistributionView = filterId === distributionViewId;
      if (!isDistributionView) {
        this.dealFilters.forEach(filter => {
          if (filter.id === filterId)
            filter.isSelected = true;
          else
            filter.isSelected = false;
        });
      }

      this.fillColumnPropertyNames(filterId);
      this.refreshDeals$.next(null);
      return this.dealFilterService.setFilterSelected(filterId)
    }),
    catchError(err => {
      return util.handleError(err, this.messageService)
    })
  )

  downloadConfirmResult$ = this.downloadConfirm$.pipe(
    switchMap(ticketNum => {
      return this.dealService.downloadConfirm(ticketNum);
    }),
    tap(res => {
      util.openOrSaveFile(res.fileBlob, res.fileName);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  downloadTicketResult$ = this.downloadTicket$.pipe(
    switchMap(ticketNum => {
      return this.dealService.downloadTicket(ticketNum);
    }),
    tap(res => {
      util.openOrSaveFile(res.fileBlob, res.fileName);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  distributeResult$ = this.distribute$.pipe(
    switchMap(() => {
      return this.dealService.distribute(this.mySelection);
    }),
    tap(distributeResponse => {
      this.isDistributing$.next(false);
      if (distributeResponse.hasErrors)
        this.messageService.throw(distributeResponse.message);
      else
        this.messageService.info(distributeResponse.message);
      this.mySelection = [];
      this.util.hideTooltips(this.allTips);
      this.refreshDeals$.next(null);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  mergeFilesResult$ = this.mergeFiles$.pipe(
    switchMap(mergeType => {
      return this.dealService.mergeFiles(this.mySelection, mergeType);
    }),
    tap(res => {
      util.openOrSaveFile(res.fileBlob, res.fileName);
      this.isMergingFiles$.next(false);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  setEditorItem(editorItem: models.DealDetail): void {
    this.configureFormVariableVolumes(
      this.deal('isVariableVolume'),
      this.deal('startDate'),
      this.deal('endDate'),
      this.deal('volumeTypeId'),
      this.deal('volume')
    );
    this.updateItemDealVolumes(editorItem);
    //use setTimeout since the form controls may not be registered yet
    setTimeout(() => {
      this.formEditor.setValue(editorItem);
      this.setupTransactionTypeChange();
      this.setupVariableVolumeChange();
      this.setupIsFixedPriceChange();
      this.setupDealPurposeChange();
    });
  }

  configureFormVariableVolumes(isVariable: boolean, startDate: Date, endDate: Date, volumeTypeId: number, volume: number) {
    if (isVariable && startDate && endDate && (volumeTypeId === 0 || volumeTypeId === 1)) {
      const loopByDay = volumeTypeId === 0;
      let loopDate = loopByDay ? new Date(startDate) : dayjs(startDate).startOf('month').toDate();
      this.formEditorVariableVolumes.clear();
      while (loopDate <= endDate) {
        const dealVol: models.DealDetailVolume = { startDate: loopDate, physicalVolume: volume };
        this.formEditorVariableVolumes.push(this.getNewVariableVolumeEditorItem(dealVol));
        loopDate = loopByDay ? dayjs(loopDate).add(1, 'day').toDate() : dayjs(loopDate).add(1, 'month').toDate();
      }

      if (this.savedDealVolumes) {
        this.savedDealVolumes.forEach(dealVol => {
          const foundControl = this.formEditorVariableVolumes.controls.find(ctrl => dayjs(ctrl.value.startDate).isSame(dealVol.startDate));
          if (foundControl)
            foundControl.patchValue({ physicalVolume: dealVol.physicalVolume });
        });
      }
    }

    if (!(volumeTypeId === 0 || volumeTypeId === 1) && volumeTypeId !== null) {
      this.formEditor.patchValue({ isVariableVolume: false });
      setTimeout(() => this.formEditor.get('isVariableVolume').disable(), 100);
    } else {
      setTimeout(() => this.formEditor.get('isVariableVolume').enable(), 100);
    }
  }

  updateItemDealVolumes(dealItem: models.DealDetail) {
    //this ensures that the we have a value for every variable volume form item
    //formEditor.setValue fails with "must supply a value" error if we have a form item without a corresponding volume array item from the database
    this.formEditorVariableVolumes.controls.forEach(ctrl => {
      const ctrlValue: models.DealDetailVolume = ctrl.value;
      const foundItem = dealItem.dealVolumes.find(item => dayjs(ctrlValue.startDate).isSame(item.startDate));
      if (!foundItem) {
        const emptyDetailVolume: models.DealDetailVolume = { startDate: ctrlValue.startDate, physicalVolume: null };
        dealItem.dealVolumes.push(emptyDetailVolume);
      }
    });
    if (dealItem.dealVolumes && dealItem.dealVolumes.length > 1)
      dealItem.dealVolumes = dealItem.dealVolumes.sort((item1, item2) => item1.startDate.getTime() - item2.startDate.getTime());
  }

  openPointsDetail(): void {
    const dealDetail: models.DealDetail = this.formEditor.getRawValue();
    const pointVolume = dealDetail.isVariableVolume ? null : dealDetail.volume;
    this.detailPointsForm.reset();
    this.detailPointsForm.setValue({ pointId: null, pointVolume: pointVolume, pointName: null });
    this.detailPointsOpened$.next(true);
  }

  openEditor(dealId: number): void {
    this.formEditor.disable();
    this.formEditorVariableVolumes.disable();
    this.editorLoading$.next(true);
    this.saveGridScrollPos();
    this.formEditor.reset();
    //set new values for an 'add' action
    this.formEditor.patchValue({ id: dealId, dealNum: null });
    this.editorOpened$.next(true);
    this.refreshDetail$.next(dealId);
  }

  addNew(): void {
    if (this.hasDealModifyPermission || this.hasDealForAccountingPermission) {
      this.openEditor(0);
      this.editorFinishedLoading();
    }
  }

  addPoints(): void {
    this.openPointsDetail();
  }

  export(): void {
    this.exportClicked$.next(null);
  }

  distribute(template: TemplateRef<unknown>): void {
    // const distributeConfirmSettings: DialogSettings = {
    //   title: "Please confirm",
    //   content: template,
    //   actions: [{ text: 'No' }, { text: 'Yes', primary: true }],
    //   cssClass: 'utilPrompt'
    // }
    const ps: PromptSettings = {
      title: "Please confirm",
      content: template,
      type: "Yes-No"
    }

    this.messageService.prompt(ps).then((result) => {
      if (result === promptAction.Yes) {
        this.isDistributing$.next(true);
        this.distribute$.next(null);
      }
    });

    // this.dialogService.open(distributeConfirmSettings).result.pipe(take(1)).subscribe(result => {
    //   if (util.getDialogAction(result) === util.dialogAction.Yes) {
    //     this.isDistributing$.next(true);
    //     this.distribute$.next(null);
    //   }
    // });
  }

  updatePhysical(template: TemplateRef<unknown>): void {
    // const updatePhysicalSettings: DialogSettings = {
    //   title: "Please confirm",
    //   content: template,
    //   actions: [{ text: 'No' }, { text: 'Yes', primary: true }],
    //   cssClass: 'utilPrompt'
    // }
    const ps: PromptSettings = {
      title: "Please confirm",
      content: template,
      type: "Yes-No"
    }

    this.messageService.prompt(ps).then((result) => {
      if (result === promptAction.Yes) {
        //this.isDistributing$.next(true);
        //this.distribute$.next(null);
      }
    })
    // this.dialogService.open(updatePhysicalSettings).result.pipe(take(1)).subscribe(result => {
    //   if (util.getDialogAction(result) === util.dialogAction.Yes) {
    //     //this.isDistributing$.next(true);
    //     //this.distribute$.next(null);
    //   }
    // });
  }

  mergeFiles(): void {
    const mergeChoiceSettings: DialogSettings = {
      title: "Please choose",
      content: "Please choose the type of files to merge.",
      actions: [{ text: 'Confirmations' }, { text: 'Tickets' }],
      cssClass: 'utilPrompt'
    }
    this.dialogService.open(mergeChoiceSettings).result.pipe(take(1)).subscribe(result => {
      if (result instanceof DialogCloseResult)
        return null;
      else {
        this.formEditor.disable();
        this.formEditorVariableVolumes.disable();

        this.isMergingFiles$.next(true);
        if (result.text === "Confirmations")
          this.mergeFiles$.next(MergeType.Confirms);
        else if (result.text === "Tickets")
          this.mergeFiles$.next(MergeType.Tickets);
      }
    });
  }

  dataStateChange(state: State): void {
    this.gridScrollTopPos = 0;
    this.gridScrollLeftPos = 0;
    util.fixKendoQueryFilter(state.filter);
    this.state = state;
    this.saveGridSettings$.next(null);
    this.refreshDeals$.next(null);
  }

  edit(dataItem: models.DealListItem, isButtonClick: boolean): void {
    if (this.mode === Mode.Deal || isButtonClick) {
      this.mySelection = [dataItem.DealNum];
      this.openEditor(dataItem.Id);
    }
  }

  openFilterEditor(): void {
    this.loading$.next(true);
    this.saveGridScrollPos();
    this.filterEditorOpened = true;
    this.loading$.next(false);
  }

  saveDeal(isSaveNew: boolean): void {
    this.formEditor.markAllAsTouched();
    if (this.formEditor.valid) {
      if (!isSaveNew && this.localDealDetail.created && this.localRequiredData.dealModifyTimeLag > 0) {
        const secondsPassed = dayjs().diff(this.localDealDetail.created, 'second');
        if (secondsPassed >= this.localRequiredData.dealModifyTimeLag) {
          // const dealModifyMessageSettings: DialogSettings = {
          //   title: "Modification approval",
          //   content: `More than ${this.localRequiredData.dealModifyTimeLag} seconds have passed since this deal was created.  Are you sure you want to modify it?`,
          //   actions: [{ text: 'No' }, { text: 'Yes', primary: true }],
          //   width: 400,
          //   cssClass: 'utilPrompt'
          // }
          const ps: PromptSettings = {
            title: "Modification approval",
            content: `More than ${this.localRequiredData.dealModifyTimeLag} seconds have passed since this deal was created.  Are you sure you want to modify it?`,
            type: "Yes-No"
          }
          this.messageService.prompt(ps).then((result) => {
            if (result === promptAction.Yes)
              this.saveValidDeal(isSaveNew);
          });
        }
        else
          this.saveValidDeal(isSaveNew);
      }
      else
        this.saveValidDeal(isSaveNew);
    } else {
      this.notify.error("validation failed");
      console.log(this.findInvalidControls());
    }
  }

  saveSourceDelivery(): void {
    this.formEditor.markAllAsTouched();
    if (this.formEditor.valid) {
      const itemToSave: models.DealDetail = this.formEditor.getRawValue();
      this.saveSourceDelivery$.next(itemToSave);
    }
    else {
      this.notify.error("validation failed");
      console.log(this.findInvalidControls());
    }
  }

  private findInvalidControls() {
    const invalid: string[] = [];
    const controls = this.formEditor.controls;
    Object.keys(controls).forEach(name => {
      if (controls[name as keyof typeof controls].invalid) {
        invalid.push(name);
      }
    });
    return invalid;
  }

  private saveValidDeal(isSaveNew: boolean): void {
    this.editorLoading$.next(true);
    const saveType = isSaveNew ? SaveType.New : SaveType.Normal;
    const itemToSave: models.DealDetail = this.formEditor.getRawValue();
    this.uniqueDealCheck$.next([itemToSave, saveType]);
  }

  deleteDeal(): void {
    // const deleteConfirmSettings: DialogSettings = {
    //   title: "Please confirm",
    //   content: "Are you sure you want to delete this deal?",
    //   actions: [{ text: 'No' }, { text: 'Yes', primary: true }],
    //   cssClass: 'utilPrompt'
    // }

    const ps: PromptSettings = {
      title: "Please confirm",
      content: "Are you sure you want to delete this deal?",
      type: "Yes-No"
    }

    this.messageService.prompt(ps).then((result) => {
      if (result === promptAction.Yes) {
        this.formEditor.disable();
        this.formEditorVariableVolumes.disable();
        this.editorLoading$.next(true);
        this.deleteDeal$.next(null);
      }
    })

    // this.dialogService.open(deleteConfirmSettings).result.pipe(take(1)).subscribe(result => {
    //   if (util.getDialogAction(result) === util.dialogAction.Yes) {
    //     this.formEditor.disable();
    //     this.formEditorVariableVolumes.disable();
    //     this.editorLoading$.next(true);
    //     this.deleteDeal$.next(null);
    //   }
    // });
  }

  toggleMode(newFilterId: number): void {
    this.loading$.next(true);
    this.util.hideTooltips(this.allTips);
    setTimeout(() => {
      if (this.mode === Mode.Deal) {
        this.mode = Mode.Distribute
        this.mySelection = [];
        this.selectableSettings.mode = 'multiple';
        const distributionView = this.dealFilters.find(x => x.name === this.distributionViewName);
        if (distributionView)
          this.filterSelected$.next(distributionView.id);
      }
      else if (this.mode === Mode.Distribute) {
        this.mode = Mode.Deal
        this.mySelection = [];
        this.selectableSettings.mode = 'single';
        let selectedView = this.dealFilters.find(x => newFilterId && x.id === newFilterId);
        if (!selectedView)
          selectedView = this.dealFilters.find(x => x.isSelected);
        if (selectedView)
          this.filterSelected$.next(selectedView.id);
      }
      this.loading$.next(false);
    });
  }

  onFilterEditorClosed(closedFilterId: number) {
    this.loading$.next(true);
    this.filterEditorOpened = false;
    this.gridScrollTopPos = 0;
    this.gridScrollLeftPos = 0;
    setTimeout(() => {
      this.refreshRequiredData$.next(closedFilterId);
    }, 100);
  }

  editorFinishedLoading(): void {
    this.editorLoading$.next(false);
    this.formEditor.enable();
    this.formEditorVariableVolumes.enable();
  }

  onEditorClosed() {
    if (this.formEditor.dirty) {
      // const unsavedChangesSettings: DialogSettings = {
      //   title: "Please confirm",
      //   content: "You have unsaved changes. What would you like to do?",
      //   actions: [{ text: 'Save', primary: true }, { text: 'Don\'t Save' }, { text: 'Cancel' }],
      //   cssClass: 'utilPrompt'
      // }
      const ps: PromptSettings = {
        title: "Please confirm",
        content: "You have unsaved changes. What would you like to do?",
        type: "Save-DontSave-Cancel"
      }

      this.messageService.prompt(ps).then((result) => {
        if (result === promptAction.Save) {
          this.saveDeal(false);
        } else if (result === promptAction.DontSave) {
          this.closeEditor(true);
        }
      })
      // this.dialogService.open(unsavedChangesSettings).result.pipe(take(1)).subscribe(result => {
      //   if (util.getDialogAction(result) === util.dialogAction.Save)
      //     this.saveDeal(false);
      //   else if (util.getDialogAction(result) === util.dialogAction.DontSave)
      //     this.closeEditor(true);
      // });
    } else
      this.closeEditor(true);
  }

  onDetailPointsClosing() {
    // util.onDetailChanging(this.detailPointsForm, this.dialogService, this.closePointEditor, this.setPoints, { confirmType: 'set' });
    util.onDetailChanging(this.detailPointsForm, this.messageService, this.closePointEditor, this.setPoints, { confirmType: 'set' });
  }

  isNewDeal(): boolean {
    const detailValue: models.DealDetail = this.formEditor.value as models.DealDetail;
    const dealId = detailValue?.id ?? 0;
    const isNewDeal = dealId === 0;
    return isNewDeal;
  }

  setPoints = () => {
    const deal = this.formEditor.value as models.DealDetail;
    const point = this.detailPointsForm.value as models.SourceDeliveryPointItem;
    const pointId = point.pointId;
    const pointVolume = deal.isVariableVolume ? 0 : point.pointVolume;

    if (!pointId) {
      this.notify.error("Point is required");
      return;
    }

    if (pointVolume == null) {
      this.notify.error("Volume is required");
      return;
    }

    //valid point and volume
    const existingPoint = deal.sourceDeliveryPointItems.find(x => x.pointId === pointId);
    if (existingPoint) //the point exists, update the volume
      existingPoint.pointVolume = pointVolume;
    else { //add the point
      const pointName = this.localRequiredData.points.find(x => x.pointId === pointId).pointName;
      const pointItem: models.SourceDeliveryPointItem = { pointId: pointId, pointName: pointName, pointVolume: pointVolume };
      deal.sourceDeliveryPointItems.push(pointItem);
    }
    this.formEditor.patchValue({ sourceDeliveryPointItems: deal.sourceDeliveryPointItems });
    this.closePointEditor();
  }

  closeEditor(isFromInterface: boolean) {
    if (this.transactionTypeSubscription)
      this.transactionTypeSubscription.unsubscribe();

    if (this.fixedPriceSubscription)
      this.fixedPriceSubscription.unsubscribe();

    if (this.variableVolumeSubscription)
      this.variableVolumeSubscription.unsubscribe();

    if (this.dealPurposeSubscription)
      this.dealPurposeSubscription.unsubscribe();

    if (!isFromInterface) {
      this.editorFinishedLoading();
      this.editorOpened$.next(false);
      this.isVariableVolumeEditorOpened = false;
    }
    else {
      this.editorFinishedLoading();
      this.loading$.next(true);
      this.editorOpened$.next(false);
      this.isVariableVolumeEditorOpened = false;
      this.loading$.next(false);
      setTimeout(() => {
        this.goToSavedGridScrollPos();
      }, 100);
    }
  }

  onFilterSelected(event: DealFilter) {
    this.localSelectedFilterId = event?.id;

    if (event?.name === this.distributionViewName && this.mode === Mode.Deal)
      this.toggleMode(null);
    else
      this.filterSelected$.next(event?.id);
  }

  pointClicked(pointItem: models.SourceDeliveryPointItem) {
    const deal: models.DealDetail = this.formEditor.getRawValue();
    if (deal.isVariableVolume)
      this.notify.error("Variable points cannot be edited");
    else {
      this.detailPointsForm.reset();
      this.detailPointsForm.setValue({ pointId: pointItem.pointId, pointVolume: pointItem.pointVolume, pointName: pointItem.pointName });
      this.isUpdatePointsOpened$.next(true);
    }
  }

  onUpdatePointsClosing() {
    // util.onDetailChanging(this.detailPointsForm, this.dialogService, this.closePointEditor, this.setPoints, { confirmType: 'set' });
    util.onDetailChanging(this.detailPointsForm, this.messageService, this.closePointEditor, this.setPoints, { confirmType: 'set' });
  }

  closePointEditor = () => {
    this.detailPointsOpened$.next(false);
    this.isUpdatePointsOpened$.next(false);
  }

  saveGridSettings() {
    this.saveGridSettings$.next(null);
  }

  downloadConfirm(ticketNum: string) {
    this.downloadConfirm$.next(ticketNum);
  }

  downloadTicket(ticketNum: string) {
    this.downloadTicket$.next(ticketNum);
  }

  showInputTooltip(e: MouseEvent): void {
    const element = e.target as HTMLElement;

    if (element.nodeName === 'INPUT' && (element as HTMLInputElement).type !== "checkbox") {
      if (util.isClipped(element) || this.isContactElem(element))
        this.tooltipInputItems.toggle(element);
      else
        this.util.hideTooltips(this.allTips);
    } else {
      this.util.hideTooltips(this.allTips);
    }
  }

  isContactElem(elem: HTMLElement): boolean {
    return elem?.parentElement?.parentElement?.parentElement?.className?.includes('contact');
  }

  getInputTooltipValues(anchor: ElementRef<HTMLElement>): string[] {
    if (this.isContactElem(anchor.nativeElement)) {
      const tipValues: string[] = [];
      const contactId = this.deal('contactId');
      if (contactId) {
        const contactInfo = this.localRequiredData.contacts.find(x => x.id === contactId);
        tipValues.push(contactInfo.name);
        tipValues.push(...contactInfo.info.split('•'));
      }
      return tipValues;
    }
    else {
      const inputElem = anchor.nativeElement as HTMLInputElement;
      return [inputElem.value];
    }
  }

  copyDealNum() {
    // const dealNum = (this.copyDealNumElem.nativeElement as HTMLInputElement).value;
    const dealNum = this.formEditor.get('dealNum').value;
    navigator.clipboard.writeText(dealNum).then(() => {
      // this.notify.show('copied', 'success', false, 'top');
      this.notify.success('copied');
    });
  }

  filterRegions(value: string) {
    this.filterRegions$.next(value);
  }

  filterStrategies(value: string) {
    this.filterStrategies$.next(value);
  }

  filterPortfolios(value: string) {
    this.filterPortfolios$.next(value);
  }

  filterBooks(value: string) {
    this.filterBooks$.next(value);
  }

  filterCounterparties(value: string) {
    this.filterCounterparties$.next(value);
  }

  filterInternalEntities(value: string) {
    this.filterInternalEntities$.next(value);
  }

  filterContacts(value: string) {
    this.filterContacts$.next(value);
  }

  filterTraders(value: string) {
    this.filterTraders$.next(value);
  }

  filterDealStatuses(value: string) {
    this.filterDealStatuses$.next(value);
  }

  filterDealPurposes(value: string) {
    this.filterDealPurposes$.next(value);
  }

  filterDealTypes(value: string) {
    this.filterDealTypes$.next(value);
  }

  filterPaidOnTypes(value: string) {
    this.filterPaidOnTypes$.next(value);
  }

  filterPriceIndexes(value: string) {
    this.filterPriceIndexes$.next(value);
  }

  filterPriceIndexesC2(value: string) {
    this.filterPriceIndexesC2$.next(value);
  }

  filterPriceIndexesC3(value: string) {
    this.filterPriceIndexesC3$.next(value);
  }

  filterPriceIndexesIc4(value: string) {
    this.filterPriceIndexesIc4$.next(value);
  }

  filterPriceIndexesNc4(value: string) {
    this.filterPriceIndexesNc4$.next(value);
  }

  filterPriceIndexesC5p(value: string) {
    this.filterPriceIndexesC5p$.next(value);
  }

  filterPipelines(value: string) {
    this.filterPipelines$.next(value);
  }

  filterPoints(value: string) {
    this.filterPoints$.next(value);
  }

  filterPipelineSourceDeliveries(value: string) {
    this.filterPipelineSourceDeliveries$.next(value);
  }

  filterPointSourceDeliveries(value: string) {
    this.filterPointSourceDeliveries$.next(value);
  }

  filterProducts(value: string) {
    this.filterProducts$.next(value);
  }

  filterBrokers(value: string) {
    this.filterBrokers$.next(value);
  }

  filterBrokerAccounts(value: string) {
    this.filterBrokerAccounts$.next(value);
  }

  filterContracts(value: string) {
    this.filterContracts$.next(value);
  }

  filterVolumeTypes(value: string) {
    this.filterVolumeTypes$.next(value);
  }

  filterFuelCalculationTypes(value: string) {
    this.filterFuelCalculationTypes$.next(value);
  }

  filterForceMajeures(value: string) {
    this.filterForceMajeures$.next(value);
  }

  filterWaspNums(value: string) {
    this.filterWaspNums$.next(value);
  }

  filterDeliveryMethods(value: string) {
    this.filterDeliveryMethods$.next(value);
  }

  counterpartyChanged(value: models.CounterpartyInfo) {
    const counterpartyId = value?.counterpartyId ?? null;
    this.counterpartyChanged$.next(counterpartyId);

    const internalEntityId = this.deal('internalEntityId');
    this.refreshContractsForEntities$.next([counterpartyId, internalEntityId])
    this.refreshWaspNums$.next(null);
  }

  internalEntityChanged(value: util.IdNameActive) {
    const counterpartyId = this.deal('counterpartyId');
    const internalEntityId = value?.id ?? null;
    this.refreshContractsForEntities$.next([counterpartyId, internalEntityId])
  }

  traderChanged(value: models.TraderInfo) {
    this.traderChanged$.next(value?.traderId ?? null);
  }

  transactionTypeChanged(transTypeId: number) {
    this.refreshWaspNums$.next(null);

    const defaultProductId = this.getDefaultProductIdForTransaction(transTypeId);
    this.formEditor.patchValue({ productId: defaultProductId });

    if (transTypeId === util.TransactionType.PhysicalCrudeOil) {
      this.formEditor.patchValue({ internalEntityId: 739 }); //VISION ENERGY LLC
      this.formEditor.patchValue({ deliveryModeId: 4 });
    }
    else {
      this.formEditor.patchValue({ internalEntityId: 315 }); //SUPERIOR NATURAL GAS CORPORATION
    }
    setTimeout(() => {
      this.filterCounterparties$.next(null);
      this.filterPipelines$.next(null);
      this.filterProducts$.next(null);
      this.productChanged$.next(this.deal('productId'));
    });
  }

  getDefaultProductIdForTransaction(transactionTypeId: number): number {
    if (transactionTypeId == util.TransactionType.PhysicalGas)
      return Product.NaturalGas;
    else if (transactionTypeId == util.TransactionType.PhysicalNGL)
      return Product.UnspecifiedNgl;
    else if (transactionTypeId == util.TransactionType.Futures)
      return null;
    else if (transactionTypeId == util.TransactionType.PhysicalCrudeOil)
      return Product.CrudeOil;
    else
      return null;
  }

  dealPurposeChanged(newItem: util.IdName) {
    if (newItem === undefined) {
      this.formEditor.patchValue({ dealPurposeId: null });
      return;
    }

    this.refreshWaspNums$.next(null);
    const isBuy = this.deal('isBuy');
    const isAutoNetbackDealPurpose =
      newItem.id == util.DealPurpose.Cashout ||
      newItem.id == util.DealPurpose.PTR ||
      newItem.id == util.DealPurpose.LU;

    if (isBuy && isAutoNetbackDealPurpose) {
      this.formEditor.patchValue({ isFixedPrice: true, fixedPrice: 0, isNetback: true });
      this.notify.info('Fixed Price and Netback changed');
    }
  }

  waspNumChanged() {
    this.refreshWaspVolPrice$.next(null);
  }

  refreshContacts() {
    this.refreshContacts$.next(null);
  }

  pipelineChanged(value: PipeInfo) {
    this.pipelineChanged$.next(value?.pipeId ?? null);
    if (this.deal('transactionTypeId') === util.TransactionType.PhysicalGas || this.deal('transactionTypeId') === util.TransactionType.PhysicalCrudeOil) {
      this.formEditor.patchValue({ pipelineSourceDeliveryId: value?.pipeId ?? null });
      this.pipelineSourceDeliveryChanged$.next(value?.pipeId ?? null);
    }
  }

  pipelineSourceDeliveryChanged(value: PipeInfo) {
    this.pipelineSourceDeliveryChanged$.next(value?.pipeId ?? null);
  }

  pointChanged(value: models.PointInfo) {
    const deal: models.DealDetail = this.formEditor.getRawValue();
    if ((deal.transactionTypeId === 1 || deal.transactionTypeId === util.TransactionType.PhysicalCrudeOil) && deal?.pipelineId === deal?.pipelineSourceDeliveryId)
      this.formEditor.patchValue({ pointSourceDeliveryIds: value?.pointId ? [value.pointId] : null });
  }

  productChanged(value: util.IdName) {
    this.productChanged$.next(value?.id ?? null);
    this.refreshWaspVolPrice$.next(null);
  }

  brokerChanged(value: util.IdName) {
    this.brokerChanged$.next(value?.id ?? null);
    this.refreshWaspVolPrice$.next(null);
  }

  deliveryMethodChanged(value: util.IdName) {
    console.log(value);
  }

  numOfContractsChanged() {
    this.refreshWaspVolPrice$.next(null);
  }

  isBuyChanged() {
    this.refreshWaspVolPrice$.next(null);
  }

  forceMajeureChanged(value: models.ForceMajeureInfo) {
    this.formEditor.patchValue({ fmlanguage: value?.language ?? null });
  }

  startDateValueChange() {
    this.startDateChanged$.next(this.deal('startDate'));
    const deal: models.DealDetail = this.formEditor.getRawValue();
    const isStartDateValid = this.formEditor.get('startDate').valid;
    if (isStartDateValid && deal.startDate) {
      this.formEditor.patchValue({ endDate: deal.startDate });
      this.formEditor.patchValue({ accountingMonth: dayjs(deal.startDate).startOf('month').toDate() });
    }
    this.refreshWaspNums$.next(null);
  }

  startDateBlur() {
    let deal: models.DealDetail = this.formEditor.getRawValue();
    const isStartDateValid = this.formEditor.get('startDate').valid;
    if (isStartDateValid && !dayjs(this.previousStartDate).isSame(deal.startDate)) {
      this.configureFormVariableVolumes(
        this.deal('isVariableVolume'),
        this.deal('startDate'),
        this.deal('endDate'),
        this.deal('volumeTypeId'),
        this.deal('volume')
      );
      deal = this.formEditor.getRawValue();
    }
    this.previousStartDate = new Date(deal.startDate);
    this.previousEndDate = new Date(deal.endDate);
  }

  endDateChanged() {
    let deal: models.DealDetail = this.formEditor.getRawValue();
    const isEndDateValid = this.formEditor.get('endDate').valid;
    if (isEndDateValid && !dayjs(this.previousEndDate).isSame(deal.endDate)) {
      if (deal.startDate && deal.endDate && deal.startDate > deal.endDate)
        this.formEditor.patchValue({ startDate: deal.endDate });

      if (deal.startDate && deal.endDate) {
        const minStartDate = dayjs(deal.endDate).add(-1, 'year').toDate();
        if (deal.startDate < minStartDate)
          this.formEditor.patchValue({ startDate: minStartDate });
      }
      this.configureFormVariableVolumes(
        this.deal('isVariableVolume'),
        this.deal('startDate'),
        this.deal('endDate'),
        this.deal('volumeTypeId'),
        this.deal('volume')
      );
    }
    deal = this.formEditor.getRawValue();
    this.previousStartDate = new Date(deal.startDate);
    this.previousEndDate = new Date(deal.endDate);
  }

  accountingMonthChanged() {
    this.refreshWaspVolPrice$.next(null);
  }

  isVariableVolumeChange(isVariable: boolean) {
    if (isVariable) { //switching to variable volume
      const contractualVolume: number = this.deal('volume');
      if (!util.isNullOrWhitespace(contractualVolume)) { //if the user entered a volume
        this.configureFormVariableVolumes(
          isVariable,
          this.deal('startDate'),
          this.deal('endDate'),
          this.deal('volumeTypeId'),
          this.deal('volume')
        );
        this.openVariableVolumeEditor();
      } //if the user did not enter a volume
      else {
        this.formEditor.patchValue({ isVariableVolume: false });
        this.notify.warning('Contractual volume required to set variable volume');
      }
    }
    if (!isVariable) { //switching off variable volume
      const deal: models.DealDetail = this.formEditor.getRawValue();
      deal.sourceDeliveryPointItems.forEach(item => {
        if (item.pointVolume === null)
          item.pointVolume = deal.volume;
      });
      this.formEditor.patchValue({ sourceDeliveryPointItems: deal.sourceDeliveryPointItems });
      if (this.isVariableVolumeEditorOpened) {
        this.closeVariableVolumeEditor();
      }
    }
  }

  volumeTypeChanged() {
    setTimeout(() => this.configureFormVariableVolumes(
      this.deal('isVariableVolume'),
      this.deal('startDate'),
      this.deal('endDate'),
      this.deal('volumeTypeId'),
      this.deal('volume')
    ));
  }

  priceChange(isIndexControl: boolean) {
    const deal: models.DealDetail = this.formEditor.getRawValue();
    if (isIndexControl && (deal.priceIndexId || deal.costPremInc || typeof deal.premiumOrDiscount === 'number'))
      this.formEditor.patchValue({ isFixedPrice: false });
    else
      this.formEditor.patchValue({ isFixedPrice: true });
    this.refreshWaspVolPrice$.next(null);
  }

  transactionTypeSubscription: Subscription;
  setupTransactionTypeChange() {
    this.transactionTypeSubscription = this.formEditor.get('transactionTypeId').valueChanges.subscribe(transactionTypeId => {
      const controlPipeDelivery = this.formEditor.get('pipelineSourceDeliveryId');
      const controlNumOfContracts = this.formEditor.get('numOfContracts');
      const controlVolume = this.formEditor.get('volume');
      const controlProduct = this.formEditor.get('productId');
      const controlPipeline = this.formEditor.get('pipelineId');
      const controlPoint = this.formEditor.get('pointId');
      const controlFixedPrice = this.formEditor.get('fixedPrice');
      const controlPriceIndex = this.formEditor.get('priceIndexId');
      const controlPriceIndex2 = this.formEditor.get('priceIndexId2');
      const controlDeliveryMethod = this.formEditor.get('deliveryModeId');
      const controlCounterparty = this.formEditor.get('counterpartyId');
      const controlSourceDeliveryPointItems = this.formEditor.get('sourceDeliveryPointItems');

      if (transactionTypeId === util.TransactionType.PhysicalGas) {
        controlPipeDelivery.setValidators([Validators.required]);
        controlVolume.setValidators([Validators.required]);
        controlNumOfContracts.setValidators(null);
        controlPipeline.setValidators([Validators.required]);
        controlPoint.setValidators([Validators.required]);
        controlProduct.setValidators([Validators.required]);
        controlDeliveryMethod.setValidators(null);
        controlCounterparty.setValidators([Validators.required]);
        controlSourceDeliveryPointItems.setValidators([Validators.required]);
        controlPriceIndex2.setValidators(null);
      } else if (transactionTypeId === util.TransactionType.Futures) {
        controlPipeDelivery.setValidators(null);
        controlVolume.setValidators(null);
        controlNumOfContracts.setValidators([Validators.required]);
        controlPipeline.setValidators(null);
        controlPoint.setValidators(null);
        controlProduct.setValidators([Validators.required]);
        controlFixedPrice.setValidators([Validators.required]);
        controlPriceIndex.setValidators(null);
        controlPriceIndex2.setValidators(null);
        controlDeliveryMethod.setValidators(null);
        controlCounterparty.setValidators(null);
        controlSourceDeliveryPointItems.setValidators(null);
      } else if (transactionTypeId === util.TransactionType.SwingSwaps) {
        controlPipeDelivery.setValidators(null);
        controlVolume.setValidators([Validators.required]);
        controlNumOfContracts.setValidators(null);
        controlPipeline.setValidators(null);
        controlPoint.setValidators(null);
        controlProduct.setValidators([Validators.required]);
        controlFixedPrice.setValidators(null);
        controlPriceIndex.setValidators([Validators.required]);
        controlPriceIndex2.setValidators([Validators.required]);
        controlDeliveryMethod.setValidators(null);
        controlCounterparty.setValidators(null);
        controlSourceDeliveryPointItems.setValidators(null);
      }
      else if (transactionTypeId == util.TransactionType.PhysicalNGL) {
        controlPipeDelivery.setValidators(null);
        controlVolume.setValidators(null);
        controlNumOfContracts.setValidators(null);
        controlPipeline.setValidators(null);
        controlPoint.setValidators(null);
        controlProduct.setValidators([Validators.required]);
        controlFixedPrice.setValidators(null);
        controlPriceIndex.setValidators(null);
        controlPriceIndex2.setValidators(null);
        controlDeliveryMethod.setValidators(null);
        controlCounterparty.setValidators([Validators.required]);
        controlSourceDeliveryPointItems.setValidators(null);
      }
      else if (transactionTypeId === util.TransactionType.PhysicalCrudeOil) {
        controlPipeDelivery.setValidators(null);
        controlVolume.setValidators([Validators.required]);
        controlNumOfContracts.setValidators(null);
        controlPipeline.setValidators(null);
        controlPoint.setValidators(null);
        controlProduct.setValidators([Validators.required]);
        controlDeliveryMethod.setValidators([Validators.required]);
        controlCounterparty.setValidators([Validators.required]);
        controlSourceDeliveryPointItems.setValidators([Validators.required]);
        controlPriceIndex2.setValidators(null);
      }

      if (transactionTypeId === util.TransactionType.PhysicalCrudeOil)
        this.priceUnit$ = '$/BBL';
      else
        this.priceUnit$ = '$/MMBTU';

      controlPipeDelivery.updateValueAndValidity();
      controlVolume.updateValueAndValidity();
      controlNumOfContracts.updateValueAndValidity();
      controlPipeline.updateValueAndValidity();
      controlPoint.updateValueAndValidity();
      controlProduct.updateValueAndValidity();
      controlFixedPrice.updateValueAndValidity();
      controlPriceIndex.updateValueAndValidity();
      controlPriceIndex2.updateValueAndValidity();
      controlCounterparty.updateValueAndValidity();
      controlSourceDeliveryPointItems.updateValueAndValidity();
    });
  }

  fixedPriceSubscription: Subscription;
  setupIsFixedPriceChange() {
    this.fixedPriceSubscription = this.formEditor.get('isFixedPrice').valueChanges.subscribe(isFixedPrice => {
      if (this.localDealDetail.transactionTypeId === util.TransactionType.PhysicalGas || this.localDealDetail.transactionTypeId === util.TransactionType.PhysicalCrudeOil) {
        const fixedPriceControl = this.formEditor.get('fixedPrice');
        const priceIndexControl = this.formEditor.get('priceIndexId');
        if (isFixedPrice) {
          fixedPriceControl.setValidators([Validators.required]);
          priceIndexControl.setValidators(null);
          this.formEditor.patchValue({ priceIndexId: null, premiumOrDiscount: null, costPremInc: null });
        }
        else {
          fixedPriceControl.setValidators(null);
          priceIndexControl.setValidators([Validators.required]);
          this.formEditor.patchValue({ fixedPrice: null, basis: null, costBasisInc: null });
        }
        fixedPriceControl.updateValueAndValidity();
        priceIndexControl.updateValueAndValidity();
      }
    });
  }

  variableVolumeSubscription: Subscription;
  setupVariableVolumeChange() {
    this.variableVolumeSubscription = this.formEditor.get('isVariableVolume').valueChanges.subscribe(isVariableVolume => {
      if (this.localDealDetail.transactionTypeId === util.TransactionType.PhysicalGas || this.localDealDetail.transactionTypeId === util.TransactionType.PhysicalCrudeOil) {
        const controlVolume = this.formEditor.get('volume');
        if (isVariableVolume)
          controlVolume.setValidators(null);
        else
          controlVolume.setValidators([Validators.required]);
        controlVolume.updateValueAndValidity();
      }
    });
  }

  dealPurposeSubscription: Subscription;
  setupDealPurposeChange() {
    this.dealPurposeSubscription = this.formEditor.get('dealPurposeId').valueChanges.subscribe(dealPurposeId => {
      this.refreshConfirmButton$.next(dealPurposeId);
      const transactionTypeId = this.deal('transactionTypeId');
      const physicalTransTypeIds = [util.TransactionType.PhysicalGas, util.TransactionType.PhysicalNGL, util.TransactionType.PhysicalCrudeOil]
      const isPhysical = physicalTransTypeIds.includes(transactionTypeId);
      const isAccounting = this.accountingDealPurposes.includes(dealPurposeId);
      const contactControl = this.formEditor.get('contactId');
      const contractNumberControl = this.formEditor.get('contractNumber');

      if (!isPhysical || isAccounting) {
        contactControl.setValidators(null);
        contractNumberControl.setValidators(null);
      } else {
        contactControl.setValidators([Validators.required]);
        contractNumberControl.setValidators([Validators.required]);
      }

      contactControl.updateValueAndValidity();
      contractNumberControl.updateValueAndValidity();
    });
  }

  getVolumeTooltipItems(): volumeTooltipItem[] {
    const deal: models.DealDetail = this.formEditor.getRawValue();
    const tooltipStrItems = deal.dealVolumes.map(dealVol => {
      const label = formatDate(dealVol.startDate, 'M/d/yyyy');
      const formatStr = deal.transactionTypeId === 11 ? '#,###.##' : '#,###';
      const valueText = this.kendoFormatNumber(dealVol.physicalVolume, formatStr);
      const tooltipItem: volumeTooltipItem = { label: label, valueText: valueText };
      return tooltipItem;
    });
    return tooltipStrItems;
  }

  kendoFormatNumber(num: number, format: string) {
    return formatNumber(num, format);
  }

  openVariableVolumeEditor() {
    if (this.deal('isVariableVolume')) {
      this.util.hideTooltips(this.allTips);
      this.isVariableVolumeEditorOpened = true;
      setTimeout(() => {
        const firstVolumeInputElem = document.querySelector('.variableVolumeEditor [formcontrolname="physicalVolume"] input') as HTMLInputElement;
        if (firstVolumeInputElem) {
          firstVolumeInputElem.focus();
          firstVolumeInputElem.select();
        }
      });
    }
  }

  closeVariableVolumeEditor() {
    this.isVariableVolumeEditorOpened = false;
    const pipelineElem = this.pipelineComboElem.searchbar.input.nativeElement as HTMLInputElement;
    pipelineElem.focus();
    pipelineElem.select();
  }

  onContextMenuSelect(e: ContextMenuSelectEvent, controlName: string, index: number) {
    const contextMenuItem: ContextMenuItem = e.item;

    if (contextMenuItem.enum === contextMenuEnum.CopyUp)
      this.CopyUp(controlName, index, true);
    else if (contextMenuItem.enum === contextMenuEnum.CopyDown)
      this.CopyDown(controlName, index, true);
  }

  CopyUp(controlName: string, index: number, markDirty: boolean) {
    const controls = this.formEditorVariableVolumes.controls;
    const valueToCopy = this.formEditorVariableVolumes.controls[index].value[controlName];

    for (let i = index - 1; i >= 0; i--) {
      controls[i].patchValue({ [controlName]: valueToCopy });
      if (markDirty)
        controls[i].markAsDirty();
    }
  }

  CopyDown(controlName: string, index: number, markDirty: boolean) {
    const controls = this.formEditorVariableVolumes.controls;
    const valueToCopy = this.formEditorVariableVolumes.controls[index].value[controlName];

    for (let i = index + 1; i < controls.length; i++) {
      controls[i].patchValue({ [controlName]: valueToCopy });
      if (markDirty)
        controls[i].markAsDirty();
    }
  }

  restOfMonth() {
    if (this.formEditor.controls['startDate'].valid && this.deal('startDate')) {
      const startDate = dayjs(this.deal('startDate')).startOf('day').toDate();
      const endOfMonth = dayjs(startDate).endOf('month').startOf('day').toDate();
      if (!dayjs(startDate).isSame(endOfMonth)) {
        this.formEditor.patchValue({ endDate: endOfMonth });
        this.formEditor.controls['endDate'].markAsDirty();
        this.configureFormVariableVolumes(
          this.deal('isVariableVolume'),
          this.deal('startDate'),
          this.deal('endDate'),
          this.deal('volumeTypeId'),
          this.deal('volume')
        );
      }
    }
  }

  getRowClass(context: RowClassArgs) {
    const deal: models.DealListItem = context.dataItem;
    const needsRedistribute = deal.Distributed && deal.Modified && deal.Modified > deal.Distributed;
    if (needsRedistribute)
      return { 'needsRedistribute': true };
    else
      return {};
  }

  getColumnClass(propName: string) {
    if (propName === 'DealNum' || propName === 'EmailDate' || propName === 'FaxDate')
      return { isDistributeColumn: true };
    else
      return {};
  }

  createNewWaspNum() {
    this.addNewWaspNum = true;
    this.refreshWaspNums$.next(null);
  }

  detailWindowStateChange(windowState: WindowState) {
    if (windowState == 'normal') {
      let newLeftPosition: number;
      newLeftPosition = (window.innerWidth - this.detailWindow.width) / 2;
      newLeftPosition = newLeftPosition > 0 ? newLeftPosition : 0;
      this.detailWindow.setOffset('left', newLeftPosition);
      this.detailWindow.setOffset('top', 50);
    }
  }

  volumeChange(volume: number) {
    if (this.localDealDetail.transactionTypeId === util.TransactionType.PhysicalGas || this.localDealDetail.transactionTypeId === util.TransactionType.PhysicalCrudeOil) {
      const deal: models.DealDetail = this.formEditor.getRawValue();
      const sourceDeliveryVol = deal.isVariableVolume ? null : volume;
      deal.sourceDeliveryPointItems.forEach(item => {
        item.pointVolume = sourceDeliveryVol;
      });
      this.formEditor.patchValue({ sourceDeliveryPointItems: deal.sourceDeliveryPointItems });
    }
  }
}
