import { Injectable, inject } from '@angular/core';
import { HttpClient } from '@angular/common/http';
import { map } from 'rxjs/operators';
import { Observable } from 'rxjs';
import { FilterValue } from '../filter-controls/control-base.component';

export interface DealFilter {
  id: number;
  name: string;
  isSelected: boolean;
  columns: string;
  state: string;
}

export interface FilterData {
  filterParams: FilterValue[];
  selectedColumns: string[];
  state: string;
}

export interface DisplayInfo {
  propName: string;
  displayName: string;
  width: number;
  filter: "boolean" | "text" | "numeric" | "date";
  format: string;
}

@Injectable({
  providedIn: 'root'
})
export class DealFilterService {
  private baseUrl = `${window.location.origin}/api/DealFilter`;
  public http = inject(HttpClient)

  constructor() { }

  getFilters(): Observable<DealFilter[]> {
    const url = `${this.baseUrl}/GetFilters`;
    return this.http.get<DealFilter[]>(url);
  }

  getAllColumnDisplayInfos(): Observable<DisplayInfo[]> {
    const url = `${this.baseUrl}/GetAllColumnDisplayInfos`;
    return this.http.get<DisplayInfo[]>(url);
  }

  addFilter(name: string): Observable<number> {
    const url = `${this.baseUrl}/AddFilter?name=${name}`;
    return this.http.post<number>(url, null);
  }

  renameFilter(id: number, name: string): Observable<string> {
    const url = `${this.baseUrl}/RenameFilter?id=${id}&name=${name}`;
    return this.http.post(url, null, { observe: 'response', responseType: 'text' }).pipe(
      map(result => {
        return result.body;
      })
    );
  }

  deleteFilter(id: number) {
    const url = `${this.baseUrl}/DeleteFilter/${id}`;
    return this.http.delete(url);
  }

  saveFilter(filterId: number, filterValues: FilterValue[], columnPropsOrdered: string[]) {
    const filterIdStr = filterId ?? '';
    const url = `${this.baseUrl}/SaveFilter?filterId=${filterIdStr}`;
    return this.http.put(url, { filterValues, columnPropsOrdered });
  }

  getFilterData(filterId: number): Observable<FilterData> {
    const url = `${this.baseUrl}/GetFilterData?filterId=${filterId}`;
    return this.http.get<FilterData>(url);
  }

  setFilterSelected(filterId: number) {
    const filterIdStr = filterId ?? '';
    const url = `${this.baseUrl}/SetFilterSelected?filterId=${filterIdStr}`;
    return this.http.put(url, null);
  }
}
