import { Component, ChangeDetectionStrategy, ChangeDetectorRef, ViewEncapsulation, Input, Output, EventEmitter, QueryList, ViewChildren, inject } from '@angular/core';
import { Title } from '@angular/platform-browser';
import { tap, take, map, catchError, switchMap, shareReplay, filter, delay } from 'rxjs/operators';
import { of, BehaviorSubject, Subject, combineLatest, Observable, pipe } from 'rxjs';
import { DealFilterService, DealFilter, DisplayInfo } from './deal-filter.service';
import { MessageService, promptAction, PromptSettings } from '../_shared/services/message.service';
import { FormsModule, UntypedFormBuilder } from '@angular/forms';
import { DialogService } from '@progress/kendo-angular-dialog';
import { NotifyService } from '../_shared/services/notify.service';
import * as util from '../_shared/utils/util';
import dayjs from 'dayjs';
import { ControlBaseComponent } from '../filter-controls/control-base.component';
import { FilterValue } from '../filter-controls/control-base.component';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../app.config';
import { KENDO_SORTABLE } from '@progress/kendo-angular-sortable';
import { ControlBookComponent } from '../filter-controls/control-book/control-book.component';
import { ControlBrokerComponent } from '../filter-controls/control-broker/control-broker.component';
import { ControlBrokerAccountComponent } from '../filter-controls/control-broker-account/control-broker-account.component';
import { ControlBuySellComponent } from '../filter-controls/control-buy-sell/control-buy-sell.component';
import { ControlProductComponent } from '../filter-controls/control-product/control-product.component';
import { ControlCounterpartyComponent } from '../filter-controls/control-counterparty/control-counterparty.component';
import { ControlDealPurposeComponent } from '../filter-controls/control-deal-purpose/control-deal-purpose.component';
import { ControlDealStatusComponent } from '../filter-controls/control-deal-status/control-deal-status.component';
import { ControlDealTypeComponent } from '../filter-controls/control-deal-type/control-deal-type.component';
import { ControlHypotheticalComponent } from '../filter-controls/control-hypothetical/control-hypothetical.component';
import { ControlInternalEntityComponent } from '../filter-controls/control-internal-entity/control-internal-entity.component';
import { ControlPipelineComponent } from '../filter-controls/control-pipeline/control-pipeline.component';
import { ControlPointComponent } from '../filter-controls/control-point/control-point.component';
import { ControlPortfolioComponent } from '../filter-controls/control-portfolio/control-portfolio.component';
import { ControlPriceIndexComponent } from '../filter-controls/control-price-index/control-price-index.component';
import { ControlDateRangeComponent } from '../filter-controls/control-date-range/control-date-range.component';
import { ControlRegionComponent } from '../filter-controls/control-region/control-region.component';
import { ControlStrategyComponent } from '../filter-controls/control-strategy/control-strategy.component';
import { ControlTraderComponent } from '../filter-controls/control-trader/control-trader.component';
import { ControlTransactionTypeComponent } from '../filter-controls/control-transaction-type/control-transaction-type.component';
import { FastSortableComponent } from '../_shared/elements/fast-sortable.component';

@Component({
  selector: 'deal-filter',
  templateUrl: './deal-filter.component.html',
  changeDetection: ChangeDetectionStrategy.OnPush,
  encapsulation: ViewEncapsulation.None,
  imports: [FAST_PAGE_COMMON, FAST_KENDO_COMMON, KENDO_SORTABLE, FormsModule,
    ControlBookComponent,
    ControlBrokerComponent,
    ControlBrokerAccountComponent,
    ControlBuySellComponent,
    ControlProductComponent,
    ControlCounterpartyComponent,
    ControlDealPurposeComponent,
    ControlDealStatusComponent,
    ControlDealTypeComponent,
    ControlHypotheticalComponent,
    ControlInternalEntityComponent,
    ControlPipelineComponent,
    ControlPointComponent,
    ControlPortfolioComponent,
    ControlPriceIndexComponent,
    ControlDateRangeComponent,
    ControlRegionComponent,
    ControlStrategyComponent,
    ControlTraderComponent,
    ControlTransactionTypeComponent,
    FastSortableComponent
  ]
})
export class DealFilterComponent {
  @ViewChildren('childControl') allChildControls: QueryList<ControlBaseComponent>;
  @Output() closed = new EventEmitter()
  @Input() filterId: number

  util = util;
  selectedFilter: DealFilter;
  editorOpened = false;
  isAddNew: boolean;
  newFilterName: string;
  selectedDataSource: string;
  expanded = true;
  hasValueCount = 0;
  resetLocalOrderables = true;

  localDataSourcesById: Record<number, string> = {}
  localOrderables: DisplayInfo[] = [];
  localAllColumnDisplayInfos: DisplayInfo[]

  private messageService = inject(MessageService);
  private titleService = inject(Title);
  private filterService = inject(DealFilterService);
  private fb = inject(UntypedFormBuilder);
  private ref = inject(ChangeDetectorRef);
  private dialogService = inject(DialogService);
  private notify = inject(NotifyService);

  constructor() {
  }

  loading$ = new BehaviorSubject<boolean>(true);

  refreshFilters$ = new BehaviorSubject<number>(null)
  refreshDataSources$ = new BehaviorSubject(null)
  filterFilters$ = new BehaviorSubject<string>(null)
  filterAllColumnDisplayInfos$ = new BehaviorSubject<string>(null)

  filterSelected$ = new Subject<DealFilter>()
  columnSelected$ = new BehaviorSubject<DisplayInfo>(null)

  saveFilter$ = new Subject()
  addFilter$ = new Subject<string>()
  renameFilter$ = new Subject<string>()
  deleteFilter$ = new Subject()

  allColumnDisplayInfos$ = this.filterService.getAllColumnDisplayInfos().pipe(
    tap(allColumnDisplayInfos => {
      this.localAllColumnDisplayInfos = allColumnDisplayInfos;
    })
  );

  filters$ = this.refreshFilters$.pipe(
    tap(() => {
      this.loading$.next(true);
    }),
    switchMap((addedFilterId) => {
      return combineLatest([of(addedFilterId), this.filterService.getFilters()]);
    }),
    map(([addedFilterId, filters]) => {
      if (addedFilterId) {
        this.selectedFilter = filters.find(x => x.id === addedFilterId);
        this.filterSelected$.next(this.selectedFilter);
      }
      else if (filters?.length > 0) {
        const inputFilter = filters.find(x => x.id === this.filterId) ?? null;
        if (inputFilter)
          this.selectedFilter = inputFilter;
        else
          this.selectedFilter = filters.find(x => x.isSelected === true) ?? null;
        this.filterSelected$.next(this.selectedFilter);
      }
      else {
        this.selectedFilter = null;
        this.filterSelected$.next(this.selectedFilter);
      }

      return filters;
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    })
  );

  filterData$ = combineLatest([this.filterSelected$, this.columnSelected$, this.allColumnDisplayInfos$]).pipe(
    filter(([filterSelected, , displayInfos]) => {
      return filterSelected !== null && filterSelected.id !== 0 && displayInfos.length > 0;
    }),
    tap(() => {
      this.loading$.next(true);
    }),
    switchMap(([filterSelected, columnSelected]) => {
      return combineLatest([this.filterService.getFilterData(filterSelected.id), of(columnSelected)]);
    }),
    map(([filterData, columnSelected]) => {
      if (this.resetLocalOrderables) {
        columnSelected = null;
        if (filterData.selectedColumns) {
          this.localOrderables = filterData.selectedColumns.map(col => {
            const colSplit = col.split(':');
            let colInfo = this.localAllColumnDisplayInfos.find(y => y.propName === colSplit[0]);
            colInfo = { ...colInfo } //clone so that we don't modify the original widths
            if (colSplit.length === 2) //if has width
              colInfo.width = parseInt(colSplit[1]);
            return colInfo;
          })
        }
        this.resetLocalOrderables = false;
      }
      if (columnSelected) {
        const hasColumn = this.localOrderables.findIndex(x => x.propName === columnSelected.propName) !== -1;
        if (!hasColumn)
          this.localOrderables.push(columnSelected);
      }

      this.localOrderables = [...this.localOrderables];
      return filterData;
    }),
    tap(filterData => {
      this.setFilterValues(filterData.filterParams);
      this.loading$.next(false);
    }),
    shareReplay(1),
    catchError(err => {
      this.loading$.next(false);
      return util.handleError(err, this.messageService)
    })
  )

  setFilterValues(params: FilterValue[]) {
    const childControlsByName: Record<string, ControlBaseComponent> = {};
    this.allChildControls.map(childControl => {
      childControlsByName[childControl.parameterName] = childControl;
      childControl.resetValue();
    })

    params.forEach(param => {
      const control = childControlsByName[param.name];
      if (control)
        control.setFilterValue(param);
    })
  }

  filterDisplayInfos = (loading: Observable<boolean>, allData: Observable<DisplayInfo[]>) => pipe(
    switchMap((filterText: string) => combineLatest([loading, allData]).pipe(
      delay(100),
      map(([loading, allData]) => {
        if (loading) //if loading then the control has just appeared and we need to reset the filter text
          filterText = null;
        const idNames: DisplayInfo[] = allData;
        return !filterText ? idNames : idNames.filter(item => item.displayName.toLowerCase().indexOf(filterText.toLowerCase()) !== -1)
      })
    ))
  )

  filtersFiltered$ = this.filterFilters$.pipe(util.filterIdNames(this.loading$, this.filters$, null));
  allColumnDisplayInfosFiltered$ = this.filterAllColumnDisplayInfos$.pipe(this.filterDisplayInfos(this.loading$, this.allColumnDisplayInfos$));

  addFilterResult$ = this.addFilter$.pipe(
    switchMap(newFilterName => {
      this.filterFilters$.next(null);
      return this.filterService.addFilter(newFilterName);
    }),
    tap(addedFilterId => {
      this.notify.success('filter added');
      this.loading$.next(false);
      this.editorOpened = false;
      this.refreshFilters$.next(addedFilterId);
    }),
    shareReplay(1),
    catchError(err => {
      this.loading$.next(false);
      return util.handleError(err, this.messageService)
    })
  )

  renameFilterResult$ = this.renameFilter$.pipe(
    switchMap((newFilterName) => {
      return this.filterService.renameFilter(this.selectedFilter.id, newFilterName);
    }),
    tap((newFilterName) => {
      this.selectedFilter.name = newFilterName;
      const filterInput = document.querySelector('.filterSelection .k-input') as HTMLInputElement;
      filterInput.value = newFilterName;
      this.notify.success('rename successful');
      this.loading$.next(false);
      this.editorOpened = false;
    }),
    shareReplay(1),
    catchError(err => {
      this.loading$.next(false);
      return util.handleError(err, this.messageService)
    })
  )

  deleteFilterResult$ = this.deleteFilter$.pipe(
    switchMap(() => {
      return this.filterService.deleteFilter(this.selectedFilter.id);
    }),
    tap(() => {
      this.notify.success('delete successful');
      this.loading$.next(false);
      this.editorOpened = false;
      this.refreshFilters$.next(null);
    }),
    shareReplay(1),
    catchError(err => {
      this.loading$.next(false);
      return util.handleError(err, this.messageService)
    })
  )

  saveFilter(): void {
    this.loading$.next(true);
    let areAllChildControlsValid = true;
    this.allChildControls.map(childControl => {
      if (!childControl.isValid)
        areAllChildControlsValid = false;
    })

    if (areAllChildControlsValid) {
      let allValues: FilterValue[] = this.getFilterValues();
      allValues = allValues.filter(x => x.value1 !== null);

      const columnPropsOrdered = this.localOrderables.map(x => x.propName + ":" + x.width);
      this.filterService.saveFilter(this.selectedFilter?.id, allValues, columnPropsOrdered).pipe(
        take(1),
        tap(() => {
          this.loading$.next(false);
          this.closed.emit(this.selectedFilter?.id);
          this.notify.success('filter saved');
        }),
        catchError(err => {
          return util.handleError(err, this.messageService)
        }),
      ).subscribe();
    }
    else {
      this.loading$.next(false);
      this.notify.error("validation failed");
    }
  }

  getFilterValues(): FilterValue[] {
    const values: FilterValue[] = [];
    this.allChildControls.forEach(childControl => {
      const value: FilterValue = childControl.getFilterValue();
      values.push(value);
    })
    return values;
  }

  closeFilter(): void {
    this.closed.emit(this.selectedFilter?.id);
  }

  openEditor(isAddNew: boolean): void {
    this.isAddNew = isAddNew;

    if (isAddNew) {
      this.newFilterName = 'New Filter ' + dayjs().format('YYYY-MM-DD h:mm:ss a');
      // TODO: double check on if we need these
      // util.focusAndSelectInputTarget();
    }
    else {
      this.newFilterName = this.selectedFilter.name;
        // TODO: double check on if we need these
      // util.focusAndSelectInputTarget();
    }

    this.editorOpened = true;
  }

  addFilter(): void {
    if (!util.isNullOrWhitespace(this.newFilterName)) {
      this.loading$.next(true);
      this.resetLocalOrderables = true;
      this.addFilter$.next(this.newFilterName);
    } else
      this.notify.error("validation failed");
  }

  renameFilter(): void {
    this.loading$.next(true);
    this.renameFilter$.next(this.newFilterName);
  }

  deleteFilter(): void {
    // const confirmSettings: DialogSettings = {
    //   title: "Please confirm",
    //   content: "Are you sure you want to delete this filter?",
    //   actions: [{ text: 'No' }, { text: 'Yes', primary: true }],
    //   cssClass: 'utilPrompt'
    // }

    const ps: PromptSettings = {
      title: "Please confirm",
      content: "Are you sure you want to delete this filter?",
      type: "Yes-No"
    }

    this.messageService.prompt(ps).then((result) => {
      if (result === promptAction.Yes) {
        this.loading$.next(true);
        this.resetLocalOrderables = true;
        this.deleteFilter$.next(null);
      }
    })

    // this.dialogService.open(confirmSettings).result.pipe(take(1)).subscribe(result => {
    //   if (util.getDialogAction(result) === util.dialogAction.Yes) {
    //     this.loading$.next(true);
    //     this.resetLocalOrderables = true;
    //     this.deleteFilter$.next(null);
    //   }
    // });
  }

  filterAllColumnDisplayInfos(value: string) {
    this.filterAllColumnDisplayInfos$.next(value);
  }

  filterFilters(value: string) {
    this.filterFilters$.next(value);
  }

  onFilterSelected($event: DealFilter) {
    this.resetLocalOrderables = true;
    this.selectedFilter = $event ?? null;
    this.filterSelected$.next(this.selectedFilter);
  }

  onColumnSelected($event: DisplayInfo) {
    this.columnSelected$.next($event);
  }

  hasDataSourceId(dataSources: util.IdName[], id: number): boolean {
    return dataSources.some(x => x.id === id);
  }

  isDataSourceIdSelected(dataSources: util.IdName[], id: number): boolean {
    let isSelected = false;
    const dataSource = dataSources.find(x => x.id === id)
    if (dataSource?.name === this.selectedDataSource)
      isSelected = true;

    return isSelected
  }

  controlValueUpdated() {
    if (this.allChildControls && this.localDataSourcesById) {
      let hasValueCount = 0;
      this.allChildControls.forEach(childControl => {
        if (childControl.hasValue)
          hasValueCount++;
      })
      this.hasValueCount = hasValueCount;
    }
  }

  removeOrderable(propName: string) {
    this.localOrderables = this.localOrderables.filter(x => x.propName !== propName);
  }
}
