import { inject, Injectable } from '@angular/core';
import { HttpClient } from '@angular/common/http';
import { OdataGridHelper } from '../_shared/utils/odata-grid-helper';
import { Observable } from 'rxjs';
import { State } from '@progress/kendo-data-query';
import { GridDataResult } from '@progress/kendo-angular-grid';
import * as util from '../_shared/utils/util';

export interface Item {
  Id: number;
  DisplayName: string;
  FullName: string;
  Title: string;
  City: string;
  State: string;
  Zip: string;
  Country: string;
  ContactCounterparties: string;
  ContactTypes: string;
}

export interface Detail {
  id: number;
  displayName: string;
  fullName: string;
  title: string;
  addressLine1: string;
  addressLine2: string;
  city: string;
  stateId: number;
  zip: string;
  countryId: number;
  officePhone: string;
  cellPhone: string;
  altPhone: string;
  fax: string;
  email: string;
  notes: string;
  contactCounterparties: util.IdDate[];
  contactTypeIds: number[];
}

export interface RequiredData {
  hasModifyPermission: boolean;
  contactTypes: util.IdName[];
  counterparties: util.IdNameActive[];
  states: util.IdName[];
  countries: util.IdName[];
}

export enum SaveType {
  New = 1,
  Normal = 2
}

@Injectable({
  providedIn: 'root'
})
export class ContactService extends OdataGridHelper {
  private baseUrl = `${window.location.origin}/api/Contact`;
  http = inject(HttpClient);

  requiredData$ = this.http.get<RequiredData>(`${this.baseUrl}/GetRequiredData`)

  getItems(state: State): Observable<GridDataResult> {
    return this.fetch(`GetContactItems`, state, null, false)
  }

  getDetail(id: number): Observable<Detail> {
    const url = `${this.baseUrl}/GetDetail?id=${id}`;
    return this.http.get<Detail>(url);
  }

  saveDetail(detail: Detail, saveType: SaveType): Observable<number> {
    const url = `${this.baseUrl}/SaveDetail?saveType=${saveType}`;
    detail.contactCounterparties = util.convertDatesToDateOnlyStrings(detail.contactCounterparties, ['date']);
    return this.http.put<number>(url, detail);
  }

  deleteDetail(id: number) {
    const url = `${this.baseUrl}/DeleteDetail/${id}`;
    return this.http.delete(url);
  }

  exportItems(state: State, fileNameOriginal: string) {
    return this.fetchExport(`GetContactItems`, state, null, fileNameOriginal);
  }
}
