import { Component, ChangeDetectionStrategy, ViewChild, ViewEncapsulation, HostListener } from '@angular/core';
import { Title } from '@angular/platform-browser';
import { tap, map, catchError, switchMap, filter, shareReplay, retry } from 'rxjs/operators';
import { of, BehaviorSubject, Subject, combineLatest, Observable } from 'rxjs';
import { State } from '@progress/kendo-data-query';
import { ContactService, Detail, Item, RequiredData, SaveType, } from './contact.service';
import { MessageService, promptAction, PromptSettings } from '../_shared/services/message.service';
import { Validators } from '@angular/forms';
import { TooltipDirective } from '@progress/kendo-angular-tooltip';
import * as util from '../_shared/utils/util';
import { SelectableSettings } from '@progress/kendo-angular-grid';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../app.config';
import { KENDO_SORTABLE } from '@progress/kendo-angular-sortable';
import { inject } from '@angular/core';
import { ToastService } from '../_shared/services/fast-toast.service';
import { GridDataResult } from '@progress/kendo-angular-grid';
import { FastSortableComponent } from '../_shared/elements/fast-sortable.component';
import { FastGridComponent } from '../_shared/elements/fast-grid.component';
import { CustomFormBuilder } from '../_shared/services/custom-form-builder.service';

@Component({
  selector: 'app-contact',
  templateUrl: './contact.component.html',
  encapsulation: ViewEncapsulation.None,
  changeDetection: ChangeDetectionStrategy.OnPush,
  imports: [FAST_PAGE_COMMON, FAST_KENDO_COMMON, KENDO_SORTABLE, FastSortableComponent]
})
export class ContactComponent {
  @ViewChild("grid") fastGrid: FastGridComponent;
  @ViewChild('tooltipGrid') tooltipGrid: TooltipDirective;
  @HostListener('window:resize') onResize() {
    //this function is empty but for some reason it helps the window to resize faster
  };

  detailForm: util.FormModel<Detail> = null;
  initialValues: Detail;
  hasModifyPermission = false;
  localCounterpartiesSelected: util.IdDate[] = [];
  localCounterparties: Record<number, util.IdNameActive> = {}
  localRequiredData: RequiredData;
  minDate: Date = new Date(1900, 0, 1);
  util = util;

  mySelection: number[] = [];
  selectableSettings: SelectableSettings = {
    checkboxOnly: false,
    mode: 'single',
    enabled: true
  }

  exporting$ = new BehaviorSubject<boolean>(false)
  detailOpened$ = new BehaviorSubject<boolean>(false)
  detailLoading$ = new BehaviorSubject<boolean>(true);
  gridLoading$ = new BehaviorSubject<boolean>(true)
  counterpartySelected$ = new BehaviorSubject<util.IdNameActive>(null)

  requiredData$: Observable<RequiredData>;
  items$: Observable<GridDataResult>;
  exportAction$: Observable<{ fileBlob: Blob; fileName: string; }>;
  detail$: Observable<Detail>;
  loadingComplete$: Observable<[Detail, RequiredData]>;
  filteredRequiredData$: Observable<RequiredData>;
  contactTypes$: Observable<util.IdName[]>;
  counterparties$: Observable<util.IdName[]>;
  states$: Observable<util.IdName[]>;
  countries$: Observable<util.IdName[]>;
  counterpartiesSelected$: Observable<util.IdDate[]>;
  saveResult$: Observable<number>;
  deleteResult$: Observable<object>;

  exportClicked$ = new Subject()

  refreshItems$ = new BehaviorSubject<string>(null)
  refreshDetail$ = new BehaviorSubject<number>(null)
  refreshRequiredData$ = new BehaviorSubject(null)
  refreshCounterpartySelected$ = new Subject<util.IdNameActive>()

  save$ = new Subject<SaveType>()
  delete$ = new Subject()


  state: State = {
    filter: null,
    group: null,
    skip: 0,
    sort: [{ field: 'DisplayName', dir: 'asc' }],
    take: 100
  };

  private messageService = inject(MessageService);
  private titleService = inject(Title);
  private service = inject(ContactService);
  private notify = inject(ToastService);
  private fb = inject(CustomFormBuilder);

  constructor() {
    this.detailForm = this.getDetailForm();
    this.initialValues = this.detailForm.getRawValue() as Detail;

    this.requiredData$ = this.refreshRequiredData$.pipe(
      switchMap(() => {
        return this.service.requiredData$;
      }),
      map(requiredData => {
        this.localRequiredData = requiredData;
        return requiredData;
      }),
      tap((requiredData) => {
        this.hasModifyPermission = requiredData.hasModifyPermission;
        requiredData.counterparties.forEach(counterparty => {
          this.localCounterparties[counterparty.id] = counterparty;
        });
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.items$ = this.refreshItems$.pipe(
      tap(() => {
        this.gridLoading$.next(true);
      }),
      switchMap(() => {
        return this.service.getItems(this.state);
      }),
      tap(() => {
        this.gridLoading$.next(false);
      }),
      shareReplay(1),
      catchError(err => {
        this.gridLoading$.next(false);
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.exportAction$ = this.exportClicked$.pipe(
      switchMap(() => {
        this.exporting$.next(true);
        return this.service.exportItems(this.state, 'Contacts.xlsx');
      }),
      tap(res => {
        util.openOrSaveFile(res.fileBlob, res.fileName);
        this.exporting$.next(false);
      }),
      shareReplay(1),
      catchError(err => {
        this.exporting$.next(false);
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.detail$ = this.refreshDetail$.pipe(
      filter(id => id !== null),
      switchMap(id => {
        this.detailLoading$.next(true);
        this.detailForm.reset();
        this.detailForm.disable();
        if (id === 0)
          return of(this.initialValues);
        else
          return this.service.getDetail(id);
      }),
      map(result => {
        if (result) {
          util.convertToDates(result);
          this.localCounterpartiesSelected = result.contactCounterparties ?? [];
          this.detailForm.setValue(result);
        }
        this.refreshCounterpartySelected$.next(null);
        return result;
      }),
      shareReplay(1),
      catchError(err => {
        this.detailForm.enable();
        this.detailLoading$.next(false);
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.loadingComplete$ = combineLatest([this.detail$, this.requiredData$]).pipe(
      tap(() => {
        this.detailForm.enable();
        this.detailLoading$.next(false);
      })
    )

    this.filteredRequiredData$ = combineLatest([this.requiredData$, this.detail$]).pipe(
      map(([, detail]) => {
        const data = { ...this.localRequiredData };
        data.counterparties = data.counterparties.filter(item => item.isActive || (detail.contactCounterparties && detail.contactCounterparties.some(counterparty => item.id === counterparty.id)));
        return data;
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      })
    )

    this.counterpartiesSelected$ = this.refreshCounterpartySelected$.pipe(
      map(counterpartySelected => {
        const tempCounterpartiesSelected = [...this.localCounterpartiesSelected];
        if (counterpartySelected) {
          const hasCounterparty = tempCounterpartiesSelected.findIndex(x => x.id === counterpartySelected.id) !== -1;
          if (!hasCounterparty) {
            const counterpartyItem: util.IdDate = { id: counterpartySelected.id, date: null };
            tempCounterpartiesSelected.push(counterpartyItem);
          }
        }
        this.localCounterpartiesSelected = tempCounterpartiesSelected;
        return this.localCounterpartiesSelected;
      })
    )

    this.saveResult$ = this.save$.pipe(
      switchMap(saveType => {
        this.detailLoading$.next(true);
        this.detailForm.disable();
        const itemToSave = this.detailForm.value as Detail;
        itemToSave.contactCounterparties = this.localCounterpartiesSelected;
        return this.service.saveDetail(itemToSave, saveType);
      }),
      tap((saveResult) => {
        this.closeDetail(false);
        this.notify.success('save successful');
        this.refreshItems$.next(null);
        this.mySelection = [saveResult];
      }),
      shareReplay(1),
      catchError(err => {
        this.detailForm.enable();
        this.detailLoading$.next(false);
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.deleteResult$ = this.delete$.pipe(
      switchMap(() => {
        this.detailLoading$.next(true);
        this.detailForm.disable();
        const itemToDelete = this.detailForm.value as Detail;
        return this.service.deleteDetail(itemToDelete.id);
      }),
      tap(() => {
        this.notify.success('delete successful');
        this.closeDetail(false);
        this.refreshItems$.next(null);
      }),
      shareReplay(1),
      catchError(err => {
        this.detailForm.enable();
        this.detailLoading$.next(false);
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

  }

  getDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<Detail> = fb.group({
      id: fb.ctr(0, Validators.required),
      displayName: fb.ctr(null, Validators.required),
      fullName: fb.ctr(null),
      title: fb.ctr(null),
      addressLine1: fb.ctr(null),
      addressLine2: fb.ctr(null),
      city: fb.ctr(null),
      stateId: fb.ctr(null),
      zip: fb.ctr(null),
      countryId: fb.ctr(null),
      officePhone: fb.ctr(null),
      cellPhone: fb.ctr(null),
      altPhone: fb.ctr(null),
      fax: fb.ctr(null),
      email: fb.ctr(null, Validators.email),
      notes: fb.ctr(null),
      contactCounterparties: fb.ctr([]),
      contactTypeIds: fb.ctr([], Validators.required)
    });
    return fg;
  }

  title$ = of('Contact').pipe(
    tap((title) => util.trySetTitle(this.titleService, title))
  )

  getCounterpartyName(counterpartyId: number) {
    return this.localCounterparties[counterpartyId].name;
  }

  openDetail(id: number): void {
    this.detailOpened$.next(true);
    this.refreshDetail$.next(id);
  }

  onDetailClosing() {
    util.onDetailChanging(this.detailForm, this.messageService, this.closeDetail, this.save);
  }

  closeDetail = (isFromInterface: boolean) => {
    this.detailOpened$.next(false);
    if (isFromInterface)
      console.log('asdf');
  }

  add(): void {
    this.openDetail(0);
  }

  edit(dataItem: Item): void {
    this.openDetail(dataItem.Id);
  }

  save = (saveType: util.SaveType) => {
    this.detailForm.markAllAsTouched();
    if (this.detailForm.valid)
      this.save$.next(saveType);
    else
      this.notify.error("validation failed");
  }

  delete(): void {
    const ps: PromptSettings = {
      title: "Please Confirm",
      content: "Are you sure you want to delete this item?",
      type: 'Yes-No'
    }

    this.messageService.prompt(ps).then((result) => {
      if (result === promptAction.Yes) {
        this.delete$.next(null);
      }
    });
  }

  export(): void {
    this.exportClicked$.next(null);
  }

  dataStateChange(state: State): void {
    util.fixKendoQueryFilter(state.filter);
    this.state = state;
    this.refreshItems$.next(null);
  }

  onCounterpartySelected(dataItem: util.IdNameActive) {
    this.refreshCounterpartySelected$.next(dataItem);
    this.detailForm.markAsDirty();
  }

  removeCounterparty(counterpartyId: number) {
    this.localCounterpartiesSelected = this.localCounterpartiesSelected.filter(x => x.id !== counterpartyId);
    this.refreshCounterpartySelected$.next(null);
    this.detailForm.markAsDirty();
  }

  inactiveDateChange(counterpartyId: number, inactiveDate?: Date) {
    const item = this.localCounterpartiesSelected.find(x => x.id === counterpartyId);
    item.date = inactiveDate;
    this.detailForm.markAsDirty();
  }

  orderChanged() {
    this.detailForm.markAsDirty();
  }
}
