import { Component, ChangeDetectionStrategy, ChangeDetectorRef, ViewChild, ElementRef, ViewEncapsulation, HostListener, inject } from '@angular/core';
import { Title } from '@angular/platform-browser';
import { tap, take, map, catchError, switchMap, filter, shareReplay, retry } from 'rxjs/operators';
import { of, BehaviorSubject, Subject, combineLatest } from 'rxjs';
import { State } from '@progress/kendo-data-query';
import { MessageService } from '../_shared/services/message.service';
import { Validators } from '@angular/forms';
import { CustomFormBuilder } from '../_shared/services/custom-form-builder.service';
import { DialogService, DialogSettings, KENDO_DIALOG } from '@progress/kendo-angular-dialog';
import { NotifyService } from '../_shared/services/notify.service';
import { saveAs } from '@progress/kendo-file-saver';
import * as util from '../_shared/utils/util';
import { BrokerService, Item, Detail, SaveType, RequiredData } from './broker.service';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../app.config';
import { KENDO_LISTVIEW } from '@progress/kendo-angular-listview';

@Component({
  selector: 'app-broker',
  templateUrl: './broker.component.html',
  encapsulation: ViewEncapsulation.None,
  changeDetection: ChangeDetectionStrategy.OnPush,
  imports: [FAST_PAGE_COMMON, FAST_KENDO_COMMON, KENDO_DIALOG, KENDO_LISTVIEW]
})
export class BrokerComponent {
  private service = inject(BrokerService);
  private messageService = inject(MessageService);
  private titleService = inject(Title);
  private fb = inject(CustomFormBuilder);
  private ref = inject(ChangeDetectorRef);
  private dialogService = inject(DialogService);
  private notify = inject(NotifyService);

  @ViewChild("grid", { read: ElementRef }) kendoGridEl: ElementRef;
  @HostListener('window:resize') onResize() {
    //this function is empty but for some reason it helps the window to resize faster
  };

  constructor() {

  }

  util = util;

  detailForm = this.getDetailForm();
  detailBrokerAccountForm = this.getDetailBrokerAccountForm()
  detailBrokerAccountInitialValues: util.IdName = this.detailBrokerAccountForm.value as util.IdName;
  gridScrollPosition: util.GridScrollPosition = { topPos: 0, leftPos: 0 };
  localRequiredData: RequiredData;
  hasModifyPermission = false;
  detailInitialValues: Detail = this.detailForm.value as Detail;
  mySelection: number[] = [];
  selectedBrokerAccount: util.IdName = null;

  gridLoading$ = new BehaviorSubject<boolean>(true)
  refreshItems$ = new BehaviorSubject<string>(null)
  exporting$ = new BehaviorSubject<boolean>(false)
  detailOpened$ = new BehaviorSubject<boolean>(false)
  detailBrokerAccountOpened$ = new BehaviorSubject<boolean>(false);
  refreshRequiredData$ = new BehaviorSubject(null)
  detailLoading$ = new BehaviorSubject<boolean>(true)
  refreshDetail$ = new BehaviorSubject<number>(null)
  save$ = new Subject<SaveType>()
  delete$ = new Subject()
  exportClicked$ = new Subject()
  addBrokerAccount$ = new Subject()
  renameBrokerAccount$ = new Subject()
  deleteBrokerAccount$ = new Subject()

  state: State = {
    filter: null,
    group: null,
    skip: 0,
    sort: [{ field: 'BrokerName', dir: 'asc' }],
    take: 50
  };

  getDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<Detail> = fb.group({
      brokerId: fb.ctr(0, Validators.required),
      brokerName: fb.ctr(null, Validators.required),
      brokerType: fb.ctr(null, Validators.required),
      brokerAccounts: fb.ctr([]),
      commissionPerContract: fb.ctr(null),
      commissionAfterHours: fb.ctr(null),
      commissionForEFS: fb.ctr(null),
      memo: fb.ctr(null)
    });

    return fg;
  }

  getDetailBrokerAccountForm() {
    const fb = this.fb;
    const fg: util.FormModel<util.IdName> = fb.group({
      id: fb.ctr(0, Validators.required),
      name: fb.ctr(null, Validators.required)
    });

    return fg;
  }

  requiredData$ = this.refreshRequiredData$.pipe(
    tap(() => this.detailLoading$.next(true)),
    switchMap(refreshType => {
      return combineLatest([this.service.requiredData$, of(refreshType)]);
    }),
    map(([requiredData, refreshType]) => {
      this.localRequiredData = requiredData;
      if (refreshType === util.RefreshType.SelfOnly)
        this.detailLoading$.next(false);
      return requiredData;
    }),
    tap((requiredData) => {
      this.hasModifyPermission = requiredData.hasModifyPermission;
      util.focusInputTarget()
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  items$ = this.refreshItems$.pipe(
    tap(() => {
      this.gridLoading$.next(true);
    }),
    switchMap(() => {
      return this.service.getItems(this.state);
    }),
    tap(() => {
      this.gridLoading$.next(false);
      util.goToSavedGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  exportAction$ = this.exportClicked$.pipe(
    tap(() => {
      this.exporting$.next(true);
    }),
    switchMap(() => {
      return this.service.exportItems(this.state, 'Brokers.xlsx');
    }),
    tap(res => {
      saveAs(res.fileBlob, res.fileName);
      this.exporting$.next(false);
    }),
    shareReplay(1),
    catchError(err => {
      this.exporting$.next(false);
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  //this updates localDetailLoading whenever detailLoading$ changes so that we can check the loading status in this class
  localDetailLoading = true;
  detailLoadingWatcher$ = this.detailLoading$.pipe(
    tap(isLoading => {
      this.localDetailLoading = isLoading;
    }),
    shareReplay(1)
  )

  detail$ = this.refreshDetail$.pipe(
    filter(id => id !== null),
    switchMap(id => {
      this.detailLoading$.next(true);
      this.detailForm.disable();
      this.detailForm.reset();
      this.detailBrokerAccountForm.reset();
      this.selectedBrokerAccount = null;
      if (id === 0)
        return of(this.detailInitialValues);
      else
        return this.service.getDetail(id);
    }),
    map(result => {
      const detail: Detail = result;
      if (detail) {
        util.convertToDates(detail);
        this.detailForm.setValue(detail);
      }
      return detail;
    }),
    tap(() => {
      this.detailFinishedLoading();
    }),
    shareReplay(1),
    catchError(err => {
      this.closeDetail(false);
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  saveResult$ = this.save$.pipe(
    switchMap(saveType => {
      this.detailLoading$.next(true);
      this.detailForm.disable();
      const itemToSave: Detail = this.detailForm.value as Detail;
      return this.service.saveDetail(itemToSave, saveType);
    }),
    tap(saveResult => {
      this.notify.success('save successful');
      this.closeDetail(false);
      this.refreshItems$.next(null);
      this.mySelection = [saveResult];
    }),
    shareReplay(1),
    catchError(err => {
      this.detailFinishedLoading();
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  deleteResult$ = this.delete$.pipe(
    switchMap(() => {
      this.detailLoading$.next(true);
      this.detailForm.disable();
      const itemToDelete: Detail = this.detailForm.getRawValue();
      return this.service.deleteDetail(itemToDelete.brokerId);
    }),
    tap(() => {
      this.notify.success('delete successful');
      this.detailFinishedLoading();
      this.closeDetail(false);
      this.refreshItems$.next(null);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  addBrokerAccountResult$ = this.addBrokerAccount$.pipe(
    switchMap(() => {
      const brokerId = (this.detailForm.value as Detail).brokerId;
      const newBrokerAccountName = (this.detailBrokerAccountForm.value as util.IdName).name;
      return this.service.addNewBrokerAccount(brokerId, newBrokerAccountName);
    }),
    tap(() => {
      this.detailBrokerAccountOpened$.next(false);
      const brokerId = (this.detailForm.value as Detail).brokerId;
      this.refreshDetail$.next(brokerId);
      this.notify.success('add successful');
      this.refreshItems$.next(null);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  renameBrokerAccountResult$ = this.renameBrokerAccount$.pipe(
    switchMap(() => {
      const brokerId = (this.detailBrokerAccountForm.value as util.IdName).id;
      const newBrokerAccountName = (this.detailBrokerAccountForm.value as util.IdName).name;
      return this.service.renameBrokerAccount(brokerId, newBrokerAccountName);
    }),
    tap(() => {
      this.detailBrokerAccountOpened$.next(false);
      const brokerId = (this.detailForm.value as Detail).brokerId;
      this.refreshDetail$.next(brokerId);
      this.notify.success('add successful');
      this.refreshItems$.next(null);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  deleteBrokerAccountResult$ = this.deleteBrokerAccount$.pipe(
    switchMap(() => {
      const brokerId = this.selectedBrokerAccount.id;
      return this.service.deleteBrokerAccount(brokerId);
    }),
    tap(() => {
      const brokerId = (this.detailForm.value as Detail).brokerId;
      this.refreshDetail$.next(brokerId);
      this.notify.success('delete successful');
      this.refreshItems$.next(null);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  openDetail(id: number): void {
    util.saveGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
    this.refreshDetail$.next(id);
    this.detailOpened$.next(true);
  }

  onDetailClosing() {
    util.onDetailChanging(this.detailForm, this.dialogService, this.closeDetail, this.save);
  }

  onDetailBrokerAccountClosing() {
    util.onDetailChanging(this.detailBrokerAccountForm, this.dialogService, this.closeDetailBrokerAccount, this.saveBrokerAccount);
  }

  closeDetail = (isFromInterface: boolean) => {
    this.detailOpened$.next(false);

    if (isFromInterface)
      util.goToSavedGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
  }

  closeDetailBrokerAccount = () => {
    this.detailBrokerAccountOpened$.next(false);
  }

  detailFinishedLoading(): void {
    this.detailForm.enable();
    this.detailLoading$.next(false);
    util.focusInputTarget();
  }

  add(): void {
    this.openDetail(0);
  }

  edit(dataItem: Item): void {
    this.openDetail(dataItem.BrokerId);
  }

  save = (saveType: util.SaveType) => {
    this.detailForm.markAllAsTouched();
    if (this.detailForm.valid)
      this.save$.next(saveType);
    else
      this.notify.error("validation failed");
  }

  delete(): void {
    const deleteConfirmSettings: DialogSettings = {
      title: "Please confirm",
      content: "Are you sure you want to delete this item?",
      actions: [{ text: 'No' }, { text: 'Yes', cssClass: 'k-primary' }],
      cssClass: 'utilPrompt'
    }

    this.dialogService.open(deleteConfirmSettings).result.pipe(take(1)).subscribe(result => {
      if (util.getDialogAction(result) === util.dialogAction.Yes)
        this.delete$.next(null);
    });
  }

  export(): void {
    this.exportClicked$.next(null);
  }

  dataStateChange(state: State): void {
    this.gridScrollPosition.topPos = 0;
    this.gridScrollPosition.leftPos = 0;
    util.fixKendoQueryFilter(state.filter);
    this.state = state;
    this.refreshItems$.next(null);
  }

  refreshDropdowns() {
    this.refreshRequiredData$.next(util.RefreshType.SelfOnly);
  }

  openAddBrokerAccount(): void {
    const brokerId = (this.detailForm.value as Detail).brokerId;
    if (brokerId === 0)
      this.notify.info('The main broker account must be saved first');
    else if (this.hasModifyPermission) {
      this.detailBrokerAccountForm.enable();
      this.selectedBrokerAccount = null;
      const newBrokerAccount: util.IdName = { id: 0, name: null };
      this.detailBrokerAccountForm.setValue(newBrokerAccount);
      this.detailBrokerAccountForm.markAsUntouched();
      this.detailBrokerAccountOpened$.next(true);
      util.focusInputTarget()
    }
  }

  openRenameBrokerAccount(): void {
    if (this.hasModifyPermission && this.selectedBrokerAccount) {
      this.detailBrokerAccountForm.enable();
      this.detailBrokerAccountForm.setValue(this.selectedBrokerAccount);
      this.detailBrokerAccountOpened$.next(true);
      util.focusAndSelectInputTarget();
    }
  }

  openDeleteBrokerAccount(): void {
    if (this.hasModifyPermission) {
      const deleteConfirmSettings: DialogSettings = {
        title: "Please confirm",
        content: "Are you sure you want to delete this broker account?",
        actions: [{ text: 'No' }, { text: 'Yes', cssClass: 'k-primary' }],
        cssClass: 'utilPrompt'
      }

      this.dialogService.open(deleteConfirmSettings).result.pipe(take(1)).subscribe(result => {
        if (util.getDialogAction(result) === util.dialogAction.Yes) {
          this.detailForm.disable();
          this.detailLoading$.next(true);
          this.deleteBrokerAccount$.next(null);
        }
      });
    }
  }

  saveBrokerAccount = () => {
    if (this.detailForm.valid) {
      this.detailBrokerAccountForm.disable();
      if (this.selectedBrokerAccount)
        this.renameBrokerAccount$.next(null);
      else
        this.addBrokerAccount$.next(null);
    } else {
      this.notify.error("validation failed");
    }
  }

  title$ = of('Broker').pipe(
    tap((title) => util.trySetTitle(this.titleService, title))
  )

  filterBrokerTypes$ = new BehaviorSubject<string>(null)

  brokerTypes$ = this.filterBrokerTypes$.pipe(util.filterSpecials(this.detailLoading$, this.requiredData$, 'brokerTypes', 'brokerType'));
}
