import { ChangeDetectionStrategy, Component, computed, effect, input, signal } from '@angular/core';
import { ControlValueAccessor, NG_VALUE_ACCESSOR } from '@angular/forms';

@Component({
  selector: 'fast-buttongroup',
  imports: [],
  template: `
    <ng-content></ng-content>
  `,
  styles: `
  `,
  changeDetection: ChangeDetectionStrategy.OnPush,
  host: {
    role: 'group',
    class: 'fast-buttongroup',
    '[class]': 'conditionalClasses()',
    '[class.fast-buttongroup-disabled]': 'isDisabled()',
    '[attr.aria-disabled]': 'isDisabled()',
  },
  providers: [
    {
      provide: NG_VALUE_ACCESSOR,
      useExisting: FastButtonGroupComponent,
      multi: true,
    },
  ],
})

export class FastButtonGroupComponent implements ControlValueAccessor {
  private readonly valueSignal = signal<unknown | string | null>(null);

  private onChange?: (value: unknown | null) => void;
  private onTouched?: () => void;

  disabled = input<boolean>(false);

  private readonly disabledSignal = signal(this.disabled());

  constructor() {
    effect(() => {
      this.disabledSignal.set(this.disabled());
    });
  }

  selectItem(value: unknown) {
    if (this.isDisabled()) {
      return;
    }
    this.valueSignal.set(value ?? null);
    this.onChange?.(this.valueSignal());
    this.onTouched?.();
  }

  isSelected(value: unknown) {
    return this.valueSignal() === value;
  }

  isDisabled() {
    return this.disabledSignal();
  }

  writeValue(value: unknown): void {
    this.valueSignal.set(value ?? null);
  }

  registerOnChange(fn: (value: unknown | null) => void): void {
    this.onChange = fn;
  }

  registerOnTouched(fn: () => void): void {
    this.onTouched = fn;
  }

  setDisabledState(isDisabled: boolean): void {
    this.disabledSignal.set(isDisabled);
  }

  conditionalClasses = computed(() => {
    const classes = [] as string[];

    classes.push(...this.getCommonClasses());
    classes.push(...this.getLightBaseClasses());
    classes.push(...this.getDarkBaseClasses());

    const conditionalClasses = this.getConditionalClassesFromArrays(classes);
    return conditionalClasses;
  });

  getConditionalClassesFromArrays(classArray: string[]): { [key: string]: boolean } {
    const classes: { [key: string]: boolean } = {};
    classArray.forEach(className => {
      classes[className] = true;
    });
    return classes;
  }

  getCommonClasses() {
    return [
      "inline-flex",
      "w-fit",
      "flex-wrap",
      "gap-px",
      "rounded-lg",
      "p-1",
      "focus-within:ring-2",
      "focus-within:ring-slate-900",
      "focus-within:ring-offset-2",
      "[&.fast-buttongroup-disabled]:cursor-not-allowed",
      "[&.fast-buttongroup-disabled]:opacity-70",
      "[&>fast-button:first-child_button]:rounded-l-lg",
      "[&>fast-button:first-child_button]:border-solid",
      "[&>fast-button:first-child_button]:border-l-1",
      "[&>fast-button:first-child_button]:border-t-1",
      "[&>fast-button:first-child_button]:border-b-1",
      "[&>fast-button:first-child_button]:border-r-0",
      "[&>fast-button:last-child_button]:border-solid",
      "[&>fast-button:last-child_button]:border-l-0",
      "[&>fast-button:last-child_button]:border-t-1",
      "[&>fast-button:last-child_button]:border-b-1",
      "[&>fast-button:last-child_button]:border-r-1",
      "[&>fast-button:last-child_button]:rounded-r-lg",
      "[&>fast-button:not(:first-child):not(:last-child)_button]:border-solid",
      "[&>fast-button:not(:first-child):not(:last-child)_button]:border-t-1",
      "[&>fast-button:not(:first-child):not(:last-child)_button]:border-b-1",
    ];
  }

  getLightBaseClasses() {
    const lightBase = ["bg-base-white-250", "text-base-black-1000", "border-base-gray-1000"]

    return lightBase;
  }

  getDarkBaseClasses() {
    const darkBase = ["dark:bg-alt-blue-250", "dark:text-base-white-500", "dark:border-alt-white-1000"]

    return darkBase;
  }
}
