import { Component, ChangeDetectionStrategy, ChangeDetectorRef, ViewEncapsulation, AfterContentInit, ViewChild, inject } from '@angular/core';
import { Title } from '@angular/platform-browser';
import { catchError, filter, map, shareReplay, switchMap, tap } from 'rxjs/operators';
import { of, BehaviorSubject, Observable, Subject, combineLatest } from 'rxjs';
import { DashService, UserScreenSort } from './dash.service';
import { MessageService } from '../_shared/services/message.service';
import { DialogService } from '@progress/kendo-angular-dialog';
import { NotifyService } from '../_shared/services/notify.service';
import * as util from '../_shared/utils/util';
import { CustomFormBuilder } from '../_shared/services/custom-form-builder.service';
import { CommonService, ModuleMenu } from '../_shared/services/common.service';
import { ActivatedRoute, Router } from '@angular/router';
// If you dont import this angular will import the wrong "Location"
import { Location } from '@angular/common';
import { HttpParams } from '@angular/common/http';
import { GlobalFavoritesComponent } from '../_shared/global-favorites/global-favorites.component';
import { DragEndEvent, KENDO_SORTABLE } from '@progress/kendo-angular-sortable';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../app.config';
import { HasModulePipe } from './has-module.pipe';
import { HoverClassDirective } from '../_shared/directives/hover-class-directive';
import { toObservable } from '@angular/core/rxjs-interop';

@Component({
  selector: 'app-dash',
  templateUrl: './dash.component.html',
  encapsulation: ViewEncapsulation.None,
  changeDetection: ChangeDetectionStrategy.OnPush,
  standalone: true,
  imports: [FAST_KENDO_COMMON, FAST_PAGE_COMMON, HasModulePipe, KENDO_SORTABLE, GlobalFavoritesComponent, HoverClassDirective]
})
export class DashComponent implements AfterContentInit {
  @ViewChild('globalFavorites') globalFavorites: GlobalFavoritesComponent;

  listOpened$ = new BehaviorSubject<boolean>(false);
  moduleMenus$: Observable<ModuleMenu[]>;
  getListItems$ = new BehaviorSubject<string>(null);
  listItems$: Observable<ModuleMenu[]>;
  moduleName$ = new BehaviorSubject<string>(null);
  screenSortChanged$ = new Subject();
  saveScreenSortResult$: Observable<object>;
  getUserScreenSortResult$: Observable<UserScreenSort[]>;

  icons = util.icons;
  reportsUrl: string;
  isScreenDragging: boolean = false;
  orderedScreenItems: ModuleMenu[] = [];
  userScreenSort: UserScreenSort[] = [];

  title$ = of('Dashboard').pipe(
    tap((title) => util.trySetTitle(this.titleService, title))
  )

  service = inject(DashService);
  messageService = inject(MessageService);
  titleService = inject(Title);
  fb = inject(CustomFormBuilder);
  dialogService = inject(DialogService);
  notify = inject(NotifyService);
  commonService = inject(CommonService);
  ref = inject(ChangeDetectorRef);
  router = inject(Router);
  location = inject(Location);
  activatedRoute = inject(ActivatedRoute);

  constructor() {
    this.moduleMenus$ = toObservable(this.commonService.commonDataRes.value).pipe(
      filter(commonData => commonData != null),
      map(commonData => {
        this.reportsUrl = commonData.moduleMenus.find(x => x.name.toLowerCase() === 'reports')?.children[0]?.path;
        return commonData.moduleMenus;
      })
    );

    this.getUserScreenSortResult$ = this.service.getUserScreenSort().pipe(
      map(result => {
        this.userScreenSort = result;
        return this.userScreenSort;
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService);
      })
    );

    this.listItems$ = combineLatest([toObservable(this.commonService.commonDataRes.value), this.getListItems$, this.getUserScreenSortResult$]).pipe(
      filter(([commonData, moduleName]) => commonData != null && moduleName !== null),
      map(([commonData, moduleName]) => {
        const userScreensInDefaultOrder = commonData.moduleMenus.find(x => x.name.toLowerCase() === moduleName).children;
        const userScreenSortForModule = this.userScreenSort.filter(x => x.moduleName === moduleName);
        //order the screens based on the user's sort
        const orderedScreens = userScreensInDefaultOrder.map(x => {
          const userScreen = userScreenSortForModule.find(y => y.screenId === x.id);
          if (userScreen)
            return { ...x, orderId: userScreen.orderId };
          else
            return { ...x, orderId: 999 };
        }).sort((a, b) => a.orderId - b.orderId);
        this.orderedScreenItems = orderedScreens;
        return this.orderedScreenItems;
      })
    );

    this.saveScreenSortResult$ = this.screenSortChanged$.pipe(
      switchMap(() => {
        const ids = this.orderedScreenItems.map(x => x.id);
        return this.service.saveScreenSort(ids);
      }),
      tap(result => {
        this.userScreenSort = result;
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService);
      })
    );
  }

  ngAfterContentInit(): void {
    this.location.subscribe(loc => {
      if (loc.type !== 'popstate') return;
      const params = new HttpParams({ fromString: loc.url.split('?')[1] });
      const moduleName: string = params.get('module');
      this.switchModule(moduleName);
    });
    this.activatedRoute.queryParams.subscribe(params => {
      const moduleName: string = params['module'];
      this.switchModule(moduleName);
    });
  }

  switchModule(moduleName: string) {
    if (moduleName) {
      this.getListItems$.next(moduleName);
      this.listOpened$.next(true);
      this.moduleName$.next(moduleName);
    }
    else
      this.listOpened$.next(false);
  }

  openList(moduleName: string) {
    this.getListItems$.next(moduleName);
    const url = this.router.createUrlTree([], { relativeTo: this.activatedRoute, queryParams: { module: moduleName } }).toString()
    this.location.go(url);
    this.listOpened$.next(true);
    this.moduleName$.next(moduleName);
  }

  closeList() {
    const url = this.router.createUrlTree([], { relativeTo: this.activatedRoute, queryParams: { module: null } }).toString()
    this.location.go(url);
    this.listOpened$.next(false);
  }

  openReports() {
    this.externalNavigate(this.reportsUrl);
  }

  externalNavigate(path: string) {
    if (!this.isScreenDragging)
      this.router.navigate(['/externalRedirect', { externalUrl: path }], {
        skipLocationChange: true,
      });
  }

  addFavorite(url: string) {
    this.globalFavorites.addFavorite(url);
  }

  screenDragStart() {
    this.isScreenDragging = true;
  }

  screenDragEnd(event: DragEndEvent) {
    if (event.index === event.oldIndex)
      //if dragged to same position then immediately set dragging to false so that tiny movements still trigger a navigation
      this.isScreenDragging = false;
    else {
      //if dragged to new position then delay setting dragging to false since we don't want actual drags to navigate to a new url
      setTimeout(() => {
        this.isScreenDragging = false;
        this.screenSortChanged$.next(null);
      });
    }
  }
}
