import { ChangeDetectionStrategy, Component, HostBinding, input } from '@angular/core';

export type opacityLevel = 'light' | 'medium' | 'dark';

@Component({
  selector: 'fast-overlay',
  changeDetection: ChangeDetectionStrategy.OnPush,
  template: `
  <ng-content></ng-content>
  `,
  // closest way we can get this to apply directly to the component, @apply seems to break on things like bg-black, top-0, left-0
  host: {
    class: 'w-full h-full bg-black top-0 left-0',
  },
})
export class FastOverlayComponent {

  layerLevel = input(1); // The level of the overlay layer (1 is the lowest) (higher levels are on top of lower levels)
  opacityLevel = input<opacityLevel>('medium'); // The darkness of the overlay ('light', 'medium', 'dark')
  isContained = input(false); // Whether the overlay is contained within a parent element

  @HostBinding('style') get hostStyles() {
    let opacityPercent: number;
    switch (this.opacityLevel()) {
      case 'light': opacityPercent = 0.2; break;
      case 'medium': opacityPercent = 0.6; break;
      case 'dark': opacityPercent = 0.8; break;
    };

    let zIndex: number;
    switch (this.layerLevel()) {
      case 1: zIndex = 10001; break;
      case 2: zIndex = 11001; break;
      default: zIndex = 11001 + (2 * (this.layerLevel() - 2)); break;
      // default case results in the following pattern:
      // case 3 = 11001 + (2 * (3-2)) = 11003
      // case 4 = 11001 + (2 * (4-2)) = 11005
      // case 5 = 11001 + (2 * (5-2)) = 11007
    };

    return {
      'z-index': zIndex,
      position: this.isContained() ? 'absolute' : 'fixed',
      opacity: opacityPercent,
    };
  }
}
