import { ChangeDetectionStrategy, ChangeDetectorRef, Component, computed, inject, input, output } from '@angular/core';
import { FAST_KENDO_COMMON } from '../../app.config';
import { ControlValueAccessor, NG_VALUE_ACCESSOR } from '@angular/forms';

@Component({
  selector: 'fast-multiselect',
  changeDetection: ChangeDetectionStrategy.OnPush,
  imports: [FAST_KENDO_COMMON],
  providers: [{
    provide: NG_VALUE_ACCESSOR,
    multi: true,
    useExisting: FastMultiselectComponent
  }],
  template: `
  <kendo-multiselect
  class=" items-center
  [&_.k-chip-list_.k-chip]:p-0.5
  [&_.k-chip-list_.k-chip]:leading-normal
  [&_.k-chip-list_.k-chip]:mt-0.5
  [&_.k-chip-list_.k-chip]:mb-0
  [&_.k-chip-list_.k-chip]:bg-transparent
  [&_.k-chip-list_.k-chip-content]:leading-4
  [&_.k-chip-list_.k-chip-content]:m-0
  [&_.k-chip-list_.k-chip-content]:p-0
  [&_.k-chip-list_.k-chip.k-chip-solid-base]:bg-transparent
  "
  [disabled]="disabled()"
  [data]="data()"
  [textField]="textField()"
  [valueField]="valueField()"
  [valuePrimitive]="valuePrimitive()"
  [filterable]="filterable()"
  [value]="value"
  [placeholder]="placeholder()"
  (filterChange)="onFilterChange($event)"
  (valueChange)="onValueChange($event)"
  [class]="conditionalClasses()">
  </kendo-multiselect>
  `
})
export class FastMultiselectComponent<T extends object> implements ControlValueAccessor { // Use T to make the component totally generic
  cdr = inject(ChangeDetectorRef); // use this injection to automatically detect changes on load and display immediately

  data = input.required<T[]>();
  textField = input<string>("");
  valueField = input<string>("");
  valuePrimitive = input<boolean>(true);
  filterable = input<boolean>(true);
  disabled = input<boolean>(false);
  placeholder = input<string>("");
  filterChange = output<string>();
  valueChange = output<T[]>();

  // we're going to be using a lot of unknown type, since the data passed in can be an array of anything (idName, string, boolean, etc.)
  value: T[];
  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  onChange = (_: T[]) => { };
  onTouched = () => { };

  writeValue(obj: T[]): void {
    this.value = obj;
    this.cdr.markForCheck();
  }

  registerOnChange(fn: () => void) {
    this.onChange = fn;
  }

  registerOnTouched(fn: () => void) {
    this.onTouched = fn;
  }

  onValueChange(event: T[]): void {
    this.value = event;
    this.onChange(this.value); // Notify the Forms API about the change
    this.valueChange.emit(event);
  }

  onFilterChange(event: string): void {
    this.filterChange.emit(event); // This emits the event to the parent
  }

  conditionalClasses = computed(() => {
    const classes = [] as string[];

    classes.push(...this.getLightBaseClasses());
    classes.push(...this.getDarkBaseClasses());

    const conditionalClasses = this.getConditionalClassesFromArrays(classes);
    return conditionalClasses;
  });


  getConditionalClassesFromArrays(classArray: string[]): { [key: string]: boolean } {
    const classes: { [key: string]: boolean } = {};
    classArray.forEach(className => {
      classes[className] = true;
    });
    return classes;
  }

  getLightBaseClasses() {
    return [
      "[&_.k-chip-list_.k-chip]:text-base-black-1000",
      "[&_.k-chip-list_.k-chip.k-chip-solid-base]:border-base-gray-500",
      "[&_.k-chip-list_.k-chip]:hover:text-base-black-250",
      "[&_.k-chip-list_.k-chip.k-chip-solid-base]:hover:bg-base-gray-250",
      "[&_.k-chip-actions]:text-base-gray-1000",
      "[&_.k-chip-actions]:hover:text-base-red-1000"];
  }

  getDarkBaseClasses() {
    return [
      "dark:[&_.k-chip-list_.k-chip]:text-base-white-500",
      "dark:[&_.k-chip-list_.k-chip.k-chip-solid-base]:border-base-gray-1000",
      "dark:[&_.k-chip-list_.k-chip]:hover:text-base-white-1000",
      "dark:[&_.k-chip-list_.k-chip.k-chip-solid-base]:hover:bg-alt-blue-250",
      "dark:[&_.k-chip-actions]:text-base-gray-1000",
      "dark:[&_.k-chip-actions]:hover:text-alt-red-500"];
  }
}
