import { Component, ChangeDetectionStrategy, Input, Output, EventEmitter, OnInit, OnDestroy } from '@angular/core';
import * as util from '../utils/util';
import { WinHeightPipe } from '../pipes/win-height.pipe';
import { WinWidthPipe } from '../pipes/win-width.pipe';
import { KENDO_WINDOW } from '@progress/kendo-angular-dialog';
import { UrlParamsPipe } from '../pipes/url-params.pipe';
import { KENDO_SVGICON } from '@progress/kendo-angular-icons';
import { FastButtonComponent } from './fast-button.component';
import { FastWindowComponent } from "./fast-window.component";
import { FastHRComponent } from "./fast-hr.component";

@Component({
  selector: 'fast-frame',
  changeDetection: ChangeDetectionStrategy.OnPush,
  imports: [WinHeightPipe, WinWidthPipe, UrlParamsPipe, FastButtonComponent, KENDO_WINDOW, KENDO_SVGICON, FastWindowComponent, FastHRComponent],
  template: `
    <fast-window (close)="close()" [height]="height | winHeight"
      [width]="width | winWidth" [title]="title">
        <!-- use an iframe even when you could use a component because kendo window popups inside the componenent don't position correctly otherwise -->
        <!-- this also helps with escape key handling and error encapsulation -->
        <iframe class="flex flex-grow" [src]="src | urlParams : externalParams"></iframe>
        @if (isCloseVisible) {
        <fast-hr />
        <div class="flex justify-end">
          <fast-button class="m-2" (click)="close()">
            <kendo-svg-icon [icon]="icons.cancelIcon"></kendo-svg-icon>Close
          </fast-button>
        </div>
        }
    </fast-window>
  `,
  styles: `:host { @apply absolute; }`,
})
export class FastFrameComponent implements OnInit, OnDestroy {
  @Input() title: string;
  @Input() src: string;
  @Input() externalParams: unknown;
  @Input() width: number = 1800;
  @Input() height: number = 0;
  @Input() isCloseVisible: boolean = true;
  @Output() closed = new EventEmitter();

  // this is needed so we can add and remove the same event listener correctly
  private boundHandleMessage: (event: MessageEvent) => void;

  icons = util.icons;

  ngOnInit(): void {
    // console.log('FastFrameComponent.ngOnInit()');
    this.boundHandleMessage = this.handleMessage.bind(this);
    this.addMessageEventListener();
  }

  ngOnDestroy(): void {
    // console.log('FastFrameComponent.ngOnDestroy()');
    this.removeMessageEventListener();
  }

  handleMessage(event: MessageEvent): void {
    // console.log('FastFrameComponent.handleMessage()');
    // console.log(event);
    if (event.data === 'close') {
      // console.log('FastFrameComponent:  window close message received');
      this.close();
    }
  }

  addMessageEventListener(): void {
    // console.log('FastFrameComponent.addMessageEventListener()');
    window.addEventListener('message', this.boundHandleMessage);
  }

  removeMessageEventListener(): void {
    // console.log('FastFrameComponent.removeMessageEventListener()');
    window.removeEventListener('message', this.boundHandleMessage);
  }

  close(): void {
    // console.log('FastFrameComponent.close()');
    this.closed.emit();
  }
}
