import { ChangeDetectionStrategy, ChangeDetectorRef, Component, computed, inject, input, output } from '@angular/core';
import { FAST_KENDO_COMMON } from '../../app.config';
import { ControlValueAccessor, NG_VALUE_ACCESSOR } from '@angular/forms';

@Component({
  selector: 'fast-combobox',
  changeDetection: ChangeDetectionStrategy.OnPush,
  imports: [FAST_KENDO_COMMON],
  providers: [{
    provide: NG_VALUE_ACCESSOR,
    multi: true,
    useExisting: FastComboboxComponent
  }],
  template: `
  <kendo-combobox
  class=" h-7 items-center [&_button]:border-none [&_button]:w-6.5 [&_button]:items-center
  "
  [disabled]="disabled()"
  [data]="data()"
  [textField]="textField()"
  [valueField]="valueField()"
  [valuePrimitive]="valuePrimitive()"
  [filterable]="filterable()"
  [value]="value"
  [placeholder]="placeholder()"
  [clearButton]="clearButton()"
  [virtual]="{ itemHeight: 28 }"
  (filterChange)="onFilterChange($event)"
  (valueChange)="onValueChange($event)"
  [class]="conditionalClasses()">
  </kendo-combobox>
  `
})
export class FastComboboxComponent<T extends object> implements ControlValueAccessor {
  cdr = inject(ChangeDetectorRef); // use this injection to automatically detect changes on load and display immediately

  data = input.required<T[]>();
  textField = input<string>("");
  valueField = input<string>("");
  valuePrimitive = input<boolean>(true);
  filterable = input<boolean>(true);
  disabled = input<boolean>(false);
  clearButton = input<boolean>(false);
  placeholder = input<string>("");
  clearOnValueChange = input<boolean>(false);
  valueChange = output<T[]>();
  filterChange = output<string>();

  // we're going to be using a lot of unknown type, since the data passed in can be an array of anything (idName, string, boolean, etc.)
  value: T[];
  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  onChange = (_: T[]) => { };
  onTouched = () => { };

  writeValue(obj: T[]): void {
    this.value = obj;
    this.cdr.markForCheck();
  }

  registerOnChange(fn: () => void) {
    this.onChange = fn;
  }

  registerOnTouched(fn: () => void) {
    this.onTouched = fn;
  }

  onValueChange(event: T[]): void {
    this.value = event;
    this.onChange(this.value); // Notify the Forms API about the change
    this.valueChange.emit(event);

    // when emitting an event, we can make it so that when the value changes it runs the method first, then clears
    // recursively hits itself to prevent null looping and/or error impls.
    if (this.clearOnValueChange()) {
      this.value = null;
      this.onChange(null);
    }
    else {
      this.value = event;
      this.onChange(this.value);
      this.valueChange.emit(event);
    }
  }

  onFilterChange(event: string): void {
    this.filterChange.emit(event);
  }

  conditionalClasses = computed(() => {
    const classes = [] as string[];

    classes.push(...this.getLightBaseClasses());
    classes.push(...this.getDarkBaseClasses());
    classes.push(...this.getLightActiveClasses());
    classes.push(...this.getDarkActiveClasses());

    const conditionalClasses = this.getConditionalClassesFromArrays(classes);
    return conditionalClasses;
  });


  getConditionalClassesFromArrays(classArray: string[]): { [key: string]: boolean } {
    const classes: { [key: string]: boolean } = {};
    classArray.forEach(className => {
      classes[className] = true;
    });
    return classes;
  }

  getLightBaseClasses() {
    return [
      "[&_button]:text-black",
      "[&_button]:bg-base-white-250",
      "[&_button]:hover:bg-base-white-750"];
  }

  getDarkBaseClasses() {
    return [
      "dark:[&_button]:text-white",
      "dark:[&_button]:bg-alt-gray-500",
      "dark:[&_button]:hover:bg-alt-blue-750"];
  }

  getLightActiveClasses() {
    return [
      "[&_button]:active:bg-base-white-1000"];
  }

  getDarkActiveClasses() {
    return [
      "dark:[&_button]:active:bg-alt-blue-250"];
  }
}
