using System.Text.RegularExpressions;
using Aspose.Words;
using Aspose.Words.Drawing;
using Aspose.Words.Tables;
using Microsoft.Extensions.FileProviders;
using static Fast.Models.Enums;

namespace Fast.Logic;

public abstract partial class DocBase
{
    public Dictionary<string, List<Run>> docRunsByKey = new(StringComparer.OrdinalIgnoreCase);
    public Document doc;
    protected readonly MyDbContext db;
    protected readonly string filesFolderPath;
    protected readonly string templatesFolderPath;
    protected readonly string signaturesFolderPath;
    protected readonly string logosFolderPath;

    public DocBase(string filesFolderPath, string templatesFolderPath, string signaturesFolderPath, string logosFolderPath, bool ignoreMissingKeys)
    {
        db = Main.CreateContext();
        this.filesFolderPath = filesFolderPath;
        this.templatesFolderPath = templatesFolderPath;
        this.signaturesFolderPath = signaturesFolderPath;
        this.logosFolderPath = logosFolderPath;

        using var provider = new PhysicalFileProvider(templatesFolderPath);
        IFileInfo fileInfo = provider.GetFileInfo(TemplateFileName);
        var stream = fileInfo.CreateReadStream();
        doc = new Document(stream);

        var docKeys = GetDocKeys();
        FillDocKeyRuns(docKeys, ignoreMissingKeys);
    }

    protected abstract HashSet<string> GetDocKeys();

    protected abstract string TemplateFileName { get; }

    public void SetText(string key, string text, bool removeRowIfEmptyText = false)
    {
        if (!docRunsByKey.TryGetValue(key, out List<Run>? docRuns))
            return;

        var tagToReplace = $"«{key}»";
        foreach (var run in docRuns)
        {
            run.Text = run.Text.Replace(tagToReplace, text ?? "", StringComparison.OrdinalIgnoreCase);
            if (removeRowIfEmptyText && string.IsNullOrWhiteSpace(text))
                RemoveParentRowOfRun(run);
        }
    }

    protected static void SetTableCellText(
        Cell tableCell,
        string text,
        bool bold = false,
        ParagraphAlignment? alignment = null)
    {
        Run run = (Run)tableCell.GetChild(NodeType.Run, 0, true);
        if (run == null)
        {
            Paragraph paragraph = (Paragraph)tableCell.GetChild(NodeType.Paragraph, 0, true);
            if (paragraph == null)
            {
                paragraph = new Paragraph(tableCell.Document);
                tableCell.AppendChild(paragraph);
            }
            run = new Run(tableCell.Document);
            paragraph.AppendChild(run);
        }

        if (alignment != null)
            run.ParentParagraph.ParagraphFormat.Alignment = alignment.Value;
        text ??= "";
        run.Text = text;
        run.Font.Bold = bold;
    }

    private void RemoveParentRowOfRun(Run run)
    {
        var parentRow = GetParentRowOfRun(run);
        parentRow?.Remove();
    }

    private Row? GetParentRowOfRun(Run run)
    {
        Row? parentRow = null;
        var parentNode = run.ParentNode;
        for (int i = 0; i < 8; i++)
        {
            if (parentNode is Row row)
            {
                parentRow = row;
                break;
            }
            else
                parentNode = parentNode.ParentNode;
        }
        return parentRow;
    }

    protected void SetImage(string? filePath, int imageIndexNum)
    {
        filePath = GetFilePathOnDisk(filePath);
        if (filePath == null)
            return;

        NodeCollection shapes = doc.GetChildNodes(NodeType.Shape, true);
        var isOutOfRange = shapes.Count < imageIndexNum + 1;
        if (isOutOfRange)
            return;

        var shape = shapes[imageIndexNum] as Shape;
        var isImage = shape?.HasImage == true;
        if (!isImage)
            return;

        shape?.ImageData.SetImage(filePath);
    }

    protected void SetImage(string? filePath, string imageNameOrTitle)
    {
        filePath = GetFilePathOnDisk(filePath);
        if (filePath == null)
            return;

        NodeCollection shapes = doc.GetChildNodes(NodeType.Shape, true);
        foreach (var item in shapes)
        {
            var shape = item as Shape;
            var isImage = shape?.HasImage == true;
            var isSameNameOrTitle = (shape?.Name) == imageNameOrTitle || (shape?.Title) == imageNameOrTitle;
            if (!isImage || !isSameNameOrTitle)
                continue;

            shape?.ImageData.SetImage(filePath);
        }
    }

    /// <summary>
    /// Gets the file path on disk by searching in a case insensitve manner
    /// On Linux the file name and extension are case sensitive and may not match the case of the filePath parameter passed in
    /// This method will return the file path with the case that it actually is on disk, regardless of the case of the filePath parameter passed in
    /// </summary>
    protected string? GetFilePathOnDisk(string? filePath)
    {
        if (filePath == null)
            return null;

        var parentDirectory = Directory.GetParent(filePath)?.FullName;
        if (parentDirectory == null)
            return null;

        var files = Directory.GetFiles(parentDirectory, "*", new EnumerationOptions { MatchCasing = MatchCasing.CaseInsensitive });
        if (files.Length == 0)
            return null;

        var filePathOnDisk = files.Where(f => f.Contains(filePath, StringComparison.OrdinalIgnoreCase)).FirstOrDefault();
        var isSignaturePath = filePath.Contains(signaturesFolderPath, StringComparison.OrdinalIgnoreCase);
        if (filePathOnDisk == null && isSignaturePath)
        {
            var sampleSigPath = Path.Join(signaturesFolderPath, "Sample.jpg");
            if (File.Exists(sampleSigPath))
                filePathOnDisk = sampleSigPath;
        }

        return filePathOnDisk;
    }

    protected static string FormatNum(decimal? num, NumFormat format)
    {
        if (!num.HasValue)
            return "";

        //manually specifying the formats
        //otherwise NumberFormatInfo may vary with OS and OS settings which result in different formats
        string result = format switch
        {
            NumFormat.n0 => num.Value.ToString("#,##0;(#,##0);0"),
            NumFormat.n1 => num.Value.ToString("#,##0.0;(#,##0.0);0.0"),
            NumFormat.n2 => num.Value.ToString("#,##0.00;(#,##0.00);0.00"),
            NumFormat.n3 => num.Value.ToString("#,##0.000;(#,##0.000);0.000"),
            NumFormat.n4 => num.Value.ToString("#,##0.0000;(#,##0.0000);0.0000"),
            NumFormat.n5 => num.Value.ToString("#,##0.00000;(#,##0.00000);0.00000"),
            NumFormat.c2 => num.Value.ToString("$ #,##0.00;$ (#,##0.00);$ 0.00"),
            NumFormat.c3 => num.Value.ToString("$ #,##0.000;$ (#,##0.000);$ 0.000"),
            NumFormat.c4 => num.Value.ToString("$ #,##0.0000;$ (#,##0.0000);$ 0.0000"),
            NumFormat.c5 => num.Value.ToString("$ #,##0.00000;$ (#,##0.00000);$ 0.00000"),
            NumFormat.p2 => num.Value.ToString("#,##0.00%;(#,##0.00%);0.00%"),
            NumFormat.p3 => num.Value.ToString("#,##0.000%;(#,##0.000%);0.000%"),
            NumFormat.p4 => num.Value.ToString("#,##0.0000%;(#,##0.0000%);0.0000%"),
            NumFormat.p5 => num.Value.ToString("#,##0.00000%;(#,##0.00000%);0.00000%"),
            NumFormat.p6 => num.Value.ToString("#,##0.000000%;(#,##0.000000%);0.000000%"),
            _ => throw new Exception($"Undefined format code: {format}")
        };

        return result;
    }

    [GeneratedRegex("«.*?»")]
    private static partial Regex TagRegex();
    protected void FillDocKeyRuns(HashSet<string> keys, bool ignoreMissingKeys)
    {
        docRunsByKey = new Dictionary<string, List<Run>>(StringComparer.OrdinalIgnoreCase);
        List<Run> runs = doc.GetChildNodes(NodeType.Run, true).Where(x => x.GetText().Length >= 1).Select(x => (Run)x).ToList();
        CombineRunTags(runs);

        foreach (Run run in runs)
        {
            var runKeyMatches = TagRegex().Matches(run.Text);
            foreach (Match match in runKeyMatches)
            {
                var runKey = match.Value.Replace("«", "").Replace("»", "");
                if (keys.Contains(runKey))
                {
                    if (!docRunsByKey.ContainsKey(runKey))
                        docRunsByKey.Add(runKey, new List<Run>());
                    docRunsByKey[runKey].Add(run);
                }
            }
        }

        if (!ignoreMissingKeys)
            foreach (string key in keys)
            {
                if (!docRunsByKey.ContainsKey(key))
                    throw new Exception($"Could not find key {{{key}}} in template file.");
            }
    }

    //sometimes tags are split between runs
    //this combines runs that were split into a single run
    //this makes it easier to replace a tag with new text
    //the old runs are not removed but they are emptied
    protected static void CombineRunTags(List<Run> runs)
    {
        Run? replacementRun = null;
        foreach (Run run in runs)
        {
            //if begin tag but no end tag after it
            var lastIndexOfBeginTag = run.Text.LastIndexOf('«');
            var lastIndexOfEndTag = run.Text.LastIndexOf('»');
            if (lastIndexOfBeginTag > lastIndexOfEndTag)
            {
                replacementRun = run; //set this run as the replacement run
            }
            //if we are mid-tag
            else if (replacementRun != null && !run.Text.Contains('»'))
            {
                replacementRun.Text += run.Text; //combine the text of this run with the previous run
                run.Text = ""; //empty the text of this run
            }
            //if end tag without begin tag
            else if (replacementRun != null && run.Text.Contains('»'))
            {
                replacementRun.Text += run.Text; //combine the text of this run with the previous run
                run.Text = ""; //empty the text of this run
                replacementRun = null; //reset the replacement run since we found an end tag
            }
        }
    }

    /// <summary>
    /// Writes an exception text file to disk.
    /// </summary>
    /// <returns>Returns the file name of the text file that was created.  This does not include the full path.</returns>
    protected string WriteExceptionToFile(Exception exception, string dealNum)
    {
        var fileName = Util.String.GetNewGuid() + ".txt";
        var fullPath = Path.Join(filesFolderPath, fileName);

        var errorText = GetErrorText(exception, dealNum);

        File.WriteAllText(fullPath, errorText);
        return fileName;
    }

    private static string GetErrorText(Exception exception, string dealNum)
    {
        var messagePrefix = $"Deal #: {dealNum}" + Environment.NewLine;
        var errorMsg = Fast.Logic.Util.String.GetExceptionMessage(exception);
        return messagePrefix + errorMsg;
    }
}
