using System.Runtime.Serialization.Json;
using Microsoft.AspNetCore.OData.Formatter;
using Microsoft.Extensions.FileProviders;

namespace Fast.Web.Controllers;

[Authorize]
[ApiController]
[Route("api/[controller]")]
public class W9FormController : ODataController
{
    private readonly MyDbContext db;
    private readonly AuthorizationHelper authHelper;
    private readonly string w9FormDocFolderPath;
    private readonly IWebHostEnvironment env;

    public W9FormController(MyDbContext context, IWebHostEnvironment env)
    {
        db = context;
        authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
        this.env = env;

        w9FormDocFolderPath = Path.Join(env?.ContentRootPath, "W9Forms");
        if (!Directory.Exists(w9FormDocFolderPath))
            Directory.CreateDirectory(w9FormDocFolderPath);
    }

    [Permission("W-9 Forms", PermissionType.View)]
    [Route("/odata/GetW9FormItems")]
    public async Task<IActionResult> GetItems(ODataQueryOptions<W9FormListItem> queryOptions, bool isExport)
    {
        queryOptions = Util.GetQueryOptionsWithConvertedDates(queryOptions);
        var itemsQueryable = GetItemsInternal();
        itemsQueryable = queryOptions.ApplyTo(itemsQueryable) as IQueryable<W9FormListItem>;
        var items = itemsQueryable == null ? null : await itemsQueryable.ToListAsync();

        if (isExport)
            return File(Util.Excel.GetExportFileStream(items), "application/octet-stream");
        else
            return Ok(items);
    }

    private IQueryable<W9FormListItem> GetItemsInternal()
    {
        IQueryable<W9FormListItem>? itemsQueryable = null;

        itemsQueryable = (
            from c in db.W9Forms
            let businessTypeName = c.CounterpartyParent.BusinessType != null ? c.CounterpartyParent.BusinessType.Name : ""
            let taxClassificationPrefix = businessTypeName == "LLC" ? "LLC - " : c.CounterpartyParent.BusinessSubTypeId == null ? businessTypeName : ""
            let taxClassificationSuffix = c.CounterpartyParent.BusinessSubTypeId != null ? businessTypeName : ""
            select new W9FormListItem
            {
                Id = c.Id,
                ParentCounterparty = c.CounterpartyParent.Name,
                Counterparty = c.Counterparty != null ? c.Counterparty.Name : "",
                TaxClassification = taxClassificationPrefix + taxClassificationSuffix,
                TaxpayerIdNum1 = c.CounterpartyParent.FederalTaxId ?? "",
                TaxpayerIdNum2 = (c.Counterparty != null && c.Counterparty.FederalTaxId != null) ? c.Counterparty.FederalTaxId! : "",  // for some reason, the compiler thinks this is a warning, using the explicit call here
                VersionMonth = c.VersionMonth
            }
        );

        return itemsQueryable;
    }

    [Permission("W-9 Forms", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetRequiredData()
    {
        var hasModifyPermission = await authHelper.IsAuthorizedAsync(User, "W-9 Forms", PermissionType.Modify);
        var counterparties = await DataHelper.GetCounterpartiesAsync(false, Enums.ProductCategory.All);
        var result = new { hasModifyPermission, counterparties };
        return Ok(result);
    }

    [Permission("W-9 Forms", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetDetail(int id)
    {

        var detail = await (
            from q in db.W9Forms
            let businessTypeName = q.CounterpartyParent.BusinessType != null ? q.CounterpartyParent.BusinessType.Name : ""
            let taxClassificationPrefix = businessTypeName == "LLC" ? "LLC - " : q.CounterpartyParent.BusinessSubTypeId == null ? businessTypeName : ""
            let taxClassificationSuffix = q.CounterpartyParent.BusinessSubTypeId != null ? businessTypeName : ""
            let addressLine1 = q.CounterpartyParent.AddressLine1
            let addressLine2 = q.CounterpartyParent.AddressLine2
            where q.Id == id
            select new W9FormDetail
            {
                Id = q.Id,
                ParentCounterpartyId = q.CounterpartyParentId,
                ChildCounterpartyId = q.CounterpartyId,
                VersionMonth = q.VersionMonth,
                Documents = q.W9FormDocs.Select(x => new DocItem { FileNameOriginal = x.FileNameOriginal, FileNameOnDisk = x.FileNameOnDisk }).ToList(), //inquire about w9formdocs
                TaxClassification = taxClassificationPrefix + taxClassificationSuffix,
                TaxpayerIdNum1 = q.CounterpartyParent.FederalTaxId,
                TaxpayerIdNum2 = q.Counterparty != null ? q.Counterparty!.FederalTaxId : "",
                Address = string.Join(Environment.NewLine, new[] { addressLine1, addressLine2 }),
                City = q.CounterpartyParent.City,
                State = q.CounterpartyParent.State != null ? q.CounterpartyParent.State!.Name : "", // compiler thinks this is a null dereference
                Zip = q.CounterpartyParent.Zip,
                Notes = q.Notes

            }
        ).FirstAsync();

        return Ok(detail);
    }

    public enum SaveType
    {
        New = 1,
        Normal = 2
    }

    [Permission("W-9 Forms", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> SaveDetail(W9FormDetail detail, SaveType saveType)
    {
        int resultId = 0;
        int userInfoId = Util.GetAppUserId(User);

        await db.Database.CreateExecutionStrategy().Execute(async () =>
        {
            using var dbContextTransaction = await db.Database.BeginTransactionAsync();
            W9Form? dbItem = null;
            if (saveType != SaveType.New)
            {
                dbItem = (
                    from q in db.W9Forms
                    .Include(x => x.W9FormDocs)
                    where q.Id == detail.Id
                    select q
                ).FirstOrDefault();
            }

            if (dbItem == null)
            {
                dbItem = new W9Form();
                db.W9Forms.Add(dbItem);
            }
            else
            {
                //remove existing items so that they get completely re-inserted
                db.W9FormDocs.RemoveRange(dbItem.W9FormDocs);
            }

            var d = detail;
            dbItem.CounterpartyId = d.ChildCounterpartyId;
            dbItem.CounterpartyParentId = d.ParentCounterpartyId;
            dbItem.VersionMonth = Util.Date.FirstDayOfMonth(d.VersionMonth);
            dbItem.Notes = d.Notes;

            //if a "Save New" is performed then we might have two different db records pointing to the same FileNameOnDisk
            //this is not a problem because files are only removed from the disk when they no longer have any references to them
            dbItem.W9FormDocs.Clear();
            var newDocs = d.Documents?.Select(x => new W9FormDoc
            {
                FileNameOriginal = x.FileNameOriginal,
                FileNameOnDisk = x.FileNameOnDisk
            }).ToList() ?? new List<W9FormDoc>();
            foreach (var item in newDocs)
                dbItem.W9FormDocs.Add(item);

            await db.SaveChangesAsync();
            resultId = dbItem.Id;

            await dbContextTransaction.CommitAsync();
        });

        return Ok(resultId);
    }

    [Permission("W-9 Forms", PermissionType.Modify)]
    [Route("[action]/{id}")]
    public async Task<IActionResult> DeleteDetail(int id)
    {
        W9Form dbItem = await db.W9Forms.Where(x => x.Id == id).FirstAsync();
        db.Entry(dbItem).State = EntityState.Deleted;

        await db.SaveChangesAsync();
        return Ok();
    }

    [Permission("W-9 Forms", PermissionType.View)]
    [Route("[action]/{counterpartyParentId?}/{counterpartyId?}")]
    public IActionResult GetW9FormInfo(int counterpartyParentId, int? counterpartyId)
    {
        var taxpayerIdNum1 = (from q in db.Counterparties where q.Id == counterpartyParentId select q.FederalTaxId).FirstOrDefault();
        var taxpayerIdNum2 = (from q in db.Counterparties where q.Id == counterpartyId select q.FederalTaxId).FirstOrDefault();

        //taxClassificationPrefix = if LLC then "LLC -" else if subtype null then "Business Type" else ""
        //taxClassificationSuffix = if subtype not null then "Business SubType" else ""
        var result = (
            from q in db.Counterparties
            let businessTypeName = q.BusinessType != null ? q.BusinessType.Name : ""
            let taxClassificationPrefix = businessTypeName == "LLC" ? "LLC - " : q.BusinessSubTypeId == null ? businessTypeName : ""
            let taxClassificationSuffix = q.BusinessSubTypeId != null ? businessTypeName : ""
            where q.Id == counterpartyParentId
            select new
            {
                taxClassification = taxClassificationPrefix + taxClassificationSuffix,
                address = string.Join(Environment.NewLine, new[] { q.AddressLine1, q.AddressLine2 }),
                city = q.City,
                state = (q.State != null) ? q.State.Name : "",
                zip = q.Zip,
                taxpayerIdNum1,
                taxpayerIdNum2
            }
        ).First();

        return Ok(result);
    }

    [Permission("W-9 Forms", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> UploadDoc(IEnumerable<IFormFile> files, [FromODataUri] string metaData)
    {
        var docItems = new List<DocItem>();

        MemoryStream ms = new(Encoding.UTF8.GetBytes(metaData));
        var serializer = new DataContractJsonSerializer(typeof(ChunkMetaData));
        ChunkMetaData? chunkData = serializer.ReadObject(ms) as ChunkMetaData;
        string? fileNameOnDisk = null;

        if (files != null)
        {
            foreach (var file in files)
            {
                if (file.Length > 0 && chunkData != null)
                {
                    fileNameOnDisk = await Util.File.SaveFileAsync(env.ContentRootPath, "W9Forms", file);
                }
            }
        }

        if (fileNameOnDisk != null && chunkData != null && chunkData.ChunkIndex == chunkData.TotalChunks - 1)
        {
            docItems.Add(new DocItem
            {
                FileNameOriginal = chunkData.FileName,
                FileNameOnDisk = fileNameOnDisk
            });
            return Ok(docItems);
        }
        else
        {
            return Ok();
        }
    }

    [Permission("W-9 Forms", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> DownloadDoc(string fileNameOnDisk)
    {
        try
        {
            var fileResponse = await Util.File.GetFileAsync(env.ContentRootPath, "W9Forms", fileNameOnDisk);
            return File(fileResponse.Stream, fileResponse.ContentType, fileResponse.FileName);
        }
        catch (Exception ex)
        {
            var prefix = "W9 Form";
            var bytes = Util.GetExceptionFilesBytes(ex, prefix);
            return File(bytes, "text/plain");
        }
    }
}
