using System.Runtime.Serialization.Json;
using Microsoft.AspNetCore.OData.Formatter;
using Microsoft.Extensions.FileProviders;

namespace Fast.Web.Controllers;

[Authorize]
[ApiController]
[Route("api/[controller]")]
public class CreditLimitController : ODataController
{
    private readonly MyDbContext db;
    private readonly AuthorizationHelper authHelper;
    private readonly string creditLimitDocuments;

    public CreditLimitController(MyDbContext context, IWebHostEnvironment env)
    {
        db = context;
        authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);

        creditLimitDocuments = Path.Join(env?.ContentRootPath, "CreditLimitApprovalDocs");
        if (!Directory.Exists(creditLimitDocuments))
            Directory.CreateDirectory(creditLimitDocuments);
    }

    [Permission("Credit Limit", PermissionType.View)]
    [Route("/odata/GetCreditLimitItems")]
    public async Task<IActionResult> GetItems(ODataQueryOptions<CreditLimitListItem> queryOptions, bool isExport)
    {
        queryOptions = Util.GetQueryOptionsWithConvertedDates(queryOptions);
        var itemsQueryable = GetItemsInternal();
        itemsQueryable = queryOptions.ApplyTo(itemsQueryable) as IQueryable<CreditLimitListItem>;
        var items = itemsQueryable == null ? null : await itemsQueryable.ToListAsync();

        if (isExport)
            return File(Util.Excel.GetExportFileStream(items), "application/octet-stream");
        else
            return Ok(items);
    }

    private IQueryable<CreditLimitListItem> GetItemsInternal()
    {
        IQueryable<CreditLimitListItem>? itemsQueryable = null;

        itemsQueryable = (
            from q in db.CreditLimits
            select new CreditLimitListItem
            {
                Id = q.Id,
                ParentCounterparty = q.ParentCounterparty.Name,
                TotalCreditLimit = q.TotalCreditLimit,
                ApprovedCreditLimit = q.ApprovedCreditLimit,
                IsOverLimitException = q.IsOverLimitException == true ? "yes" : "no",
                ExceptionCreditLimit = q.ExceptionCreditLimit,
                ExceptionExpirationDate = q.ExceptionExpirationDate,
                ApprovedDate = q.ApprovedDate,
                ReviewedDate = q.ReviewedDate,
                ExpirationDate = q.ExpirationDate,
            }
        );

        return itemsQueryable;
    }

    [Permission("Credit Limit", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetRequiredData()
    {

        var hasModifyPermission = await authHelper.IsAuthorizedAsync(User, "Credit Limit", PermissionType.Modify);
        var counterparties = await DataHelper.GetCounterpartiesAsync(false, Enums.ProductCategory.All);
        var counterpartyParents = (from q in db.CounterpartyParents select new { q.CounterpartyId, q.ParentId }).ToList();
        var collateralTypes = (from q in db.CollateralTypes orderby q.Name select new { q.Id, q.Name }).ToList();
        var productGroups = (from q in db.ProductCategories select new { q.Id, q.Name }).ToList();
        var result = new { hasModifyPermission, counterparties, counterpartyParents, collateralTypes, productGroups };

        return Ok(result);
    }

    [Permission("Credit Limit", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetProductCounterparties(int productGroupId)
    {
        var commoditySelection = productGroupId switch
        {
            (int)Enums.ProductCategory.NaturalGasAndLng => Enums.ProductCategory.NaturalGasAndLng,
            (int)Enums.ProductCategory.CrudeOil => Enums.ProductCategory.CrudeOil,
            _ => Enums.ProductCategory.All
        };

        var addItemCounterparties = await DataHelper.GetCounterpartiesAsync(false, commoditySelection);

        return Ok(addItemCounterparties);
    }

    [Permission("Credit Limit", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetDetail(int id)
    {
        var detail = await (
        from q in db.CreditLimits
        where q.Id == id
        select new CreditLimitDetail
        {
            Id = q.Id,
            ParentCounterpartyId = q.ParentCounterparty.Id,
            TotalCreditLimit = q.TotalCreditLimit,
            ApprovedCreditLimit = q.ApprovedCreditLimit,
            IsOverLimitException = q.IsOverLimitException,
            ExceptionCreditLimit = q.ExceptionCreditLimit,
            ExceptionExpirationDate = q.ExceptionExpirationDate,
            ApprovedDate = q.ApprovedDate,
            ReviewedDate = q.ReviewedDate,
            ExpirationDate = q.ExpirationDate,
            ApprovalDocSize = q.CreditLimitApprovals.Count,
            ProductGroups = q.CreditLimitCounterparties.Select(x => new ProductGroupDetail { CreditLimitId = q.Id, CreditLimitCounterpartyId = x.Id, ProductGroupId = x.ProductCategoryId, CounterpartyId = x.CounterpartyId, CreditLimitAmount = x.CreditLimitAmount ?? 0, CollateralSize = x.CreditLimitCollaterals.Count }).ToList(),
        }
        ).FirstAsync();

        return Ok(detail);
    }

    public enum SaveType
    {
        New = 1,
        Normal = 2
    }

    [Permission("Credit Limit", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> SaveDetail(CreditLimitDetail detail, SaveType saveType)
    {
        int resultId = 0;
        int userInfoId = Util.GetAppUserId(User);

        await db.Database.CreateExecutionStrategy().Execute(async () =>
        {
            using var dbContextTransaction = await db.Database.BeginTransactionAsync();
            ILookup<(int, int), CreditLimitCollateral>? oldCollateralsByProductGroupAndCounterparty = null;

            CreditLimit? dbItem = null;
            if (saveType != SaveType.New)
            {
                dbItem = (
                    from q in db.CreditLimits
                    .Include(x => x.CreditLimitApprovals)
                    .Include(x => x.CreditLimitCounterparties)
                    .ThenInclude(x => x.CreditLimitCollaterals)
                    .ThenInclude(x => x.CreditLimitCollateralProviders)
                    .Include(x => x.CreditLimitCounterparties)
                    .ThenInclude(x => x.CreditLimitCollaterals)
                    .ThenInclude(x => x.CreditLimitCollateralBeneficiaries)
                    .Include(x => x.CreditLimitCounterparties)
                    .ThenInclude(x => x.CreditLimitCollaterals)
                    .ThenInclude(x => x.CreditLimitCollateralDocs)
                    where q.Id == detail.Id
                    select q
                ).FirstOrDefault();
            }

            if (dbItem == null)
            {
                dbItem = new CreditLimit();
                db.CreditLimits.Add(dbItem);
            }
            else
            {
                //we store the old collaterals so that, after we remove CreditLimitCounterparties, we can re-create the collaterals that were previously attached to them
                oldCollateralsByProductGroupAndCounterparty = dbItem.CreditLimitCounterparties
                    .SelectMany(x => x.CreditLimitCollaterals)
                    .ToLookup(x => (x.CreditLimitCounterparty.ProductCategoryId, x.CreditLimitCounterparty.CounterpartyId));

                //remove existing items so that they get completely re-inserted
                db.CreditLimitCounterparties.RemoveRange(dbItem.CreditLimitCounterparties);
            }

            var d = detail;
            dbItem.ParentCounterpartyId = d.ParentCounterpartyId;
            dbItem.ApprovedCreditLimit = d.ApprovedCreditLimit;
            dbItem.TotalCreditLimit = d.TotalCreditLimit;
            dbItem.IsOverLimitException = d.IsOverLimitException;
            dbItem.ExceptionCreditLimit = d.ExceptionCreditLimit;
            dbItem.ExceptionExpirationDate = d.ExceptionExpirationDate;
            dbItem.ApprovedDate = d.ApprovedDate;
            dbItem.ReviewedDate = d.ReviewedDate;
            dbItem.ExpirationDate = d.ExpirationDate;
            var combinedCreditLimitCounterparties = new List<CreditLimitCounterparty>();
            foreach (var pg in d.ProductGroups)
            {
                var clc = new CreditLimitCounterparty();
                clc.ProductCategoryId = pg.ProductGroupId;
                clc.CounterpartyId = pg.CounterpartyId;
                clc.CreditLimitAmount = pg.CreditLimitAmount;
                combinedCreditLimitCounterparties.Add(clc);

                if (pg.CollateralSize > 0 && oldCollateralsByProductGroupAndCounterparty != null)
                {
                    //re-create any previously existing collaterals
                    var oldCollaterals = oldCollateralsByProductGroupAndCounterparty[(pg.ProductGroupId, pg.CounterpartyId)];
                    foreach (var oldCollateral in oldCollaterals)
                    {
                        var newCollateral = new CreditLimitCollateral();
                        newCollateral.CollateralAmount = oldCollateral.CollateralAmount;
                        newCollateral.CollateralTypeId = oldCollateral.CollateralTypeId;
                        newCollateral.CreditLimitCollateralProviders = oldCollateral.CreditLimitCollateralProviders.Select(x => new CreditLimitCollateralProvider { ProviderId = x.ProviderId }).ToList();
                        newCollateral.CreditLimitCollateralBeneficiaries = oldCollateral.CreditLimitCollateralBeneficiaries.Select(x => new CreditLimitCollateralBeneficiary { BeneficiaryId = x.BeneficiaryId }).ToList();
                        newCollateral.EffectiveDate = oldCollateral.EffectiveDate;
                        newCollateral.ExpirationDate = oldCollateral.ExpirationDate;
                        newCollateral.ExpirationNotice = oldCollateral.ExpirationNotice;
                        newCollateral.Notes = oldCollateral.Notes;
                        newCollateral.CreditLimitCollateralDocs = oldCollateral.CreditLimitCollateralDocs.Select(x => new CreditLimitCollateralDoc { FileNameOriginal = x.FileNameOriginal, FileNameOnDisk = x.FileNameOnDisk }).ToList();
                        clc.CreditLimitCollaterals.Add(newCollateral);
                    }
                }
            }
            dbItem.CreditLimitCounterparties = combinedCreditLimitCounterparties;

            await db.SaveChangesAsync();
            resultId = dbItem.Id;

            await dbContextTransaction.CommitAsync();
        });

        return Ok(resultId);
    }

    [Permission("Credit Limit", PermissionType.Modify)]
    [Route("[action]/{id}")]
    public async Task<IActionResult> DeleteDetail(int id)
    {
        CreditLimit dbItem = await db.CreditLimits.Where(x => x.Id == id).FirstAsync();
        db.Entry(dbItem).State = EntityState.Deleted;

        await db.SaveChangesAsync();

        return Ok();
    }


}
