using System.Collections.Concurrent;
using System.Drawing.Printing;
using System.Runtime.Serialization.Json;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.OData.Formatter;
using Microsoft.Extensions.FileProviders;

namespace Fast.Web.Controllers;

[Authorize]
[ApiController]
[Route("api/[controller]")]
public class CreditCollateralController : ODataController
{
    private readonly MyDbContext db;
    private readonly AuthorizationHelper authHelper;
    private readonly IWebHostEnvironment env;
    private readonly string creditCollateralDocuments;

    public CreditCollateralController(MyDbContext context, IWebHostEnvironment env)
    {
        db = context;
        authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
        this.env = env;

        creditCollateralDocuments = Path.Join(env?.ContentRootPath, "CreditLimitCollateralDocs");
        if (!Directory.Exists(creditCollateralDocuments))
            Directory.CreateDirectory(creditCollateralDocuments);

    }

    [Permission("Credit Limit", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetItems(int creditLimitCounterpartyId)
    {
        var items = await (
            from clco in db.VwCreditLimitCollateralOverviews
            join clc in db.CreditLimitCollaterals on clco.Id equals clc.Id
            join ct in db.CollateralTypes on clco.CollateralTypeId equals ct.Id
            where clco.CreditLimitCounterpartyId == creditLimitCounterpartyId
            let attachCount = clc.CreditLimitCollateralDocs.Count
            select new CreditCollateralListItem
            {
                CreditLimitCounterpartyId = clco.CreditLimitCounterpartyId!.Value,
                CreditLimitId = clco.Id!.Value,
                CollateralType = ct.Name ?? "",
                CollateralAmount = clco.CollateralAmount,
                Beneficiaries = clco.Beneficiaries ?? "",
                Providers = clco.Providers ?? "",
                EffectiveDate = clco.EffectiveDate,
                ExpirationDate = clco.ExpirationDate,
                ExpirationNotice = clco.ExpirationNotice,
                Attachments = attachCount == 0 ? "" : attachCount.ToString() + (attachCount > 1 ? " items" : " item"),
                IsAmendment = clc.IsAmendment ? "Yes" : "No"
            }
        ).ToListAsync();

        return Ok(items);
    }

    [Permission("Credit Limit", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetRequiredData(int creditLimitCounterpartyId)
    {
        var creditLimitCounterpartyData = await (
            from q in db.CreditLimitCounterparties
            where q.Id == creditLimitCounterpartyId
            select new
            {
                ProductGroupName = q.ProductCategory.Name,
                CounterpartyName = q.Counterparty.Name,
                q.ProductCategoryId
            }
        ).FirstAsync();

        var hasModifyPermission = await authHelper.IsAuthorizedAsync(User, "Credit Limit", PermissionType.Modify);
        var collateralTypes = (from q in db.CollateralTypes orderby q.Name select new { q.Id, q.Name }).ToList();
        var counterparties = new List<EntityInfo>();
        var commoditySelection = creditLimitCounterpartyData.ProductCategoryId switch
        {
            (int)(Enums.ProductCategory.NaturalGasAndLng) => Enums.ProductCategory.NaturalGasAndLng,
            (int)(Enums.ProductCategory.CrudeOil) => Enums.ProductCategory.CrudeOil,
            _ => Enums.ProductCategory.All
        };
        counterparties = await DataHelper.GetCounterpartiesAsync(false, commoditySelection);
        var productGroups = (from q in db.ProductCategories select new { q.Id, q.Name }).ToList();
        var productGroupName = creditLimitCounterpartyData.ProductGroupName;
        var counterpartyName = creditLimitCounterpartyData.CounterpartyName;

        var result = new { hasModifyPermission, collateralTypes, counterparties, productGroups, productGroupName, counterpartyName };
        return Ok(result);
    }


    [Permission("Credit Limit", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetDetail(int id)
    {
        var detail = await (
        from q in db.CreditLimitCollaterals
        where q.Id == id
        select new CreditCollateralDetail
        {
            Id = q.Id,
            CreditLimitCounterpartyId = q.CreditLimitCounterpartyId,
            CollateralAmount = q.CollateralAmount,
            CollateralTypeId = q.CollateralTypeId,
            ProviderIds = q.CreditLimitCollateralProviders.Select(x => x.ProviderId).ToList(),
            BeneficiaryIds = q.CreditLimitCollateralBeneficiaries.Select(x => x.BeneficiaryId).ToList(),
            EffectiveDate = q.EffectiveDate,
            ExpirationDate = q.ExpirationDate,
            ExpirationNotice = q.ExpirationNotice,
            Attachments = q.CreditLimitCollateralDocs.Select(x => new DocItem { FileNameOriginal = x.FileNameOriginal, FileNameOnDisk = (x.FileNameOnDisk ?? "") }).ToList(),
            Notes = q.Notes ?? "",
            IsAmendment = q.IsAmendment,
        }
        ).FirstAsync();

        return Ok(detail);
    }

    public enum SaveType
    {
        New = 1,
        Normal = 2
    }

    [Permission("Credit Limit", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> SaveDetail(CreditCollateralDetail detail, SaveType saveType, int creditLimitCounterpartyId)
    {
        int resultId = 0;
        int userInfoId = Util.GetAppUserId(User);

        await db.Database.CreateExecutionStrategy().Execute(async () =>
        {
            using var dbContextTransaction = await db.Database.BeginTransactionAsync();
            CreditLimitCollateral? dbItem = null;
            if (saveType != SaveType.New)
            {
                dbItem = (
                    from q in db.CreditLimitCollaterals
                    .Include(x => x.CreditLimitCollateralProviders)
                    .Include(x => x.CreditLimitCollateralBeneficiaries)
                    .Include(x => x.CreditLimitCollateralDocs)
                    where q.Id == detail.Id
                    select q
                ).FirstOrDefault();
            }

            if (dbItem == null)
            {
                dbItem = new CreditLimitCollateral();
                db.CreditLimitCollaterals.Add(dbItem);
            }
            else
            {
                //remove existing items so that they get completely re-inserted
                db.CreditLimitCollateralDocs.RemoveRange(dbItem.CreditLimitCollateralDocs);
            }

            var d = detail;
            dbItem.CollateralAmount = d.CollateralAmount;
            dbItem.CollateralTypeId = d.CollateralTypeId;
            dbItem.CreditLimitCollateralProviders = d.ProviderIds.Select(x => new CreditLimitCollateralProvider { ProviderId = x }).ToList();
            dbItem.CreditLimitCollateralBeneficiaries = d.BeneficiaryIds.Select(x => new CreditLimitCollateralBeneficiary { BeneficiaryId = x, }).ToList();
            dbItem.EffectiveDate = d.EffectiveDate;
            dbItem.ExpirationDate = d.ExpirationDate;
            dbItem.ExpirationNotice = d.ExpirationNotice;
            dbItem.CreditLimitCounterpartyId = creditLimitCounterpartyId;
            dbItem.Notes = d.Notes;
            dbItem.IsAmendment = d.IsAmendment;

            //if a "Save New" is performed then we might have two different db records pointing to the same FileNameOnDisk
            //this is not a problem because files are only removed from the disk when they no longer have any references to them
            dbItem.CreditLimitCollateralDocs.Clear();
            var newDocs = d.Attachments?.Select(x => new CreditLimitCollateralDoc
            {
                FileNameOriginal = x.FileNameOriginal,
                FileNameOnDisk = x.FileNameOnDisk
            }).ToList() ?? new List<CreditLimitCollateralDoc>();
            foreach (var item in newDocs)
                dbItem.CreditLimitCollateralDocs.Add(item);

            await db.SaveChangesAsync();
            resultId = dbItem.Id;

            await dbContextTransaction.CommitAsync();
        });

        return Ok(resultId);
    }

    [Permission("Credit Limit", PermissionType.Modify)]
    [Route("[action]/{id}")]
    public async Task<IActionResult> DeleteDetail(int id)
    {
        CreditLimitCollateral dbItem = await db.CreditLimitCollaterals.Where(x => x.Id == id).FirstAsync();
        db.Entry(dbItem).State = EntityState.Deleted;

        await db.SaveChangesAsync();

        return Ok();
    }

    [Permission("Credit Limit", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> UploadDoc(IEnumerable<IFormFile> files, [FromODataUri] string metaData)
    {
        var docItems = new List<DocItem>();

        MemoryStream ms = new(Encoding.UTF8.GetBytes(metaData));
        var serializer = new DataContractJsonSerializer(typeof(ChunkMetaData));
        ChunkMetaData? chunkData = serializer.ReadObject(ms) as ChunkMetaData;
        string? fileNameOnDisk = null;

        if (files != null)
        {
            foreach (var file in files)
            {
                if (file.Length > 0 && chunkData != null)
                {
                    fileNameOnDisk = await Util.File.SaveFileAsync(env.ContentRootPath, "CreditLimitCollateralDocs", file);
                }
            }
        }

        if (fileNameOnDisk != null && chunkData != null && chunkData.ChunkIndex == chunkData.TotalChunks - 1)
        {
            docItems.Add(new DocItem
            {
                FileNameOriginal = chunkData.FileName,
                FileNameOnDisk = fileNameOnDisk
            });
            return Ok(docItems);
        }
        else
        {
            return Ok();
        }
    }

    [Permission("Credit Limit", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> DownloadDoc(string fileNameOnDisk)
    {
        try
        {
            var fileResponse = await Util.File.GetFileAsync(env.ContentRootPath, "CreditLimitCollateralDocs", fileNameOnDisk
            );

            return File(fileResponse.Stream, fileResponse.ContentType, fileResponse.FileName);
        }
        catch (Exception ex)
        {
            var docType = "CreditLimitCollateralDocs";
            var messagePrefix = $"Download Failure ({docType})";
            var bytes = Util.GetExceptionFilesBytes(ex, messagePrefix);
            return File(bytes, "text/plain");
        }
    }
}
