using System.Collections.Concurrent;

namespace Fast.Web.Controllers;

[Authorize]
[ApiController]
[Route("api/[controller]")]
public class CounterpartyController : ODataController
{
    private readonly MyDbContext db;
    private readonly AuthorizationHelper authHelper;

    public CounterpartyController(MyDbContext context)
    {
        db = context;
        authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
    }

    [Permission("Counterparty", PermissionType.View)]
    [Route("/odata/GetCounterpartyItems")]
    public async Task<IActionResult> GetItems(ODataQueryOptions<CounterpartyListItem> queryOptions, bool isExport)
    {
        queryOptions = Util.GetQueryOptionsWithConvertedDates(queryOptions);
        var itemsQueryable = GetItemsInternal();
        itemsQueryable = queryOptions.ApplyTo(itemsQueryable) as IQueryable<CounterpartyListItem>;
        var items = itemsQueryable == null ? null : await itemsQueryable.ToListAsync();

        if (isExport)
            return File(Util.Excel.GetExportFileStream(items), "application/octet-stream");
        else
            return Ok(items);
    }

    private IQueryable<CounterpartyListItem> GetItemsInternal()
    {
        IQueryable<CounterpartyListItem>? itemsQueryable = null;

        itemsQueryable = (
            from c in db.Counterparties
            join co in db.VwCounterpartyOverviewInfos on c.Id equals co.CounterpartyId into j1
            from co in j1.DefaultIfEmpty()
            join i in db.Industries on c.IndustryId equals i.Id into j2
            from i in j2.DefaultIfEmpty()
            select new CounterpartyListItem
            {
                Id = c.Id,
                Name = c.Name,
                ShortName = c.ShortName ?? "",
                Parents = co.Parents ?? "",
                InactiveDate = c.InactiveDate,
                Products = co.Products ?? "",
                Relationships = co.Relationships ?? "",
                Industry = c.Industry == null ? "" : c.Industry.Name,
                Duns = co.Duns ?? ""
            }
        );

        return itemsQueryable;
    }

    [Permission("Counterparty", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetRequiredData()
    {
        var hasModifyPermission = await authHelper.IsAuthorizedAsync(User, "Counterparty", PermissionType.Modify);
        var products = await DataHelper.GetProductsAsync();
        var industries = await (from q in db.Industries orderby q.Name select new { q.Id, q.Name }).ToListAsync();
        var states = await (from q in db.Territories orderby q.Name select new { q.Id, q.Name }).ToListAsync();
        var countries = await (from q in db.Countries orderby q.Id == 1 descending, q.Name select new { q.Id, q.Name }).ToListAsync();
        var counterparties = await (from q in db.Counterparties orderby q.Name select new { q.Id, q.Name }).ToListAsync();
        var relationships = await (from q in db.BusinessRelationships orderby q.Name select new { q.Id, q.Name }).ToListAsync();
        var businessTypes = (from q in db.BusinessTypes orderby q.Name select new { q.Id, q.Name }).ToList();
        var businessSubTypes = (from q in db.BusinessSubTypes orderby q.Name select new { q.Id, q.Name, q.BusinessTypeId }).ToList();
        var result = new { hasModifyPermission, products, industries, relationships, states, countries, counterparties, businessTypes, businessSubTypes };

        return Ok(result);
    }

    [Permission("Counterparty", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetDetail(int id)
    {
        var detail = await (
            from q in db.Counterparties
            where q.Id == id
            select new CounterpartyDetail
            {
                Id = q.Id,
                Name = q.Name,
                ShortName = q.ShortName,
                ParentItems = q.CounterpartyParentCounterparties.Select(x => new CounterpartyParentItem { CounterpartyId = id, ParentId = x.ParentId, ParentName = x.Parent.Name, ParentShortName = x.Parent.ShortName ?? "", OwnershipPercent = x.OwnershipPercent }).ToList(),
                InactiveDate = q.InactiveDate,
                BusinessTypeId = q.BusinessTypeId,
                BusinessSubTypeId = q.BusinessSubTypeId,
                ProductIds = q.CounterpartyProducts.Select(x => x.ProductId).ToList(),
                DunsNums = q.CounterpartyDuns.Select(x => x.Duns).ToList(),
                RelationshipIds = q.CounterpartyRelationships.Select(x => x.BusinessRelationshipId).ToList(),
                IndustryId = q.IndustryId,
                Website = q.Website,
                CountryId = q.CountryId,
                AddressLine1 = q.AddressLine1,
                AddressLine2 = q.AddressLine2,
                City = q.City,
                StateId = q.StateId,
                JurisdictionId = q.JurisdictionId,
                HqLocationId = q.HqLocationId,
                Zip = q.Zip,
                Phone = q.Phone,
                FederalTaxId = q.FederalTaxId,
                Notes = q.Notes,
                OwnershipTreeItems = new List<CounterpartyTreeItem>(),
                InternalCustomerNum = q.InternalCustomerNum,
                InternalVendorNum = q.InternalVendorNum,
            }
        ).FirstAsync();
        detail.OwnershipTreeItems = GetOwnershipTreeItems(id);

        return Ok(detail);
    }

    public enum SaveType
    {
        New = 1,
        Normal = 2
    }

    [Permission("Counterparty", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> SaveDetail(CounterpartyDetail detail, SaveType saveType)
    {
        if (detail.ParentItems.Any(x => x.CounterpartyId == x.ParentId))
            throw new Exception("The parent of a counterparty may not be set to itself");

        int resultId = 0;
        int userInfoId = Util.GetAppUserId(User);

        await db.Database.CreateExecutionStrategy().Execute(async () =>
        {
            using var dbContextTransaction = await db.Database.BeginTransactionAsync();
            Counterparty? dbItem = null;
            if (saveType != SaveType.New)
            {
                dbItem = (
                    from q in db.Counterparties
                        .Include(x => x.CounterpartyRelationships)
                        .Include(x => x.CounterpartyProducts)
                        .Include(x => x.CounterpartyDuns)
                        .Include(x => x.CounterpartyParentCounterparties)
                    where q.Id == detail.Id
                    select q
                ).FirstOrDefault();
            }

            if (dbItem == null) //if the item does not exist then add it
            {
                dbItem = new Counterparty();
                db.Counterparties.Add(dbItem);
            }
            else
            {
                //remove existing items so that they get completely re-inserted
                db.CounterpartyParents.RemoveRange(dbItem.CounterpartyParentCounterparties);
                db.CounterpartyRelationships.RemoveRange(dbItem.CounterpartyRelationships);
                db.CounterpartyProducts.RemoveRange(dbItem.CounterpartyProducts);
                db.CounterpartyDuns.RemoveRange(dbItem.CounterpartyDuns);
            }

            var d = detail;
            dbItem.Name = d.Name.Trim();
            dbItem.ShortName = d.ShortName;
            dbItem.CounterpartyParentCounterparties = d.ParentItems.Select(x => new CounterpartyParent { CounterpartyId = d.Id, ParentId = x.ParentId, OwnershipPercent = x.OwnershipPercent }).ToList();
            dbItem.InactiveDate = d.InactiveDate;
            dbItem.BusinessTypeId = d.BusinessTypeId;
            dbItem.BusinessSubTypeId = d.BusinessSubTypeId;
            dbItem.CounterpartyProducts = d.ProductIds.Select(x => new CounterpartyProduct { CounterpartyId = d.Id, ProductId = x }).ToList();
            dbItem.CounterpartyRelationships = d.RelationshipIds.Select(x => new CounterpartyRelationship { CounterpartyId = d.Id, BusinessRelationshipId = x }).ToList();
            dbItem.CounterpartyDuns = d.DunsNums.Select(x => new CounterpartyDun { CounterpartyId = d.Id, Duns = x }).ToList();
            dbItem.IndustryId = d.IndustryId;
            dbItem.Website = d.Website;
            dbItem.CountryId = d.CountryId;
            dbItem.AddressLine1 = d.AddressLine1;
            dbItem.AddressLine2 = d.AddressLine2;
            dbItem.City = d.City;
            dbItem.StateId = d.StateId;
            dbItem.JurisdictionId = d.JurisdictionId;
            dbItem.HqLocationId = d.HqLocationId;
            dbItem.Zip = d.Zip;
            dbItem.Phone = d.Phone;
            dbItem.FederalTaxId = d.FederalTaxId;
            dbItem.Notes = d.Notes;
            dbItem.InternalCustomerNum = d.InternalCustomerNum;
            dbItem.InternalVendorNum = d.InternalVendorNum;

            await db.SaveChangesAsync();
            resultId = dbItem.Id;

            await dbContextTransaction.CommitAsync();
        });

        return Ok(resultId);
    }

    [Permission("Counterparty", PermissionType.Modify)]
    [Route("[action]/{id}")]
    public async Task<IActionResult> DeleteDetail(int id)
    {
        Counterparty dbItem = await db.Counterparties.Where(x => x.Id == id).FirstAsync();
        db.Entry(dbItem).State = EntityState.Deleted;

        await db.SaveChangesAsync();

        return Ok();
    }

    [Permission("Counterparty", PermissionType.Modify)]
    [Route("[action]/{counterpartyId}/{parentId}/{ownershipPercent}")]
    public async Task<IActionResult> GetNewParentItem(int counterpartyId, int parentId, decimal ownershipPercent)
    {
        var parentItem = await (
            from q in db.Counterparties
            where q.Id == parentId
            select new CounterpartyParentItem
            {
                CounterpartyId = counterpartyId,
                ParentId = parentId,
                ParentName = q.Name,
                ParentShortName = q.ShortName ?? "",
                OwnershipPercent = ownershipPercent
            }
        ).FirstOrDefaultAsync();

        return Ok(parentItem);
    }

    private static List<CounterpartyTreeItem> GetOwnershipTreeItems(int initialCounterpartyId)
    {
        var db = Main.CreateContext();
        var allCounterpartiesById = db.Counterparties.Select(x => new { x.Id, x.Name, x.ShortName }).ToDictionary(x => x.Id);
        var allCounterpartyParents = db.CounterpartyParents.ToList();
        var counterpartyParentsByCptyId = allCounterpartyParents.ToLookup(x => x.CounterpartyId);
        var counterpartyParentsByParentId = allCounterpartyParents.ToLookup(x => x.ParentId);

        List<CounterpartyTreeItem> rootTreeItems = new();
        ConcurrentBag<int> bottomLevelChildIds = new();
        ConcurrentBag<int> topLevelParentIds = new();

        void FillBottomLevelChildIds(int counterpartyId)
        {
            var children = counterpartyParentsByParentId[counterpartyId].ToList();
            if (children.Count == 0)
            {
                if (!bottomLevelChildIds.Contains(counterpartyId))
                    bottomLevelChildIds.Add(counterpartyId);
            }
            foreach (var child in children)
            {
                FillBottomLevelChildIds(child.CounterpartyId);
            }
        }

        void FillTopLevelParentIds(int counterpartyId)
        {
            var parents = counterpartyParentsByCptyId[counterpartyId].ToList();
            if (!parents.Any())
            {
                if (!topLevelParentIds.Contains(counterpartyId))
                    topLevelParentIds.Add(counterpartyId);
            }
            foreach (var parent in parents)
            {
                FillTopLevelParentIds(parent.ParentId);
            }
        }

        FillBottomLevelChildIds(initialCounterpartyId);
        foreach (var bottomLevelChildId in bottomLevelChildIds)
        {
            FillTopLevelParentIds(bottomLevelChildId);
        }

        void BuildTree(CounterpartyTreeItem treeItem)
        {
            var children = counterpartyParentsByParentId[treeItem.Id].ToList();
            foreach (var child in children)
            {
                var childItem = allCounterpartiesById[child.CounterpartyId];
                CounterpartyTreeItem childTreeItem = new(childItem.Id, childItem.Name, childItem.ShortName ?? "", child.OwnershipPercent);
                treeItem.Items.Add(childTreeItem);
                BuildTree(childTreeItem);
            }
        }

        foreach (int topLevelParentId in topLevelParentIds)
        {
            var parentItem = allCounterpartiesById[topLevelParentId];
            CounterpartyTreeItem parentTreeItem = new(parentItem.Id, parentItem.Name, parentItem.ShortName ?? "", null);
            rootTreeItems.Add(parentTreeItem);
            BuildTree(parentTreeItem);
        }

        return rootTreeItems;
    }
}
