namespace Fast.Web.Controllers;

[Authorize]
[ApiController]
[Route("api/[controller]")]
public class CollateralTypeController : ODataController
{
    private readonly MyDbContext db;
    private readonly AuthorizationHelper authHelper;

    public CollateralTypeController(MyDbContext context)
    {
        db = context;
        authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
    }

    [Permission("Collateral Type", PermissionType.View)]
    [Route("/odata/GetCollateralTypeItems")]
    public async Task<IActionResult> GetItems(ODataQueryOptions<CollateralTypeListItem> queryOptions, bool isExport)
    {
        var itemsQueryable = GetItemsInternal();
        itemsQueryable = queryOptions.ApplyTo(itemsQueryable) as IQueryable<CollateralTypeListItem>;
        var items = itemsQueryable == null ? null : await itemsQueryable.ToListAsync();

        if (isExport)
            return File(Util.Excel.GetExportFileStream(items), "application/octet-stream");
        else
            return Ok(items);
    }

    private IQueryable<CollateralTypeListItem> GetItemsInternal()
    {
        IQueryable<CollateralTypeListItem>? itemsQueryable = null;

        itemsQueryable = (
            from q in db.CollateralTypes
            select new CollateralTypeListItem
            {
                Id = q.Id,
                Name = q.Name
            }
        );

        return itemsQueryable;
    }

    [Permission("Collateral Type", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetRequiredData()
    {
        var hasModifyPermission = await authHelper.IsAuthorizedAsync(User, "Collateral Type", PermissionType.Modify);
        var result = new { hasModifyPermission };
        return Ok(result);
    }

    [Permission("Collateral Type", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetDetail(int id)
    {
        var detail = await (
            from q in db.CollateralTypes
            where q.Id == id
            select new CollateralTypeDetail
            {
                Id = q.Id,
                Name = q.Name
            }
        ).FirstAsync();

        return Ok(detail);
    }

    public enum SaveType
    {
        New = 1,
        Normal = 2
    }

    [Permission("Collateral Type", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> SaveDetail(CollateralTypeDetail detail, SaveType saveType)
    {
        int resultId = 0;
        await db.Database.CreateExecutionStrategy().Execute(async () =>
        {
            using var dbContextTransaction = await db.Database.BeginTransactionAsync();
            CollateralType? dbItem = null;
            if (saveType != SaveType.New)
            {
                dbItem = (
                    from q in db.CollateralTypes
                    where q.Id == detail.Id
                    select q
                ).FirstOrDefault();
            }

            if (dbItem == null) //if the item does not exist then add it
            {
                dbItem = new CollateralType();
                db.CollateralTypes.Add(dbItem);
            }
            else
            {
                //remove existing items so that they get completely re-inserted
            }

            var d = detail;
            dbItem.Name = d.Name.Trim();
            await db.SaveChangesAsync();
            resultId = dbItem.Id;

            await dbContextTransaction.CommitAsync();
        });

        return Ok(resultId);
    }

    [Permission("Collateral Type", PermissionType.Modify)]
    [Route("[action]/{id}")]
    public async Task<IActionResult> DeleteDetail(int id)
    {
        CollateralType dbItem = db.CollateralTypes.Where(x => x.Id == id).First();
        db.Entry(dbItem).State = EntityState.Deleted;

        await db.SaveChangesAsync();

        return Ok();
    }
}
