namespace Fast.Web.Controllers;

[Authorize]
[ApiController]
[Route("api/[controller]")]
public class BusinessTypeController : ODataController
{
    private readonly MyDbContext db;
    private readonly AuthorizationHelper authHelper;

    public BusinessTypeController(MyDbContext context)
    {
        db = context;
        authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
    }

    [Permission("Business Type", PermissionType.View)]
    [Route("/odata/GetBusinessTypeItems")]
    public async Task<IActionResult> GetItems(ODataQueryOptions<BusinessTypeListItem> queryOptions, bool isExport)
    {
        var itemsQueryable = GetItemsInternal();
        itemsQueryable = queryOptions.ApplyTo(itemsQueryable) as IQueryable<BusinessTypeListItem>;
        var items = itemsQueryable == null ? null : await itemsQueryable.ToListAsync();

        if (isExport)
            return File(Util.Excel.GetExportFileStream(items), "application/octet-stream");
        else
            return Ok(items);
    }

    private IQueryable<BusinessTypeListItem> GetItemsInternal()
    {
        IQueryable<BusinessTypeListItem>? itemsQueryable = null;

        itemsQueryable = (
            from q in db.VwBusinessTypeOverviewInfos
            select new BusinessTypeListItem
            {
                Id = q.BusinessTypeId!.Value,
                Name = q.BusinessTypeName ?? "",
                SubTypes = q.BusinessSubTypes ?? ""
            }
        );

        return itemsQueryable;
    }

    [Permission("Business Type", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetRequiredData()
    {
        var hasModifyPermission = await authHelper.IsAuthorizedAsync(User, "Business Type", PermissionType.Modify);
        var result = new { hasModifyPermission };
        return Ok(result);
    }

    [Permission("Business Type", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetDetail(int id)
    {
        var detail = await (
            from q in db.BusinessTypes.Include(x => x.BusinessSubTypes)
            where q.Id == id
            select new BusinessTypeDetail
            {
                Id = q.Id,
                Name = q.Name,
                SubTypes = q.BusinessSubTypes.Select(x => new IdName(x.Id, x.Name)).ToList()
            }
        ).FirstAsync();

        return Ok(detail);
    }

    public enum SaveType
    {
        New = 1,
        Normal = 2
    }

    [Permission("Business Type", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> SaveDetail(BusinessTypeDetail detail, SaveType saveType)
    {
        int resultId = 0;
        int userInfoId = Util.GetAppUserId(User);

        await db.Database.CreateExecutionStrategy().Execute(async () =>
        {
            using var dbContextTransaction = await db.Database.BeginTransactionAsync();
            BusinessType? dbItem = null;
            if (saveType != SaveType.New)
            {
                dbItem = await (
                    from q in db.BusinessTypes.Include(x => x.BusinessSubTypes)
                    where q.Id == detail.Id
                    select q
                ).FirstOrDefaultAsync();
            }

            if (dbItem == null) //if the item does not exist then add it
            {
                dbItem = new BusinessType();
                db.BusinessTypes.Add(dbItem);
            }
            else
            {
                //remove sub-types that previously existed
            }

            var d = detail;
            dbItem.Name = d.Name.Trim();

            await db.SaveChangesAsync();
            resultId = dbItem.Id;

            await dbContextTransaction.CommitAsync();
        });

        return Ok(resultId);
    }

    [Permission("Business Type", PermissionType.Modify)]
    [Route("[action]/{id}")]
    public async Task<IActionResult> DeleteDetail(int id)
    {
        BusinessType dbItem = db.BusinessTypes.Where(x => x.Id == id).First();
        db.Entry(dbItem).State = EntityState.Deleted;
        await db.SaveChangesAsync();

        return Ok();
    }

    [Permission("Business Type", PermissionType.Modify)]
    [Route("[action]/{businessTypeId}/{subTypeName}")]
    public async Task<IActionResult> AddNewSubType(int businessTypeId, string subTypeName)
    {
        var subType = new BusinessSubType
        {
            BusinessTypeId = businessTypeId,
            Name = subTypeName
        };
        db.BusinessSubTypes.Add(subType);
        await db.SaveChangesAsync();

        return Ok();
    }

    [Permission("Business Type", PermissionType.Modify)]
    [Route("[action]/{subTypeId}/{subTypeName}")]
    public async Task<IActionResult> RenameSubType(int subTypeId, string subTypeName)
    {
        var subType = db.BusinessSubTypes.Where(x => x.Id == subTypeId).First();
        subType.Name = subTypeName;
        await db.SaveChangesAsync();

        return Ok();
    }

    [Permission("Business Type", PermissionType.Modify)]
    [Route("[action]/{subTypeId}")]
    public async Task<IActionResult> DeleteSubType(int subTypeId)
    {
        var subType = db.BusinessSubTypes.Where(x => x.Id == subTypeId).First();
        db.BusinessSubTypes.Remove(subType);
        await db.SaveChangesAsync();

        return Ok();
    }
}
