using System.Collections.Concurrent;
using System.Runtime.Serialization.Json;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.OData.Formatter;
using Microsoft.Extensions.FileProviders;

namespace Fast.Web.Controllers;

[Authorize]
[ApiController]
[Route("api/[controller]")]
public class ApprovalDocsController : ODataController
{
    private readonly MyDbContext db;
    private readonly AuthorizationHelper authHelper;
    private readonly string creditLimitDocuments;

    private readonly IWebHostEnvironment env;

    public ApprovalDocsController(MyDbContext context, IWebHostEnvironment env)
    {
        db = context;
        authHelper = new AuthorizationHelper(Main.IsAuthenticationEnabled);
        this.env = env;

        creditLimitDocuments = Path.Join(env?.ContentRootPath, "CreditLimitApprovalDocs");
        if (!Directory.Exists(creditLimitDocuments))
            Directory.CreateDirectory(creditLimitDocuments);
    }


    [Permission("Credit Limit", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> GetDetail(int id)
    {
        var detail = await (
        from q in db.CreditLimits
        where q.Id == id
        select new ApprovalDocsDetail
        {
            CreditLimitId = q.Id,
            Documents = q.CreditLimitApprovals.Select(x => new DocItem { FileNameOriginal = x.FileNameOriginal, FileNameOnDisk = (x.FileNameOnDisk ?? "") }).ToList(),
            Notes = q.Notes ?? ""
        }
        ).FirstAsync();

        return Ok(detail);
    }

    public enum SaveType
    {
        New = 1,
        Normal = 2
    }

    [Permission("Credit Limit", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> SaveDetail(ApprovalDocsDetail detail, SaveType saveType)
    {
        int resultId = 0;
        int userInfoId = Util.GetAppUserId(User);


        await db.Database.CreateExecutionStrategy().Execute(async () =>
        {
            using var dbContextTransaction = await db.Database.BeginTransactionAsync();
            CreditLimit? dbItem = null;
            if (saveType != SaveType.New)
            {
                dbItem = (
                    from q in db.CreditLimits
                    .Include(x => x.CreditLimitApprovals)
                    where q.Id == detail.CreditLimitId
                    select q
                ).FirstOrDefault();
            }

            if (dbItem == null)
            {
                dbItem = new CreditLimit();
                db.CreditLimits.Add(dbItem);
            }
            else
            {
                //remove existing items so that they get completely re-inserted
                db.CreditLimitApprovals.RemoveRange(dbItem.CreditLimitApprovals);
            }

            var d = detail;
            dbItem.Notes = d.Notes;

            //if a "Save New" is performed then we might have two different db records pointing to the same FileNameOnDisk
            //this is not a problem because files are only removed from the disk when they no longer have any references to them
            dbItem.CreditLimitApprovals.Clear();
            var newDocs = d.Documents?.Select(x => new CreditLimitApproval
            {
                FileNameOriginal = x.FileNameOriginal,
                FileNameOnDisk = x.FileNameOnDisk
            }).ToList() ?? new List<CreditLimitApproval>();
            foreach (var item in newDocs)
                dbItem.CreditLimitApprovals.Add(item);

            await db.SaveChangesAsync();
            resultId = dbItem.Id;

            await dbContextTransaction.CommitAsync();
        });

        return Ok(resultId);
    }

    [Permission("Credit Limit", PermissionType.Modify)]
    [Route("[action]")]
    public async Task<IActionResult> UploadDoc(IEnumerable<IFormFile> files, [FromODataUri] string metaData)
    {
        var docItems = new List<DocItem>();

        MemoryStream ms = new(Encoding.UTF8.GetBytes(metaData));
        var serializer = new DataContractJsonSerializer(typeof(ChunkMetaData));
        ChunkMetaData? chunkData = serializer.ReadObject(ms) as ChunkMetaData;
        string? fileNameOnDisk = null;

        if (files != null)
        {
            foreach (var file in files)
            {
                if (file.Length > 0 && chunkData != null)
                {
                    fileNameOnDisk = await Util.File.SaveFileAsync(env.ContentRootPath, "CreditLimitApprovalDocs", file);
                }
            }
        }

        if (fileNameOnDisk != null && chunkData != null && chunkData.ChunkIndex == chunkData.TotalChunks - 1)
        {
            docItems.Add(new DocItem
            {
                FileNameOriginal = chunkData.FileName,
                FileNameOnDisk = fileNameOnDisk
            });
            return Ok(docItems);
        }
        else
        {
            return Ok();
        }
    }

    [Permission("Credit Limit", PermissionType.View)]
    [Route("[action]")]
    public async Task<IActionResult> DownloadDoc(string fileNameOnDisk)
    {
        try
        {
            var fileResponse = await Util.File.GetFileAsync(env.ContentRootPath, "CreditLimitApprovalDocs", fileNameOnDisk);

            return File(fileResponse.Stream, fileResponse.ContentType, fileResponse.FileName);
        }
        catch (Exception ex)
        {
            var creditLimitDocTypeStr = "Approval Doc Type";
            var messagePrefix = $"Approval Doc Type: {creditLimitDocTypeStr}";
            var bytes = Util.GetExceptionFilesBytes(ex, messagePrefix);
            return File(bytes, "text/plain");
        }
    }
}
