import { inject, Injectable, signal, computed } from '@angular/core';
import { State } from '@progress/kendo-data-query';
import * as util from '../_shared/utils/util';
import { convertDatesToDateOnlyStrings } from '../_shared/utils/util';
import { FastResourceService } from '../_shared/services/fast-resource.service';

export interface GridItem {
  Id: number;
  ParentCounterparty: string;
  Counterparty: string;
  TaxClassification: string;
  TaxpayerIdNum1: string;
  TaxpayerIdNum2: string;
  VersionMonth?: Date;
}

export interface Detail {
  id: number;
  parentCounterpartyId: number;
  childCounterpartyId: number;
  taxClassification: string;
  taxpayerIdNum1: string;
  taxpayerIdNum2: string;
  versionMonth: Date;
  address: string;
  city: string;
  state: string;
  zip: string;
  notes: string;
  documents: util.DocItem[];
}

export interface ReadOnlyInfo {
  taxClassification: string;
  taxpayerIdNum1: string;
  taxpayerIdNum2: string;
  address: string;
  city: string;
  state: string;
  zip: string;
}

export interface RequiredData {
  hasModifyPermission: boolean;
  counterparties: util.Entity[];
}

@Injectable({
  providedIn: 'root'
})
export class W9FormService {
  baseUrl = `${window.location.origin}/api/W9Form`;
  fastRes = inject(FastResourceService);

  hasModifyPermission = computed(() =>
    this.requiredDataRes.value()?.hasModifyPermission ?? false
  );

  counterparties = computed(() =>
    this.requiredDataRes.value()?.counterparties ?? []
  );

  refreshGridItems = signal<{ uid: string, gridState: State }>(null);
  gridItemsRes = this.fastRes.odata(() => {
    const state = this.refreshGridItems()?.gridState;
    if (!state) return undefined;
    return util.getOdataUrl('GetW9FormItems', state, null);
  });

  refreshRequiredData = signal<{ uid: string }>(null);
  requiredDataRes = this.fastRes.http<RequiredData>(() => {
    this.refreshRequiredData();
    return { url: `${this.baseUrl}/getRequiredData` };
  });

  refreshReadOnlyInfo = signal<{ uid: string, parentCounterpartyId: number; childCounterpartyId?: number }>(null);
  readOnlyInfoRes = this.fastRes.http<ReadOnlyInfo>(() => {
    const params = this.refreshReadOnlyInfo();
    if (!params) return undefined;
    const url = `${this.baseUrl}/GetW9FormInfo/${params.parentCounterpartyId}${params.childCounterpartyId ? `/${params.childCounterpartyId}` : ''}`
    return { url: url };
  });

  refreshDetail = signal<{ uid: string, detailId: number }>(null);
  detailRes = this.fastRes.http<Detail>(() => {
    const id = this.refreshDetail()?.detailId;
    if (!id) return undefined;
    return { url: `${this.baseUrl}/GetDetail?id=${id}` };
  });

  saveDetail = signal<{ uid: string, detail: Detail; saveType: util.SaveType }>(null);
  saveDetailRes = this.fastRes.http<number>(() => {
    const detail = this.saveDetail()?.detail;
    const saveType = this.saveDetail()?.saveType;
    if (!detail) return undefined;

    const processedDetail = convertDatesToDateOnlyStrings(detail, ['versionMonth']);
    return {
      url: `${this.baseUrl}/SaveDetail?saveType=${saveType}`,
      method: 'PUT',
      body: processedDetail
    };
  });

  deleteDetail = signal<{ uid: string, detailId: number }>(null);
  deleteDetailRes = this.fastRes.http<boolean>(() => {
    const id = this.deleteDetail()?.detailId;
    if (!id) return undefined;

    return {
      url: `${this.baseUrl}/DeleteDetail/${id}`,
      method: 'DELETE'
    };
  });

  exportItems = signal<{ uid: string, gridState: State, fileName: string }>(null);
  exportItemsRes = this.fastRes.odataExport(() => {
    const state = this.exportItems()?.gridState;
    if (!state) return undefined;

    const url = util.getOdataExportUrl('GetW9FormItems', state, null);
    return { url: url, fileName: this.exportItems().fileName };
  });

  downloadDoc = signal<{ uid: string, fileNameOriginal: string, fileNameOnDisk: string }>(null);
  downloadDocRes = this.fastRes.httpFile(() => {
    const fileNameOnDisk = this.downloadDoc()?.fileNameOnDisk;
    if (!fileNameOnDisk) return undefined;

    const url = `${this.baseUrl}/DownloadDoc?fileNameOnDisk=${fileNameOnDisk}`;
    return { url: url, fileName: this.downloadDoc().fileNameOriginal };
  });
}
