import { Component, ChangeDetectionStrategy, ViewChild, ElementRef, ViewEncapsulation, signal, computed, inject, model, untracked, effect, Signal } from '@angular/core';
import { toSignal } from '@angular/core/rxjs-interop';
import { Title } from '@angular/platform-browser';
import { Validators } from '@angular/forms';
import { State } from '@progress/kendo-data-query';
import { MessageService } from '../_shared/services/message.service';
import { DialogService, DialogSettings } from '@progress/kendo-angular-dialog';
import { NotifyService } from '../_shared/services/notify.service';
import * as util from '../_shared/utils/util';
import { Detail, GridItem, W9FormService } from './w9form.service';
import { CustomFormBuilder } from '../_shared/services/custom-form-builder.service';
import { SuccessEvent, ErrorEvent, UploadComponent } from '@progress/kendo-angular-upload';
import { SelectableSettings } from '@progress/kendo-angular-grid';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../app.config';
import { uid } from '../_shared/utils/util';
import { FastGridColComponent } from "../_shared/elements/fast-grid-col.component";

interface W9FormModel {
  id: number;
  parentCounterpartyId: number;
  childCounterpartyId: number;
  taxClassification: string;
  versionMonth: Date;
  taxpayerIdNum1: string;
  taxpayerIdNum2: string;
  address: string;
  city: string;
  state: string;
  zip: string;
  notes: string;
  documents: util.DocItem[];
}

@Component({
  selector: 'app-w9form',
  changeDetection: ChangeDetectionStrategy.OnPush,
  templateUrl: './w9form.component.html',
  encapsulation: ViewEncapsulation.None,
  imports: [FAST_KENDO_COMMON, FAST_PAGE_COMMON, FastGridColComponent]
})
export class W9FormComponent {
  @ViewChild("grid", { read: ElementRef }) kendoGridEl!: ElementRef<HTMLElement>;
  @ViewChild('docUploadTarget') docFileUploadElem!: UploadComponent;

  // Properties
  title = 'W-9 Forms';
  docSaveUrl = `${window.location.origin}/api/W9Form/UploadDoc`;
  util = util;
  icons = util.icons;
  gridScrollPosition: util.GridScrollPosition = { topPos: 0, leftPos: 0 };
  detailForm: util.FormModel<W9FormModel>;
  docsChanged: Signal<util.DocItem[]>;
  parentCounterpartyChanged: Signal<number>;
  childCounterpartyChanged: Signal<number>;

  // Injectables
  service = inject(W9FormService);
  messageService = inject(MessageService);
  titleService = inject(Title);
  fb = inject(CustomFormBuilder);
  dialogService = inject(DialogService);
  notify = inject(NotifyService);

  // Signals
  detailOpened = signal(false);
  gridSelection = model<number[]>([]);
  parentCounterpartiesFilter = signal('');
  childCounterpartiesFilter = signal('');

  gridState = signal<State>({
    filter: null,
    group: null,
    skip: 0,
    sort: [{ field: 'ParentCounterparty', dir: 'asc' }],
    take: 50
  });

  constructor() {
    util.trySetTitle(this.titleService, this.title);
    this.service.refreshGridItems.set({ uid: uid(), gridState: this.gridState() });

    this.detailForm = this.createDetailForm();
    this.docsChanged = toSignal(this.detailForm.get('documents')?.valueChanges);
    this.parentCounterpartyChanged = toSignal(this.detailForm.get('parentCounterpartyId')?.valueChanges);
    this.childCounterpartyChanged = toSignal(this.detailForm.get('childCounterpartyId')?.valueChanges);

    effect(() => {
      const readOnlyInfo = this.service.readOnlyInfoRes.value();
      if (readOnlyInfo === undefined || this.service.readOnlyInfoRes.isLoading())
        return;

      this.detailForm.patchValue({
        taxClassification: readOnlyInfo.taxClassification,
        taxpayerIdNum1: readOnlyInfo.taxpayerIdNum1,
        taxpayerIdNum2: readOnlyInfo.taxpayerIdNum2,
        address: readOnlyInfo.address,
        city: readOnlyInfo.city,
        state: readOnlyInfo.state,
        zip: readOnlyInfo.zip
      });
    });

    effect(() => {
      const saveResult = this.service.saveDetailRes.value();
      if (saveResult === undefined || this.service.saveDetailRes.isLoading())
        return;

      untracked(() => {
        this.notify.success('save successful');
        this.closeDetail();
        this.service.refreshGridItems.set({ uid: uid(), gridState: this.gridState() });
        this.gridSelection.set([saveResult]);
      });
    });

    effect(() => {
      const deleteResult = this.service.deleteDetailRes.value();
      if (deleteResult === undefined || this.service.deleteDetailRes.isLoading())
        return;

      untracked(() => {
        this.notify.success('delete successful');
        this.closeDetail();
        this.service.refreshGridItems.set({ uid: uid(), gridState: this.gridState() });
      });
    });

    effect(() => {
      const myDetail = this.detail();
      if (myDetail === undefined)
        return;

      this.detailForm.setValue(myDetail);
      util.focusInputTarget();
    });

    effect(() => {
      this.parentCounterpartyChanged();

      untracked(() => {
        const childCounterparties = this.childCounterparties();
        const parentCounterpartyId = this.detailForm.get('parentCounterpartyId')?.value;
        const childCounterpartyId = this.detailForm.get('childCounterpartyId')?.value;

        // Clear child if not valid for new parent
        if (childCounterparties && childCounterparties.findIndex(x => x.entityId === childCounterpartyId) === -1)
          this.detailForm.patchValue({ childCounterpartyId: null }, { emitEvent: false });

        // Auto-select if only one child
        if (!this.detailLoading() && childCounterparties && childCounterparties.length === 1)
          this.detailForm.patchValue({ childCounterpartyId: childCounterparties[0].entityId });

        if (!this.detailLoading())
          this.partyChange(parentCounterpartyId, this.detailForm.get('childCounterpartyId')?.value);
      });
    });

    effect(() => {
      this.childCounterpartyChanged();

      untracked(() => {
        if (this.detailLoading())
          return;

        const parentCounterpartyId = this.detailForm.get('parentCounterpartyId')?.value;
        const childCounterpartyId = this.detailForm.get('childCounterpartyId')?.value;
        this.partyChange(parentCounterpartyId, childCounterpartyId);
      });
    });
  }

  gridLoading = computed(() => this.service.gridItemsRes.isLoading());
  gridExporting = computed(() => this.service.exportItemsRes.isLoading());
  detailLoading = computed(() =>
    this.service.detailRes.isLoading() ||
    this.service.readOnlyInfoRes.isLoading() ||
    this.service.requiredDataRes.isLoading() ||
    this.service.saveDetailRes.isLoading() ||
    this.service.deleteDetailRes.isLoading()
  );

  hasModifyPermission = computed(() => this.service.requiredDataRes.value()?.hasModifyPermission);
  gridItems = computed(() => this.service.gridItemsRes.value());

  docs = computed(() => {
    this.docsChanged(); //marks a dependency on docsChanged signal, otherwise the computed won't recalculate
    const myDocs = this.detailForm.get('documents')?.value || []
    return myDocs;
  });

  selectableSettings: SelectableSettings = {
    checkboxOnly: false,
    mode: 'single',
    enabled: true
  };

  parentCounterpartiesFiltered = computed(() => {
    const filter = this.parentCounterpartiesFilter();
    const counterparties = this.service.requiredDataRes.value()?.counterparties || [];
    return util.filterText(filter, counterparties, 'fullName');
  });

  childCounterparties = computed(() => {
    const requiredData = this.service.requiredDataRes.value();
    const detail = this.detail();
    this.parentCounterpartyChanged();
    if (requiredData === undefined || detail === undefined)
      return undefined;

    const parentCounterpartyId = this.detailForm.get('parentCounterpartyId')?.value;
    if (!parentCounterpartyId)
      return [];

    const parentCounterparty = requiredData.counterparties.find(x => x.entityId === parentCounterpartyId);
    if (!parentCounterparty)
      return [];

    return requiredData.counterparties.filter(counterparty =>
      parentCounterparty.childIds.includes(counterparty.entityId)
    );
  });

  childCounterpartiesFiltered = computed(() => {
    const filter = this.childCounterpartiesFilter();
    const childCounterparties = this.childCounterparties();
    return util.filterText(filter, childCounterparties, 'fullName');
  });

  detail = computed(() => {
    const myDetail = this.service.detailRes.value();
    if (myDetail === undefined || this.service.detailRes.isLoading())
      return undefined;

    util.convertToDates(myDetail);
    return myDetail;
  });

  createDetailForm(): util.FormModel<W9FormModel> {
    return this.fb.group({
      id: this.fb.ctr(0, Validators.required),
      parentCounterpartyId: this.fb.ctr(null, Validators.required),
      childCounterpartyId: this.fb.ctr(null),
      taxClassification: this.fb.ctr(null),
      versionMonth: this.fb.ctr(null, Validators.required),
      taxpayerIdNum1: this.fb.ctr(null),
      taxpayerIdNum2: this.fb.ctr(null),
      address: this.fb.ctr(null),
      city: this.fb.ctr(null),
      state: this.fb.ctr(null),
      zip: this.fb.ctr(null),
      notes: this.fb.ctr(null),
      documents: this.fb.ctr([]),
    });
  }

  dataStateChange(state: State): void {
    this.gridScrollPosition.topPos = 0;
    this.gridScrollPosition.leftPos = 0;
    util.fixKendoQueryFilter(state.filter);
    this.gridState.set(state);
    this.service.refreshGridItems.set({ uid: uid(), gridState: this.gridState() });
  }

  partyChange(parentCounterpartyId: number, childCounterpartyId: number): void {
    if (parentCounterpartyId != null)
      this.service.refreshReadOnlyInfo.set({ uid: uid(), parentCounterpartyId, childCounterpartyId });
    else {
      this.detailForm.patchValue({
        taxClassification: null,
        taxpayerIdNum1: null,
        taxpayerIdNum2: null,
        address: null,
        city: null,
        state: null,
        zip: null
      });
    }
  }

  openDetail(id: number): void {
    util.saveGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
    this.detailOpened.set(true);

    const detailInitialValues = this.createDetailForm().getRawValue();
    this.detailForm.reset(detailInitialValues, { emitEvent: false });

    if (id === 0) {
      // For new items, just use the initial values
      util.convertToDates(detailInitialValues);
      this.detailForm.setValue(detailInitialValues);
      util.focusInputTarget();
    } else
      this.service.refreshDetail.set({ uid: uid(), detailId: id });
  }

  onDetailClosing(): void {
    util.onDetailChanging(this.detailForm, this.dialogService, this.closeDetail, this.save);
  }

  closeDetail = (): void => {
    this.detailOpened.set(false);
    util.goToSavedGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
  };

  add(): void {
    this.openDetail(0);
  }

  edit(dataItem: GridItem): void {
    this.openDetail(dataItem.Id);
  }

  save = (saveType: util.SaveType): void => {
    this.detailForm.markAllAsTouched();
    if (this.detailForm.valid) {
      const detail: Detail = this.detailForm.value as Detail;
      this.service.saveDetail.set({ uid: uid(), detail, saveType });
    } else {
      this.notify.error("validation failed");
    }
  };

  delete(): void {
    const deleteConfirmSettings: DialogSettings = {
      title: "Please confirm",
      content: "Are you sure you want to delete this item?",
      actions: [{ text: 'No' }, { text: 'Yes', cssClass: 'k-primary' }],
      cssClass: 'utilPrompt'
    };

    this.dialogService.open(deleteConfirmSettings).result.subscribe(result => {
      if (util.getDialogAction(result) === util.dialogAction.Yes) {
        const itemToDelete: Detail = this.detailForm.getRawValue();
        this.service.deleteDetail.set({ uid: uid(), detailId: itemToDelete.id });
      }
    });
  }

  export() {
    this.service.exportItems.set({ uid: uid(), gridState: this.gridState(), fileName: 'W-9Forms.xlsx' });
  }

  docFileUploadSuccess(value: SuccessEvent): void {
    const newDocItems: util.DocItem[] = value.response.body;
    let docs: util.DocItem[] = this.detailForm.get('documents')?.value || [];

    if (docs) {
      docs = docs.concat(newDocItems);
    } else {
      docs = newDocItems;
    }

    this.detailForm.patchValue({ documents: docs });
    this.detailForm.markAsDirty();
  }

  downloadFileDoc(doc: util.DocItem): void {
    if (doc && doc.fileNameOnDisk !== null)
      this.service.downloadDoc.set({ uid: uid(), fileNameOriginal: doc.fileNameOriginal, fileNameOnDisk: doc.fileNameOnDisk });
  }

  removeFileDoc(doc: util.DocItem): void {
    const docs: util.DocItem[] = this.detailForm.get('documents')?.getRawValue() || [];
    const indexToRemove = docs.findIndex(x => x.fileNameOnDisk === doc.fileNameOnDisk);

    if (indexToRemove > -1) {
      docs.splice(indexToRemove, 1);
      this.detailForm.patchValue({ documents: docs });
      this.detailForm.markAsDirty();
    }
  }

  docComplete(): void {
    if (this.docFileUploadElem) {
      this.docFileUploadElem.clearFiles();
    }
  }

  uploadProgress(): void {
    // The kendo-upload uploadProgress must be registered for percentages to update
    // but we don't need to do anything with it
  }

  docUploadError(value: ErrorEvent): void {
    util.handleError(value.response, this.messageService);
  }

  refreshDropdowns() {
    this.service.refreshRequiredData.set({ uid: uid() });
  }
}
