import { inject, Injectable } from '@angular/core';
import { HttpClient } from '@angular/common/http';
import { OdataGridHelper } from '../_shared/utils/odata-grid-helper';
import { Observable } from 'rxjs';
import { State } from '@progress/kendo-data-query';
import { GridDataResult } from '@progress/kendo-angular-grid';
import * as util from '../_shared/utils/util';
import { convertDatesToDateOnlyStrings } from '../_shared/utils/util';

export interface Item {
  Id: number;
  ParentCounterparty: string;
  TotalCreditLimit: number;
  ApprovedCreditLimit: number;
  IsOverLimitException: string;
  ExceptionCreditLimit: number;
  ExceptionExpirationDate: Date;
  ApprovedDate: Date;
  ReviewedDate: Date;
  ExpirationDate: Date;
}

export interface Detail {
  id: number;
  parentCounterpartyId: number;
  totalCreditLimit: number;
  approvedCreditLimit: number;
  isOverLimitException: boolean;
  exceptionCreditLimit: number;
  exceptionExpirationDate: Date;
  approvedDate: Date;
  reviewedDate: Date;
  expirationDate: Date;
  approvalDocSize: number;
  productGroups: ProductGroupDetail[];
}

export interface ProductGroupDetail {
  creditLimitId: number;
  creditLimitCounterpartyId: number;
  productGroupId: number;
  counterpartyId: number;
  creditLimitAmount: number;
  collateralSize: number;
  index: number
}

export interface AddItemDetail {
  productGroupId: number;
  counterpartyId: number;
}

export interface AddFormCounterpartiesData {
  addItemCounterparties: util.Entity[];
}

export interface RequiredData {
  hasModifyPermission: boolean;
  collateralTypes: util.IdName[];
  counterpartyParents: util.IdName[];
  counterparties: util.Entity[];
  productGroups: util.IdName[];
}

@Injectable({
  providedIn: 'root'
})
export class CreditLimitService extends OdataGridHelper {
  readonly baseUrl = `${window.location.origin}/api/CreditLimit`;
  http = inject(HttpClient)

  getRequiredData(): Observable<RequiredData> {
    const url = (`${this.baseUrl}/GetRequiredData`)
    return this.http.get<RequiredData>(url);
  }

  getItems(state: State): Observable<GridDataResult> {
    return this.fetch(`GetCreditLimitItems`, state, null, false)
  }

  getDetail(id: number): Observable<Detail> {
    const url = `${this.baseUrl}/GetDetail?id=${id}`;
    return this.http.get<Detail>(url);
  }

  getProductCounterparties(id: number): Observable<util.Entity[]> {
    const url = `${this.baseUrl}/GetProductCounterparties?productGroupId=${id}`;
    return this.http.get<util.Entity[]>(url);
  }

  saveDetail(detail: Detail, saveType: util.SaveType): Observable<number> {
    const url = `${this.baseUrl}/SaveDetail?saveType=${saveType}`;
    detail = convertDatesToDateOnlyStrings(detail, ['approvedDate', 'expirationDate', 'reviewedDate', 'exceptionExpirationDate']);
    return this.http.put<number>(url, detail);
  }

  deleteDetail(id: number) {
    const url = `${this.baseUrl}/DeleteDetail/${id}`;
    return this.http.delete(url);
  }

  exportItems(state: State, fileNameOriginal: string) {
    return this.fetchExport(`GetCreditLimitItems`, state, null, fileNameOriginal);
  }
}
