import { ChangeDetectionStrategy, Component, ElementRef, inject, ViewChild, ViewEncapsulation } from '@angular/core';
import * as util from '../_shared/utils/util';
import { GridDataResult, SelectableSettings } from '@progress/kendo-angular-grid';
import { tap, take, map, catchError, filter, switchMap, shareReplay, retry } from 'rxjs/operators';
import { of, BehaviorSubject, Subject, combineLatest, Observable } from 'rxjs';
import { State } from '@progress/kendo-data-query';
import { MessageService } from '../_shared/services/message.service';
import { Title } from '@angular/platform-browser';
import { CustomFormBuilder } from '../_shared/services/custom-form-builder.service';
import { DialogService, DialogSettings } from '@progress/kendo-angular-dialog';
import { NotifyService } from '../_shared/services/notify.service';
import { ActivatedRoute } from '@angular/router';
import { AddItemDetail, CreditLimitService, Detail, Item, ProductGroupDetail, RequiredData } from './credit-limit.service';
import { AbstractControl, UntypedFormArray, Validators } from '@angular/forms';
import { saveAs } from '@progress/kendo-file-saver';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../app.config';
import { ApprovalDocsComponent } from '../approval-docs/approval-docs.component';
import { Dictionary } from '../_shared/utils/dictionary';
import { CreditCollateralComponent } from "../credit-collateral/credit-collateral.component";
import { FastGridColComponent } from "../_shared/elements/fast-grid-col.component";

@Component({
  selector: 'app-credit-limit',
  changeDetection: ChangeDetectionStrategy.OnPush,
  templateUrl: './credit-limit.component.html',
  encapsulation: ViewEncapsulation.None,
  imports: [FAST_KENDO_COMMON, FAST_PAGE_COMMON, ApprovalDocsComponent, CreditCollateralComponent, FastGridColComponent]
})
export class CreditLimitComponent {
  @ViewChild("grid", { read: ElementRef }) kendoGridEl: ElementRef;

  util = util;
  icons = util.icons;
  hasModifyPermission = false;
  localRequiredData: RequiredData;
  gridScrollPosition: util.GridScrollPosition = { topPos: 0, leftPos: 0 };
  mySelection: number[] = [];
  isExpanded = () => true;
  detailForm: util.FormModel<Detail>;
  detailInitialValues: Detail;
  productGroupItems: UntypedFormArray;
  creditLimitCounterpartyId = 0;
  addItemForm: util.FormModel<AddItemDetail>;
  distinctProductIds: number[];
  productGroupControlsDictionary = new Dictionary<AbstractControl[]>();

  gridLoading$ = new BehaviorSubject<boolean>(true)
  refreshItems$ = new BehaviorSubject<string>(null)
  detailLoading$ = new BehaviorSubject<boolean>(true)
  refreshRequiredData$ = new BehaviorSubject<util.RefreshType>(null)
  refreshDetail$ = new Subject<number>()

  detailOpened$ = new BehaviorSubject<boolean>(false)
  save$ = new Subject<[util.SaveType, util.AfterCompletedAction]>()
  delete$ = new Subject()
  counterpartyChanged$ = new Subject<number>()
  exporting$ = new BehaviorSubject<boolean>(false)
  exportClicked$ = new Subject()
  approvalDocsOpened$ = new BehaviorSubject<boolean>(false)
  creditCollateralsOpened$ = new BehaviorSubject<boolean>(false)
  approvedLimitChange$: Observable<number>;
  exceptionLimitChange$: Observable<number>;
  overLimitExceptionChange$: Observable<boolean>;
  addItemCounterparties$: Observable<util.Entity[]>;

  title$: Observable<string>;
  requiredData$: Observable<RequiredData>;
  items$: Observable<GridDataResult>;
  detail$: Observable<Detail>;
  exportAction$: Observable<{ fileBlob: Blob; fileName: string; }>;
  saveResult$: Observable<[number, util.AfterCompletedAction]>;
  deleteResult$: Observable<object>;

  filterCounterparties$ = new BehaviorSubject<string>(null)
  filterProductGroups$ = new BehaviorSubject<string>(null)
  filterAssociatedCounterparties$ = new BehaviorSubject<string>(null)

  counterparties$: Observable<util.Entity[]>;
  productGroups$: Observable<util.IdName[]>;
  associatedCounterparties$: Observable<util.Entity[]>;

  selectableSettings: SelectableSettings = {
    checkboxOnly: false,
    mode: 'single',
    enabled: true
  }

  state: State = {
    filter: null,
    group: null,
    skip: 0,
    sort: [{ field: 'ParentCounterparty', dir: 'asc' }],
    take: 50
  };

  service = inject(CreditLimitService);
  messageService = inject(MessageService);
  titleService = inject(Title);
  fb = inject(CustomFormBuilder);
  dialogService = inject(DialogService);
  notify = inject(NotifyService);
  activatedRoute = inject(ActivatedRoute);

  constructor() {
    this.detailForm = this.getDetailForm();
    this.addItemForm = this.getAddItemForm();
    this.detailInitialValues = this.detailForm.getRawValue() as Detail;

    this.approvedLimitChange$ = this.detailForm.get('approvedCreditLimit').valueChanges.pipe(
      filter(() => {
        return !this.detailLoading$.value
      }),
      tap((approvedCreditLimit: number) => {
        const exception = this.detailForm.get('exceptionCreditLimit').value;
        if (approvedCreditLimit === null)
          approvedCreditLimit = 0;
        this.totalCreditLimitPatch(approvedCreditLimit, exception);
      })
    )

    this.exceptionLimitChange$ = this.detailForm.get('exceptionCreditLimit').valueChanges.pipe(
      filter(() => {
        return !this.detailLoading$.value;
      }),
      tap((exceptionCreditLimit: number) => {
        const approved = this.detailForm.get('approvedCreditLimit').value;
        this.totalCreditLimitPatch(approved, exceptionCreditLimit ?? 0);
      })
    )

    this.overLimitExceptionChange$ = this.detailForm.get('isOverLimitException').valueChanges.pipe(
      filter(() => {
        return !this.detailLoading$.value
      }),
      tap((isOverLimitException: boolean) => {
        if (isOverLimitException === false) {
          const hasCurrentExceptionLimitAmount = this.detailForm.get('exceptionCreditLimit').value != null;
          if (hasCurrentExceptionLimitAmount)
            this.detailForm.patchValue({ exceptionCreditLimit: null });
          this.detailForm.patchValue({ exceptionExpirationDate: null });
          this.detailForm.controls['exceptionCreditLimit'].disable({ emitEvent: false });
          this.detailForm.controls['exceptionExpirationDate'].disable();
        }
        else {
          this.detailForm.controls['exceptionCreditLimit'].enable({ emitEvent: false });
          this.detailForm.controls['exceptionExpirationDate'].enable();
        }
      })
    )

    this.addItemCounterparties$ = this.addItemForm.get('productGroupId').valueChanges.pipe(
      filter(() => {
        return !this.detailLoading$.value
      }),
      tap(() => this.detailLoading$.next(true)),
      switchMap((productGroupId) => {
        this.addItemForm.patchValue({ counterpartyId: null });
        if (productGroupId === null || productGroupId === undefined) {
          productGroupId = 0;
          return of([]); // prevention of null exception forcing counterparties to be null.
        }
        return this.service.getProductCounterparties(productGroupId);
      }),
      tap(() => {
        util.focusInputTarget();
        this.detailLoading$.next(false);
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.title$ = of('Credit Limit').pipe(
      tap((title) => util.trySetTitle(this.titleService, title))
    )

    this.requiredData$ = this.refreshRequiredData$.pipe(
      tap(() => this.detailLoading$.next(true)),
      switchMap(refreshType => {
        return combineLatest([this.service.getRequiredData(), of(refreshType)]);
      }),
      map(([requiredData, refreshType]) => {
        this.localRequiredData = requiredData;
        if (refreshType === util.RefreshType.SelfOnly)
          this.detailLoading$.next(false);
        return requiredData;
      }),
      tap((requiredData) => {
        this.hasModifyPermission = requiredData.hasModifyPermission;
        util.focusInputTarget();
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.items$ = this.refreshItems$.pipe(
      tap(() => {
        this.gridLoading$.next(true);
      }),
      switchMap(() => {
        return this.service.getItems(this.state);
      }),
      tap(() => {
        this.gridLoading$.next(false);
        util.goToSavedGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.detail$ = this.refreshDetail$.pipe(
      filter(id => id !== null),
      switchMap(id => {
        this.detailLoading$.next(true);
        this.detailForm.reset();
        if (id === 0)
          return of(this.detailInitialValues);
        else
          return this.service.getDetail(id);
      }),
      map(detail => {
        util.convertToDates(detail);
        this.addItemForm.reset();
        this.createProductGroups(detail);
        this.detailForm.setValue(detail);
        if (detail.id === 0)
          this.detailForm.markAsDirty();

        return detail;
      }),
      tap(detail => {
        this.counterpartyChanged$.next(detail.parentCounterpartyId);
        const overLimit = this.detailForm.get('isOverLimitException').value;
        if (overLimit) {
          this.detailForm.controls['exceptionCreditLimit'].enable({ emitEvent: false });
          this.detailForm.controls['exceptionExpirationDate'].enable();
        } else {
          this.detailForm.controls['exceptionCreditLimit'].disable({ emitEvent: false });
          this.detailForm.controls['exceptionExpirationDate'].disable();
        }
        this.detailLoading$.next(false);
        util.focusInputTarget();
      }),
      shareReplay(1),
      catchError(err => {
        this.closeDetail();
        return util.handleError(err, this.messageService);
      }), retry(10)
    )

    this.exportAction$ = this.exportClicked$.pipe(
      switchMap(() => {
        this.exporting$.next(true);
        return this.service.exportItems(this.state, 'Credit Limits.xlsx');
      }),
      tap(res => {
        saveAs(res.fileBlob, res.fileName);
        this.exporting$.next(false);
      }),
      shareReplay(1),
      catchError(err => {
        this.exporting$.next(false);
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.saveResult$ = this.save$.pipe(
      switchMap(([saveType, completedAction]) => {
        this.detailLoading$.next(true);
        const itemToSave: Detail = this.detailForm.value as Detail;
        return combineLatest([this.service.saveDetail(itemToSave, saveType), of(completedAction)]);
      }),
      tap(([saveResult, action]) => {
        this.notify.success('save successful');

        if (action === util.AfterCompletedAction.CloseDetail) {
          this.mySelection = [saveResult];
          this.closeDetail();
        }
        else
          this.refreshDetail$.next(saveResult);
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.deleteResult$ = this.delete$.pipe(
      switchMap(() => {
        this.detailLoading$.next(true);
        const itemToDelete: Detail = this.detailForm.getRawValue();
        return this.service.deleteDetail(itemToDelete.id);
      }),
      tap(() => {
        this.notify.success('delete successful');
        this.closeDetail();
      }),
      shareReplay(1),
      catchError(err => {
        return util.handleError(err, this.messageService)
      }), retry(10)
    )

    this.counterparties$ = this.filterCounterparties$.pipe(util.filterSpecials(this.detailLoading$, this.requiredData$, 'counterparties', 'fullName'));
    this.productGroups$ = this.filterProductGroups$.pipe(util.filterSpecials(this.detailLoading$, this.requiredData$, 'productGroups', 'name'));
    this.associatedCounterparties$ = this.filterAssociatedCounterparties$.pipe(util.filterSpecials(this.detailLoading$, this.addItemCounterparties$, null, 'fullName'));
  }


  totalCreditLimitPatch(approved: number, exception: number): void {
    approved = Number(approved);
    exception = Number(exception);
    let total = approved + exception;
    if (total == 0)
      total = null;
    this.detailForm.patchValue({ totalCreditLimit: total }, { emitEvent: false });
  }

  dataStateChange(state: State): void {
    this.gridScrollPosition = { topPos: 0, leftPos: 0 };
    util.fixKendoQueryFilter(state.filter);
    this.state = state;
    this.refreshItems$.next(null);
  }

  getDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<Detail> = this.fb.group({
      id: fb.ctr(0, Validators.required),
      parentCounterpartyId: fb.ctr(null, Validators.required),
      totalCreditLimit: fb.ctr(0, Validators.required),
      approvedCreditLimit: fb.ctr(0, Validators.required),
      isOverLimitException: fb.ctr(false, Validators.required),
      exceptionCreditLimit: fb.ctr(null),
      exceptionExpirationDate: fb.ctr({ value: null, disabled: true }),
      approvedDate: fb.ctr(util.currentDate.toDate(), Validators.required),
      reviewedDate: fb.ctr(null),
      expirationDate: fb.ctr(null),
      approvalDocSize: fb.ctr(0),
      productGroups: fb.arr([]),
    });

    return fg;
  }

  getAddItemForm() {
    const fb = this.fb;
    const fg: util.FormModel<AddItemDetail> = this.fb.group({
      productGroupId: fb.ctr(0, Validators.required),
      counterpartyId: fb.ctr(null, Validators.required),
    });

    return fg;
  }

  getProductGroupForm(pg: ProductGroupDetail) {
    const fb = this.fb;
    const fg: util.FormModel<ProductGroupDetail> = this.fb.group({
      creditLimitId: fb.ctr(pg ? pg.creditLimitId : 0, Validators.required),
      creditLimitCounterpartyId: fb.ctr(pg ? pg.creditLimitCounterpartyId : 0, Validators.required),
      productGroupId: fb.ctr(pg ? pg.productGroupId : 0, Validators.required),
      counterpartyId: fb.ctr(pg ? pg.counterpartyId : 0, Validators.required),
      creditLimitAmount: fb.ctr(pg ? pg.creditLimitAmount : 0, Validators.required),
      collateralSize: fb.ctr(pg ? pg.collateralSize : 0, Validators.required),
      index: fb.ctr(pg ? pg.index : 0, Validators.required),
    });

    return fg;
  }

  add(): void {
    this.openDetail(0);
  }

  edit(dataItem: Item): void {
    this.openDetail(dataItem.Id);
  }

  save = (saveType: util.SaveType, action: util.AfterCompletedAction) => {
    this.detailForm.markAllAsTouched();
    if (this.detailForm.valid)
      this.save$.next([saveType, action]);
    else
      this.notify.error("validation failed");
  }

  delete(): void {
    const deleteConfirmSettings: DialogSettings = {
      title: "Please confirm",
      content: "Are you sure you want to delete this item?",
      actions: [{ text: 'No' }, { text: 'Yes', cssClass: 'k-primary' }],
      cssClass: 'utilPrompt'
    }

    this.dialogService.open(deleteConfirmSettings).result.pipe(take(1)).subscribe(result => {
      if (util.getDialogAction(result) === util.dialogAction.Yes)
        this.delete$.next(null);
    });
  }

  export(): void {
    this.exportClicked$.next(null);
  }

  refreshDropdowns() {
    this.refreshRequiredData$.next(util.RefreshType.SelfOnly);
  }

  openDetail(id: number): void {
    this.distinctProductIds = [];
    this.productGroupControlsDictionary.clear();
    util.saveGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
    this.refreshDetail$.next(id);
    this.detailOpened$.next(true);
  }

  openApproval() {
    if (this.detailForm.dirty)
      util.onDetailChanging(this.detailForm, this.dialogService, null, this.save, { extraActionParamValue: util.AfterCompletedAction.DoNothing });
    else
      this.approvalDocsOpened$.next(true);
  }

  openCollateral(creditLimitCounterpartyId: number) {
    this.creditLimitCounterpartyId = creditLimitCounterpartyId;
    if (this.detailForm.dirty)
      util.onDetailChanging(this.detailForm, this.dialogService, null, this.save, { extraActionParamValue: util.AfterCompletedAction.DoNothing });
    else
      this.creditCollateralsOpened$.next(true);
  }

  onDetailClosing() {
    util.onDetailChanging(this.detailForm, this.dialogService, this.closeDetail, this.save, { extraActionParamValue: util.AfterCompletedAction.CloseDetail });
  }

  closeDetail = () => {
    this.detailOpened$.next(false);
    this.detailLoading$.next(false);
    this.refreshItems$.next(null);
    util.goToSavedGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
  }

  closeApprovalWindow() {
    this.approvalDocsOpened$.next(false);
    this.refreshDetail$.next(this.detailForm.get('id').value);
  }

  closeCollateralWindow() {
    this.creditCollateralsOpened$.next(false);
    this.refreshDetail$.next(this.detailForm.get('id').value);
  }

  createProductGroups(detail: Detail) {
    this.productGroupItems = this.detailForm.controls['productGroups'] as UntypedFormArray;
    this.productGroupItems.clear();
    if (detail.productGroups.length > 0) {
      for (let i = 0; i < detail.productGroups.length; i++) {
        const pg = detail.productGroups[i];
        pg.index = i;
        this.productGroupItems.insert(pg.index, this.getProductGroupForm(pg));
      }
    }
    this.fillProductGroupControlsDictionary();
  }

  addProductGroup() {
    if (this.addItemForm.invalid) {
      this.addItemForm.markAllAsTouched();
      this.notify.error('Please select a product group and counterparty');
      return;
    }
    const pg: ProductGroupDetail = {
      creditLimitId: this.detailForm.get('id').value,
      creditLimitCounterpartyId: 0,
      productGroupId: this.addItemForm.get('productGroupId').value,
      counterpartyId: this.addItemForm.get('counterpartyId').value,
      creditLimitAmount: 0,
      collateralSize: 0,
      index: this.productGroupItems.length
    }

    this.productGroupItems.push(this.getProductGroupForm(pg));
    this.fillProductGroupControlsDictionary();
    this.addItemForm.reset();
    this.detailForm.markAsDirty();
  }

  removeProductGroup(idx: number, count: number) {
    const deleteConfirmSettings: DialogSettings = {
      title: "Please confirm",
      content: "Deleting this row will remove any associated collateral items. Are you sure you want to remove this row?",
      actions: [{ text: 'No' }, { text: 'Yes', cssClass: 'k-primary' }],
      cssClass: 'utilPrompt'
    }
    if (count > 0) {
      this.dialogService.open(deleteConfirmSettings).result.pipe(take(1)).subscribe(result => {
        if (util.getDialogAction(result) === util.dialogAction.Yes) {
          this.productGroupItems.removeAt(idx);
          this.productGroupItems.markAsDirty();
          this.fillProductGroupControlsDictionary();
        }
      });
    }
    else {
      this.productGroupItems.removeAt(idx);
      this.productGroupItems.markAsDirty();
      this.fillProductGroupControlsDictionary();
    }
  }

  fillProductGroupControlsDictionary() {
    this.distinctProductIds = this.detailForm.value.productGroups.map((pg) => pg.productGroupId).filter((value, index, self) => self.indexOf(value) === index);
    this.productGroupControlsDictionary.clear();
    this.productGroupItems.controls.forEach(control => {
      const productGroupId = control.get('productGroupId').value;
      if (this.productGroupControlsDictionary.has(productGroupId))
        this.productGroupControlsDictionary.get(productGroupId).push(control);
      else
        this.productGroupControlsDictionary.set(productGroupId, [control]);
    });
  }

  getProductGroupName(productId: number) {
    const prodName = this.localRequiredData.productGroups.find(pg => pg.id === productId).name;
    return prodName;
  }
}
