import { inject, Injectable } from "@angular/core";
import { OdataGridHelper } from "../_shared/utils/odata-grid-helper";
import { HttpClient } from "@angular/common/http";
import { map, Observable } from "rxjs";
import * as util from '../_shared/utils/util';

export interface GridItem {
  id: number;
  creditLimitId: number;
  collateralAmount: number;
  collateralType: string;
  providers: string;
  beneficiaries: string;
  effectiveDate?: Date;
  expirationDate?: Date;
  attachments: string;
}

export interface Detail {
  id: number;
  creditLimitCounterpartyId: number;
  collateralAmount: number;
  collateralTypeId: number;
  effectiveDate: Date;
  expirationDate: Date;
  expirationNotice: string;
  isExpirationNotice: boolean;
  providerIds: number[];
  beneficiaryIds: number[];
  attachments: util.DocItem[];
  notes: string;
  isAmendment: boolean;
}

export interface RequiredData {
  hasModifyPermission: boolean;
  collateralTypes: util.IdName[];
  counterparties: util.Entity[];
  productGroups: util.IdName[];
  productGroupName: string;
  counterpartyName: string;
}


@Injectable({
  providedIn: 'root'
})
export class CreditCollateralService extends OdataGridHelper {
  readonly baseUrl = `${window.location.origin}/api/CreditCollateral`;
  http = inject(HttpClient)

  getRequiredData(creditLimitCounterpartyId: number): Observable<RequiredData> {
    const url = (`${this.baseUrl}/GetRequiredData?creditLimitCounterpartyId=${creditLimitCounterpartyId}`)
    return this.http.get<RequiredData>(url);
  }

  getItems(id: number): Observable<GridItem[]> {
    const url = `${this.baseUrl}/GetItems?creditLimitCounterpartyId=${id}`;
    return this.http.get<GridItem[]>(url);
  }

  getDetail(id: number): Observable<Detail> {
    const url = `${this.baseUrl}/GetDetail?id=${id}`;
    return this.http.get<Detail>(url);
  }

  saveDetail(detail: Detail, saveType: util.SaveType, creditLimitCounterpartyId: number): Observable<number> {
    const url = `${this.baseUrl}/SaveDetail?saveType=${saveType}&creditLimitCounterpartyId=${creditLimitCounterpartyId}`;
    detail = util.convertDatesToDateOnlyStrings(detail, ['effectiveDate', 'expirationDate']);
    return this.http.put<number>(url, detail);
  }

  deleteDetail(id: number) {
    const url = `${this.baseUrl}/DeleteDetail/${id}`;
    return this.http.delete(url);
  }

  downloadDoc(fileNameOriginal: string, fileNameOnDisk: string): Observable<{ fileBlob: Blob, fileName: string }> {
    const url = `${this.baseUrl}/DownloadDoc?fileNameOnDisk=${fileNameOnDisk}`;
    return this.http.get(url, { observe: 'response', responseType: 'blob' }).pipe(
      map(result => {
        const fileBlob = result.body;
        const fileName: string = result.body.type === "text/plain" ? "Error.txt" : fileNameOriginal;
        return { fileBlob, fileName };
      })
    );
  }
}
