import { ChangeDetectionStrategy, Component, ElementRef, inject, input, output, ViewChild, ViewEncapsulation } from "@angular/core";
import * as util from '../_shared/utils/util';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from "../app.config";
import { CreditCollateralService, Detail, GridItem } from "./credit-collateral.service";
import { CustomFormBuilder } from "../_shared/services/custom-form-builder.service";
import { MessageService } from '../_shared/services/message.service';
import { Title } from "@angular/platform-browser";
import { DialogService, DialogSettings } from "@progress/kendo-angular-dialog";
import { NotifyService } from "../_shared/services/notify.service";
import { BehaviorSubject, catchError, combineLatest, filter, map, Observable, of, retry, shareReplay, Subject, switchMap, take, tap } from "rxjs";
import { SelectableSettings } from '@progress/kendo-angular-grid';
import { ActivatedRoute } from "@angular/router";
import { Validators } from "@angular/forms";
import { ErrorEvent, SuccessEvent, UploadComponent, UploadProgressEvent } from "@progress/kendo-angular-upload";
import { FastGridColComponent } from "../_shared/elements/fast-grid-col.component";

@Component({
  selector: 'app-credit-collateral',
  changeDetection: ChangeDetectionStrategy.OnPush,
  templateUrl: './credit-collateral.component.html',
  encapsulation: ViewEncapsulation.None,
  imports: [FAST_KENDO_COMMON, FAST_PAGE_COMMON, FastGridColComponent]
})
export class CreditCollateralComponent {
  @ViewChild("grid", { read: ElementRef }) kendoGridEl: ElementRef;
  @ViewChild('docRateUploadTarget') docFileUploadElem: UploadComponent;

  // global variables, forms, etc  (these comments are purely for organization, will be removed once the component is complete)
  util = util;
  icons = util.icons;
  creditLimitCounterpartyId = input.required<number>();
  closed = output();
  hasModifyPermission = false;
  gridScrollPosition: util.GridScrollPosition = { topPos: 0, leftPos: 0 };
  mySelection: number[] = [];

  detailForm: util.FormModel<Detail>;
  detailInitialValues: Detail;

  docRateSaveUrl = `${window.location.origin}/api/CreditCollateral/UploadDoc`;

  // subjects, behavior subjects, observables
  refreshRequiredData$ = new BehaviorSubject<util.RefreshType>(null)
  gridLoading$ = new BehaviorSubject<boolean>(true)
  refreshItems$ = new BehaviorSubject<string>(null)
  refreshDetail$ = new BehaviorSubject<number>(null)
  detailOpened$ = new BehaviorSubject<boolean>(false)
  detailLoading$ = new BehaviorSubject<boolean>(true)
  save$ = new Subject<util.SaveType>()
  delete$ = new Subject()
  expirationNoticeChange$: Observable<boolean>;

  downloadDoc$ = new Subject<[util.DocItem]>()

  // injectables, services, etc
  service = inject(CreditCollateralService);
  fb = inject(CustomFormBuilder);
  messageService = inject(MessageService);
  titleService = inject(Title);
  dialogService = inject(DialogService);
  notify = inject(NotifyService);
  activatedRoute = inject(ActivatedRoute);

  // states, models, etc
  selectableSettings: SelectableSettings = {
    checkboxOnly: false,
    mode: 'single',
    enabled: true
  }

  constructor() {
    this.detailForm = this.getDetailForm();
    this.detailInitialValues = this.detailForm.getRawValue() as Detail;

    this.expirationNoticeChange$ = this.detailForm.get('isExpirationNotice').valueChanges.pipe(
      filter(() => {
        return !this.detailLoading$.value
      }),
      tap(() => {
        const isExpirationNotice = this.detailForm.get('isExpirationNotice').value;
        if (isExpirationNotice === false) {
          this.detailForm.patchValue({ expirationNotice: null });
        }
        else if (isExpirationNotice === true) {
          this.detailForm.patchValue({ expirationDate: null });
        }
      })

    )
  }

  //observables
  requiredData$ = this.refreshRequiredData$.pipe(
    tap(() => this.detailLoading$.next(true)),
    switchMap(refreshType => {
      return combineLatest([this.service.getRequiredData(this.creditLimitCounterpartyId()), of(refreshType)]);
    }),
    map(([requiredData, refreshType]) => {
      if (refreshType === util.RefreshType.SelfOnly)
        this.detailLoading$.next(false);
      return requiredData;
    }),
    tap(requiredData => {
      this.hasModifyPermission = requiredData.hasModifyPermission;
      util.focusInputTarget();
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  items$ = this.refreshItems$.pipe(
    tap(() => {
      this.gridLoading$.next(true);
    }),
    switchMap(() => {
      return this.service.getItems(this.creditLimitCounterpartyId());
    }),
    map((items) => {
      util.convertToDates(items);
      return items;
    }),
    tap(() => {
      this.gridLoading$.next(false);
      util.goToSavedGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  detail$ = this.refreshDetail$.pipe(
    filter(id => id !== null),
    switchMap(id => {
      this.detailLoading$.next(true);
      this.detailForm.reset();
      if (id === 0)
        return of(this.detailInitialValues);
      else
        return this.service.getDetail(id);
    }),
    map(result => {
      const detail: Detail = result;
      if (detail) {
        util.convertToDates(detail);
        this.detailForm.patchValue(detail);
      }
      return detail;
    }),
    tap(result => {
      if (result.expirationDate == null)
        this.detailForm.patchValue({ isExpirationNotice: true });
      else
        this.detailForm.patchValue({ isExpirationNotice: false });
      this.detailLoading$.next(false);
      util.focusInputTarget();
    }),
    shareReplay(1),
    catchError(err => {
      this.closeDetail();
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  saveResult$ = this.save$.pipe(
    switchMap(saveType => {
      this.detailLoading$.next(true);
      const itemToSave: Detail = this.detailForm.value as Detail;
      return this.service.saveDetail(itemToSave, saveType, this.creditLimitCounterpartyId());
    }),
    tap(() => {
      this.detailLoading$.next(false);
      this.notify.success('save successful');
      this.refreshItems$.next(null);
      this.closeDetail();
    }),
    shareReplay(1),
    catchError(err => {
      this.detailLoading$.next(false);
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  deleteResult$ = this.delete$.pipe(
    switchMap(() => {
      this.detailLoading$.next(true);
      const itemToDelete: Detail = this.detailForm.getRawValue();
      return this.service.deleteDetail(itemToDelete.id);
    }),
    tap(() => {
      this.notify.success('delete successful');
      this.detailLoading$.next(false);
      this.closeDetail();
      this.refreshItems$.next(null);
    }),
    shareReplay(1),
    catchError(err => {
      this.detailLoading$.next(false);
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  downloadDocResult$ = this.downloadDoc$.pipe(
    filter(([docItem]) => {
      return docItem && docItem.fileNameOnDisk !== null;
    }),
    switchMap(([docItem]) => {
      return this.service.downloadDoc(docItem.fileNameOriginal, docItem.fileNameOnDisk);
    }),
    tap(res => {
      util.openOrSaveFile(res.fileBlob, res.fileName);
    }),
    catchError(err => {
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  getDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<Detail> = this.fb.group({
      id: fb.ctr(0, Validators.required),
      creditLimitCounterpartyId: fb.ctr(0, Validators.required),
      collateralAmount: fb.ctr(0, Validators.required),
      collateralTypeId: fb.ctr(null, Validators.required),
      effectiveDate: fb.ctr(null, Validators.required),
      expirationDate: fb.ctr(null),
      expirationNotice: fb.ctr(null),
      isExpirationNotice: fb.ctr(false),
      providerIds: fb.ctr([]),
      beneficiaryIds: fb.ctr([]),
      attachments: fb.ctr([]),
      notes: fb.ctr(null),
      isAmendment: fb.ctr(false)
    });

    return fg;
  }

  refreshDropdowns() {
    this.refreshRequiredData$.next(util.RefreshType.SelfOnly);
  }

  add(): void {
    this.openDetail(0);
  }

  edit(dataItem: GridItem): void {
    this.openDetail(dataItem.creditLimitId);
  }

  delete(): void {
    const deleteConfirmSettings: DialogSettings = {
      title: "Please confirm",
      content: "Are you sure you want to delete this item?",
      actions: [{ text: 'No' }, { text: 'Yes', cssClass: 'k-primary' }],
      cssClass: 'utilPrompt'
    }

    this.dialogService.open(deleteConfirmSettings).result.pipe(take(1)).subscribe(result => {
      if (util.getDialogAction(result) === util.dialogAction.Yes)
        this.delete$.next(null);
    });
  }

  openDetail(id: number): void {
    util.saveGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
    this.refreshDetail$.next(id);
    this.detailOpened$.next(true);
  }

  onDetailClosing() {
    util.onDetailChanging(this.detailForm, this.dialogService, this.closeDetail, this.save);
  }

  closedClicked = () => {
    this.closed.emit();
  }

  save = (saveType: util.SaveType) => {
    this.detailForm.markAllAsTouched();
    if (this.detailForm.valid) {
      this.save$.next(saveType);
    }
    else
      this.notify.error("validation failed");
  }

  closeDetail = () => {
    this.detailOpened$.next(false);
    util.goToSavedGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
  }

  docFileUploadSuccess(value: SuccessEvent) {
    const newdocitems: util.DocItem[] = value.response.body;
    let docs: util.DocItem[] = this.detailForm.get('attachments').value;
    if (docs)
      docs = docs.concat(newdocitems);
    else
      docs = newdocitems;
    this.detailForm.patchValue({ attachments: docs });
    this.detailForm.markAsDirty();
  }

  downloadFileDoc(doc: util.DocItem) {
    this.downloadDoc$.next([doc]);
  }

  removeFileDoc(doc: util.DocItem) {
    const docs: util.DocItem[] = this.detailForm.get('attachments').getRawValue();
    const indexToRemove = docs.findIndex(x => x.fileNameOnDisk === doc.fileNameOnDisk);
    docs.splice(indexToRemove, 1);
    this.detailForm.patchValue({ attachments: docs });
    this.detailForm.markAsDirty();
  }

  docComplete() {
    if (this.docFileUploadElem)
      this.docFileUploadElem.clearFiles();
  }

  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  uploadProgress(event: UploadProgressEvent) {
    //the kendo-upload uploadProgress must be registered for percentages to update
    //but we don't need to do anything with it
  }

  docUploadError(value: ErrorEvent) {
    util.handleError(value.response, this.messageService);
  }

  // filters
  filterCollaterals$ = new BehaviorSubject<string>(null)
  collaterals$ = this.filterCollaterals$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'collateralTypes'));

  filterBeneficiaries$ = new BehaviorSubject<string>(null)
  beneficiaries$ = this.filterBeneficiaries$.pipe(util.filterSpecials(this.detailLoading$, this.requiredData$, 'counterparties', 'fullName'));

  filterProviders$ = new BehaviorSubject<string>(null)
  providers$ = this.filterProviders$.pipe(util.filterSpecials(this.detailLoading$, this.requiredData$, 'counterparties', 'fullName'));
}
