import { inject, Injectable } from '@angular/core';
import { HttpClient } from '@angular/common/http';
import { OdataGridHelper } from '../_shared/utils/odata-grid-helper';
import { Observable } from 'rxjs';
import { State } from '@progress/kendo-data-query';
import { GridDataResult } from '@progress/kendo-angular-grid';
import * as util from '../_shared/utils/util';
import { convertDatesToDateOnlyStrings } from '../_shared/utils/util';

export interface Item {
  Id: number;
  Name: string;
  ShortName: string;
  Parents: string;
  InactiveDate?: Date;
  Products: string;
  Industry: string;
  Duns: string;
}

export interface Detail {
  id: number;
  name: string;
  shortName: string;
  isSupplier: boolean;
  inactiveDate: Date;
  businessTypeId: number;
  businessSubTypeId: number;
  parentItems: CounterpartyParentItem[];
  productIds: number[];
  relationshipIds: number[];
  dunsNums: string[];
  industryId: number;
  website: string;
  countryId: number;
  addressLine1: string;
  addressLine2: string;
  city: string;
  stateId: number;
  jurisdictionId: number;
  hqLocationId: number;
  zip: string;
  phone: string;
  federalTaxId: string;
  notes: string;
  ownershipTreeItems: CounterpartyTreeItem[];
  internalCustomerNum: string;
  internalVendorNum: string;
}

export interface CounterpartyTreeItem {
  id: number;
  name: string;
  shortName: string;
  ownershipPercent: number;
  text: string;
  items: CounterpartyTreeItem[]
}

export interface CounterpartyParentItem {
  counterpartyId: number;
  parentId: number;
  parentName: string;
  parentShortName: string;
  ownershipPercent: number;
  parentPreview: string;
}

export interface DunsItem {
  dunsNum: string;
}

export interface RequiredData {
  hasModifyPermission: boolean;
  products: util.IdName[];
  industries: util.IdName[];
  states: util.IdName[];
  countries: util.IdName[];
  counterparties: util.IdName[];
  relationships: util.IdName[];
  businessTypes: util.IdName[];
  businessSubTypes: SubTypeItem[];
}

export interface SubTypeItem {
  id: number;
  name: string;
  businessTypeId: number;
}

@Injectable({
  providedIn: 'root'
})
export class CounterpartyService extends OdataGridHelper {
  readonly baseUrl = `${window.location.origin}/api/Counterparty`;
  http = inject(HttpClient)

  requiredData$ = this.http.get<RequiredData>(`${this.baseUrl}/getRequiredData`)

  getItems(state: State): Observable<GridDataResult> {
    return this.fetch(`GetCounterpartyItems`, state, null, false)
  }

  getDetail(id: number): Observable<Detail> {
    const url = `${this.baseUrl}/GetDetail?id=${id}`;
    return this.http.get<Detail>(url);
  }

  saveDetail(detail: Detail, saveType: util.SaveType): Observable<number> {
    const url = `${this.baseUrl}/SaveDetail?saveType=${saveType}`;
    detail = convertDatesToDateOnlyStrings(detail, ['inactiveDate']);
    return this.http.put<number>(url, detail);
  }

  deleteDetail(id: number) {
    const url = `${this.baseUrl}/DeleteDetail/${id}`;
    return this.http.delete(url);
  }

  exportItems(state: State, fileNameOriginal: string) {
    return this.fetchExport(`GetCounterpartyItems`, state, null, fileNameOriginal);
  }

  getNewParentItem(counterpartyId: number, parentId: number, ownershipPercent: number) {
    const url = `${this.baseUrl}/GetNewParentItem/${counterpartyId}/${parentId}/${ownershipPercent}`;
    return this.http.get<CounterpartyParentItem>(url);
  }
}
