import { Component, ChangeDetectionStrategy, ViewChild, ElementRef, ViewEncapsulation, AfterViewInit, inject } from '@angular/core';
import { Title } from '@angular/platform-browser';
import { tap, take, map, catchError, switchMap, filter, shareReplay, retry } from 'rxjs/operators';
import { of, BehaviorSubject, Subject, combineLatest, zip } from 'rxjs';
import { CompositeFilterDescriptor, State } from '@progress/kendo-data-query';
import { MessageService } from '../_shared/services/message.service';
import { FormsModule, ReactiveFormsModule, Validators } from '@angular/forms';
import { DialogService, DialogSettings } from '@progress/kendo-angular-dialog';
import { NotifyService } from '../_shared/services/notify.service';
import { saveAs } from '@progress/kendo-file-saver';
import * as util from '../_shared/utils/util';
import { SelectableSettings } from '@progress/kendo-angular-grid';
import { CounterpartyService, Detail, Item, RequiredData, CounterpartyParentItem, DunsItem, SubTypeItem } from './counterparty.service';
import { PreventableEvent } from '@progress/kendo-angular-dropdowns';
import { ActivatedRoute } from '@angular/router';
import { HttpParams } from '@angular/common/http';
import { CustomFormBuilder } from '../_shared/services/custom-form-builder.service';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../app.config';
import { KENDO_TREEVIEW } from '@progress/kendo-angular-treeview';
import { FastGridColComponent } from "../_shared/elements/fast-grid-col.component";

@Component({
  selector: 'app-counterparty',
  changeDetection: ChangeDetectionStrategy.OnPush,
  templateUrl: './counterparty.component.html',
  encapsulation: ViewEncapsulation.None,
  imports: [FAST_KENDO_COMMON, FAST_PAGE_COMMON, KENDO_TREEVIEW, FormsModule, ReactiveFormsModule, FastGridColComponent],
  providers: [CustomFormBuilder]
})
export class CounterpartyComponent implements AfterViewInit {
  @ViewChild("grid", { read: ElementRef }) kendoGridEl: ElementRef;

  util = util;
  icons = util.icons;
  hasModifyPermission = false;
  detailForm: util.FormModel<Detail>;
  detailParentForm: util.FormModel<CounterpartyParentItem>;
  detailDunsForm: util.FormModel<DunsItem>;
  detailInitialValues: Detail;
  detailParentInitialValues: CounterpartyParentItem;
  detailDunsInitialValues: DunsItem;
  localRequiredData: RequiredData;
  gridScrollPosition: util.GridScrollPosition = { topPos: 0, leftPos: 0 };
  mySelection: number[] = [];
  isExpanded = () => true;

  gridLoading$ = new BehaviorSubject<boolean>(true)
  exporting$ = new BehaviorSubject<boolean>(false)
  detailLoading$ = new BehaviorSubject<boolean>(true)
  detailOpened$ = new BehaviorSubject<boolean>(false)
  detailParentOpened$ = new BehaviorSubject<boolean>(false)
  detailDunsOpened$ = new BehaviorSubject<boolean>(false)
  refreshItems$ = new BehaviorSubject<string>(null)
  refreshDetail$ = new BehaviorSubject<number>(null)
  refreshRequiredData$ = new BehaviorSubject(null)
  save$ = new Subject<util.SaveType>()
  delete$ = new Subject()
  exportClicked$ = new Subject()
  businessTypeChanged$ = new Subject<number>()
  localDetail: Detail;
  manualChangesTriggering = true;

  selectableSettings: SelectableSettings = {
    checkboxOnly: false,
    mode: 'single',
    enabled: true
  }

  state: State = {
    filter: null,
    group: null,
    skip: 0,
    sort: [{ field: 'Name', dir: 'asc' }],
    take: 50
  };

  service = inject(CounterpartyService);
  messageService = inject(MessageService);
  titleService = inject(Title);
  fb = inject(CustomFormBuilder);
  dialogService = inject(DialogService);
  notify = inject(NotifyService);
  activatedRoute = inject(ActivatedRoute);

  constructor() {
    this.detailForm = this.getDetailForm();
    this.detailParentForm = this.getDetailParentForm();
    this.detailDunsForm = this.getDetailDunsForm();
    this.detailInitialValues = this.detailForm.value as Detail;
    this.detailParentInitialValues = this.detailParentForm.value as CounterpartyParentItem;
    this.detailDunsInitialValues = this.detailDunsForm.value as DunsItem;
    this.detailParentInitialValues = this.detailParentForm.value as CounterpartyParentItem;
    this.detailDunsInitialValues = this.detailDunsForm.value as DunsItem;
  }

  ngAfterViewInit(): void {
    this.activatedRoute.queryParams.subscribe(x => {
      const params = new HttpParams({ fromObject: x });
      const filter: CompositeFilterDescriptor = { filters: [], logic: 'and' };
      const filters = filter.filters;

      const counterparty = params.get('counterparty');

      if (counterparty)
        filters.push({ field: 'Name', operator: 'eq', value: counterparty });

      this.state.filter = filter;
      this.dataStateChange(this.state);
    });
  }

  getDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<Detail> = this.fb.group({
      id: fb.ctr(0, Validators.required),
      name: fb.ctr(null, Validators.required),
      isSupplier: fb.ctr(false),
      shortName: fb.ctr(null),
      inactiveDate: fb.ctr(null),
      businessTypeId: fb.ctr(null),
      businessSubTypeId: fb.ctr(null),
      parentItems: fb.ctr([]),
      productIds: fb.ctr([]),
      relationshipIds: fb.ctr([]),
      dunsNums: fb.ctr([]),
      industryId: fb.ctr(null),
      website: fb.ctr(null),
      countryId: fb.ctr(null),
      addressLine1: fb.ctr(null),
      addressLine2: fb.ctr(null),
      city: fb.ctr(null),
      stateId: fb.ctr(null),
      jurisdictionId: fb.ctr(null),
      hqLocationId: fb.ctr(null),
      zip: fb.ctr(null),
      phone: fb.ctr(null),
      federalTaxId: fb.ctr(null),
      notes: fb.ctr(null),
      ownershipTreeItems: fb.ctr([]),
      internalCustomerNum: fb.ctr(null),
      internalVendorNum: fb.ctr(null),
    });

    return fg;
  }

  getDetailParentForm() {
    const fb = this.fb;
    const fg: util.FormModel<CounterpartyParentItem> = this.fb.group({
      counterpartyId: fb.ctr(null, Validators.required),
      parentId: fb.ctr(null, Validators.required),
      parentName: fb.ctr(null, Validators.required),
      parentShortName: fb.ctr(null, Validators.required),
      ownershipPercent: fb.ctr(100, Validators.required),
      parentPreview: fb.ctr(null, Validators.required)
    });

    return fg;
  }

  getDetailDunsForm() {
    const fb = this.fb;
    const fg: util.FormModel<DunsItem> = this.fb.group({
      dunsNum: fb.ctr(null, Validators.required),
    });

    return fg;
  }

  title$ = of('Counterparty').pipe(
    tap((title) => util.trySetTitle(this.titleService, title))
  )

  requiredData$ = this.refreshRequiredData$.pipe(
    tap(() => this.detailLoading$.next(true)),
    switchMap(refreshType => {
      return combineLatest([this.service.requiredData$, of(refreshType)]);
    }),
    map(([requiredData, refreshType]) => {
      this.localRequiredData = requiredData;
      if (refreshType === util.RefreshType.SelfOnly)
        this.detailLoading$.next(false);
      return requiredData;
    }),
    tap((requiredData) => {
      this.hasModifyPermission = requiredData.hasModifyPermission;
      util.focusInputTarget();
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  items$ = this.refreshItems$.pipe(
    tap(() => {
      this.gridLoading$.next(true);
    }),
    switchMap(() => {
      return this.service.getItems(this.state);
    }),
    tap(() => {
      this.gridLoading$.next(false);
      util.goToSavedGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  exportAction$ = this.exportClicked$.pipe(
    switchMap(() => {
      this.exporting$.next(true);
      return this.service.exportItems(this.state, 'Counterparties.xlsx');
    }),
    tap(res => {
      saveAs(res.fileBlob, res.fileName);
      this.exporting$.next(false);
    }),
    shareReplay(1),
    catchError(err => {
      this.exporting$.next(false);
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  detail$ = this.refreshDetail$.pipe(
    filter(id => id !== null),
    switchMap(id => {
      this.manualChangesTriggering = true;
      this.detailLoading$.next(true);
      this.detailForm.disable();
      this.detailForm.reset();
      this.detailParentForm.reset();
      this.detailDunsForm.reset();
      if (id === 0)
        return of(this.detailInitialValues);
      else
        return this.service.getDetail(id);
    }),
    map(result => {
      const detail: Detail = result;
      if (detail) {
        util.convertToDates(detail);
        this.detailForm.setValue(detail);
      }
      this.localDetail = detail;
      return detail;
    }),
    tap(result => {
      this.businessTypeChanged$.next(result.businessTypeId);
    }),
    shareReplay(1),
    catchError(err => {
      this.closeDetail();
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  saveResult$ = this.save$.pipe(
    switchMap(saveType => {
      this.detailLoading$.next(true);
      this.detailForm.disable();
      const itemToSave: Detail = this.detailForm.value as Detail;
      return this.service.saveDetail(itemToSave, saveType);
    }),
    tap((saveResult) => {
      this.notify.success('save successful');
      this.closeDetail();
      this.refreshItems$.next(null);
      this.mySelection = [saveResult];
    }),
    shareReplay(1),
    catchError(err => {
      this.detailFinishedLoading();
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  deleteResult$ = this.delete$.pipe(
    switchMap(() => {
      this.detailLoading$.next(true);
      this.detailForm.disable();
      const itemToDelete: Detail = this.detailForm.getRawValue();
      return this.service.deleteDetail(itemToDelete.id);
    }),
    tap(() => {
      this.notify.success('delete successful');
      this.detailFinishedLoading();
      this.closeDetail();
      this.refreshItems$.next(null);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  subTypesForBusinessType$ = this.businessTypeChanged$.pipe(
    map(businessTypeId => {
      let subTypes: SubTypeItem[] = [];
      if (businessTypeId) {
        subTypes = this.localRequiredData.businessSubTypes.filter(subTypeItem => {
          return subTypeItem.businessTypeId === businessTypeId;
        });
      }

      const subTypeId = this.localDetail.businessSubTypeId;
      if (subTypes && subTypes.findIndex(x => x.id === subTypeId) === -1)
        this.detailForm.patchValue({ businessSubTypeId: null }, { emitEvent: false });

      if (!this.detailLoading$.value && subTypes && subTypes.length === 1)
        this.detailForm.patchValue({ businessSubTypeId: subTypes[0].id }, { emitEvent: false });

      return subTypes;

    }),
    shareReplay(1)
  )

  manualTriggersCompleting$ = zip(this.businessTypeChanged$, this.subTypesForBusinessType$).pipe(
    filter(() => this.manualChangesTriggering),
    tap(() => {
      this.detailFinishedLoading();
      this.manualChangesTriggering = false;
    })
  )

  openDetail(id: number): void {
    util.saveGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
    this.refreshDetail$.next(id);
    this.detailOpened$.next(true);
  }

  openParentDetail(item: CounterpartyParentItem): void {
    this.detailForm.markAllAsTouched();
    if (this.detailForm.valid) {
      this.detailParentForm.reset();
      if (item === null)
        item = this.detailParentInitialValues;
      this.detailParentForm.setValue(item);
      this.detailParentOpened$.next(true);
      util.focusInputTarget();
    } else
      this.notify.error("validation failed");
  }

  openDunsDetail(duns: DunsItem): void {
    this.detailForm.markAllAsTouched();
    if (this.detailForm.valid) {
      this.detailDunsForm.reset();
      if (duns === null)
        duns = this.detailDunsInitialValues;
      this.detailDunsForm.setValue(duns);
      this.detailDunsOpened$.next(true);
      util.focusInputTarget();
    } else
      this.notify.error("validation failed");
  }

  onDetailClosing() {
    util.onDetailChanging(this.detailForm, this.dialogService, this.closeDetail, this.save);
  }

  onDetailParentClosing() {
    util.onDetailChanging(this.detailParentForm, this.dialogService, this.closeDetailParent, this.setParent, { confirmType: 'set' });
  }

  onDetailDunsClosing() {
    util.onDetailChanging(this.detailDunsForm, this.dialogService, this.closeDetailDuns, this.setDuns, { confirmType: 'set' });
  }

  closeDetail = () => {
    this.detailOpened$.next(false);
    util.goToSavedGridScrollPos(this.kendoGridEl, this.gridScrollPosition);
  }

  closeDetailParent = () => {
    this.detailParentOpened$.next(false);
  }

  closeDetailDuns = () => {
    this.detailDunsOpened$.next(false);
  }

  detailFinishedLoading(): void {
    this.detailLoading$.next(false);
    this.detailForm.enable();
    util.focusInputTarget();
  }

  add(): void {
    if (this.hasModifyPermission) {
      this.openDetail(0);
      this.detailFinishedLoading();
    }
  }

  addParent(): void {
    this.openParentDetail(null);
  }

  addDuns(): void {
    this.openDunsDetail(null);
  }

  edit(item: Item): void {
    this.openDetail(item.Id);
  }

  save = (saveType: util.SaveType) => {
    this.detailForm.markAllAsTouched();
    if (this.detailForm.valid)
      this.save$.next(saveType);
    else
      this.notify.error("validation failed");
  }

  setParent = () => {
    this.detailParentForm.markAllAsTouched();
    if (this.detailParentForm.controls['parentId'].valid && this.detailParentForm.controls['ownershipPercent'].valid) {
      const detailValue: Detail = this.detailForm.value as Detail;
      const newDetailParent: CounterpartyParentItem = this.detailParentForm.value as CounterpartyParentItem;
      const counterpartyId = detailValue.id;
      const parentId = newDetailParent.parentId;
      const ownershipPercent = newDetailParent.ownershipPercent;
      const parentAlreadyExists = detailValue.parentItems.find(x => x.parentId === parentId);
      if (!parentAlreadyExists) {
        this.service.getNewParentItem(counterpartyId, parentId, ownershipPercent).subscribe({
          next: (newParentItem: CounterpartyParentItem) => {
            detailValue.parentItems.push(newParentItem);
            this.detailForm.setValue(detailValue);
            this.detailParentOpened$.next(false);
            this.detailForm.markAsDirty();
          },
          error: error => {
            this.notify.error(error);
          }
        });
      }
    }
    else
      this.notify.error("validation failed");
  }

  setDuns = () => {
    this.detailDunsForm.markAllAsTouched();
    if (this.detailDunsForm.valid) {
      const detailValue: Detail = this.detailForm.value as Detail;
      const newDetailDuns: DunsItem = this.detailDunsForm.value as DunsItem;
      detailValue.dunsNums.push(newDetailDuns.dunsNum);
      this.detailForm.setValue(detailValue);
      this.detailDunsOpened$.next(false);
      this.detailForm.markAsDirty();
    }
    else
      this.notify.error("validation failed");
  }

  delete(): void {
    const deleteConfirmSettings: DialogSettings = {
      title: "Please confirm",
      content: "Are you sure you want to delete this item?",
      actions: [{ text: 'No' }, { text: 'Yes', cssClass: 'k-primary' }],
      cssClass: 'utilPrompt'
    }

    this.dialogService.open(deleteConfirmSettings).result.pipe(take(1)).subscribe(result => {
      if (util.getDialogAction(result) === util.dialogAction.Yes)
        this.delete$.next(null);
    });
  }

  export(): void {
    this.exportClicked$.next(null);
  }

  dataStateChange(state: State): void {
    this.gridScrollPosition = { topPos: 0, leftPos: 0 };
    util.fixKendoQueryFilter(state.filter);
    this.state = state;
    this.refreshItems$.next(null);
  }

  refreshDropdowns() {
    this.refreshRequiredData$.next(util.RefreshType.SelfOnly);
  }

  preventOpen(event: PreventableEvent) {
    event.preventDefault();
  }

  businessTypeChanged(value: util.IdName) {
    const businessTypeId = value?.id;
    this.businessTypeChanged$.next(businessTypeId);
  }

  filterProducts$ = new BehaviorSubject<string>(null)
  products$ = this.filterProducts$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'products'));

  filterRelationships$ = new BehaviorSubject<string>(null)
  relationships$ = this.filterRelationships$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'relationships'));

  filterCounterparties$ = new BehaviorSubject<string>(null)
  counterparties$ = this.filterCounterparties$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'counterparties'));

  filterIndustries$ = new BehaviorSubject<string>(null)
  industries$ = this.filterIndustries$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'industries'));

  filterStates$ = new BehaviorSubject<string>(null)
  states$ = this.filterStates$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'states'));

  filterJurisdictions$ = new BehaviorSubject<string>(null)
  jurisdictions$ = this.filterJurisdictions$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'states'));

  filterHeadquarters$ = new BehaviorSubject<string>(null)
  headquarters$ = this.filterHeadquarters$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'states'));

  filterCountries$ = new BehaviorSubject<string>(null)
  countries$ = this.filterCountries$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'countries'));

  filterBusinessTypes$ = new BehaviorSubject<string>(null)
  businessTypes$ = this.filterBusinessTypes$.pipe(util.filterIdNames(this.detailLoading$, this.requiredData$, 'businessTypes'));

  filterBusinessSubTypes$ = new BehaviorSubject<string>(null)
  businessSubTypes$ = this.filterBusinessSubTypes$.pipe(util.filterSpecials(this.detailLoading$, this.subTypesForBusinessType$, null, 'name'));
}
