import { Component, ChangeDetectionStrategy, ViewEncapsulation, inject } from '@angular/core';
import { Title } from '@angular/platform-browser';
import { tap, take, map, catchError, switchMap, filter, shareReplay, retry } from 'rxjs/operators';
import { of, BehaviorSubject, Subject, combineLatest } from 'rxjs';
import { State } from '@progress/kendo-data-query';
import { MessageService } from '../_shared/services/message.service';
import { Validators } from '@angular/forms';
import { DialogService, DialogSettings, KENDO_DIALOG } from '@progress/kendo-angular-dialog';
import { KENDO_LISTVIEW } from '@progress/kendo-angular-listview';
import { NotifyService } from '../_shared/services/notify.service';
import { saveAs } from '@progress/kendo-file-saver';
import * as util from '../_shared/utils/util';
import { SelectableSettings } from '@progress/kendo-angular-grid';
import { BusinessTypeService, Detail, Item, RequiredData } from './business-type.service';
import { CustomFormBuilder } from '../_shared/services/custom-form-builder.service';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../app.config';
import { FastGridColComponent } from "../_shared/elements/fast-grid-col.component";

@Component({
  selector: 'app-business-type',
  changeDetection: ChangeDetectionStrategy.OnPush,
  templateUrl: './business-type.component.html',
  encapsulation: ViewEncapsulation.None,
  imports: [FAST_KENDO_COMMON, FAST_PAGE_COMMON, KENDO_DIALOG, KENDO_LISTVIEW, FastGridColComponent]
})
export class BusinessTypeComponent {
  util = util;
  icons = util.icons;
  hasModifyPermission = false;
  detailForm: util.FormModel<Detail>;
  detailSubTypeForm: util.FormModel<util.IdName>;
  detailInitialValues: Detail;
  detailSubTypeInitialValues: util.IdName;
  localRequiredData: RequiredData;
  gridScrollPosition: util.GridScrollPosition = { topPos: 0, leftPos: 0 };
  mySelection: number[] = [];
  selectedSubType: util.IdName = null;

  gridLoading$ = new BehaviorSubject<boolean>(true)
  exporting$ = new BehaviorSubject<boolean>(false)
  detailLoading$ = new BehaviorSubject<boolean>(true)
  detailOpened$ = new BehaviorSubject<boolean>(false)
  detailSubTypeOpened$ = new BehaviorSubject<boolean>(false);
  refreshItems$ = new BehaviorSubject<string>(null)
  refreshDetail$ = new BehaviorSubject<number>(null)
  refreshRequiredData$ = new BehaviorSubject(util.RefreshType.WithOthers)
  save$ = new Subject<util.SaveType>()
  delete$ = new Subject()
  exportClicked$ = new Subject()
  addSubType$ = new Subject()
  renameSubType$ = new Subject()
  deleteSubType$ = new Subject()

  selectableSettings: SelectableSettings = {
    checkboxOnly: false,
    mode: 'single',
    enabled: true
  }

  state: State = {
    filter: null,
    group: null,
    skip: 0,
    sort: [{ field: 'Name', dir: 'asc' }],
    take: 50
  };

  service = inject(BusinessTypeService);
  messageService = inject(MessageService);
  titleService = inject(Title);
  fb = inject(CustomFormBuilder);
  dialogService = inject(DialogService);
  notify = inject(NotifyService);

  constructor() {
    this.detailForm = this.getDetailForm();
    this.detailSubTypeForm = this.getDetailSubTypeForm();
    this.detailInitialValues = this.detailForm.value as Detail;
    this.detailSubTypeInitialValues = this.detailSubTypeForm.value as util.IdName;
  }

  getDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<Detail> = this.fb.group({
      id: fb.ctr(0, Validators.required),
      name: fb.ctr(null, Validators.required),
      subTypes: fb.ctr([])
    });

    return fg;
  }

  getDetailSubTypeForm() {
    const fb = this.fb;
    const fg: util.FormModel<util.IdName> = this.fb.group({
      id: fb.ctr(0, Validators.required),
      name: fb.ctr(null, Validators.required)
    });

    return fg;
  }

  //this updates localDetailLoading whenever detailLoading$ changes so that we can check the loading status in this class
  localDetailLoading = true;
  detailLoadingWatcher$ = this.detailLoading$.pipe(
    tap(isLoading => {
      this.localDetailLoading = isLoading;
    }),
    shareReplay(1)
  )

  title$ = of('Business Type').pipe(
    tap((title) => util.trySetTitle(this.titleService, title))
  )

  requiredData$ = this.refreshRequiredData$.pipe(
    tap(() => this.detailLoading$.next(true)),
    switchMap(refreshType => {
      return combineLatest(this.service.requiredData$, of(refreshType));
    }),
    map(([requiredData, refreshType]) => {
      this.localRequiredData = requiredData;
      if (refreshType === util.RefreshType.SelfOnly)
        this.detailLoading$.next(false);
      return requiredData;
    }),
    tap((requiredData) => {
      this.hasModifyPermission = requiredData.hasModifyPermission;
      util.focusInputTarget();
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  items$ = this.refreshItems$.pipe(
    tap(() => {
      this.gridLoading$.next(true);
    }),
    switchMap(() => {
      return this.service.getItems(this.state);
    }),
    tap(() => {
      this.gridLoading$.next(false);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  exportAction$ = this.exportClicked$.pipe(
    switchMap(() => {
      this.exporting$.next(true);
      return this.service.exportItems(this.state, 'Business Types.xlsx');
    }),
    tap(res => {
      saveAs(res.fileBlob, res.fileName);
      this.exporting$.next(false);
    }),
    shareReplay(1),
    catchError(err => {
      this.exporting$.next(false);
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  detail$ = this.refreshDetail$.pipe(
    filter(id => id !== null),
    switchMap(id => {
      this.detailLoading$.next(true);
      this.detailForm.disable();
      this.detailForm.reset();
      this.detailSubTypeForm.reset();
      if (id === 0)
        return of(this.detailInitialValues);
      else
        return this.service.getDetail(id);
    }),
    map(result => {
      const detail: Detail = result;
      if (detail) {
        util.convertToDates(detail);
        this.detailForm.setValue(detail);
      }
      return detail;
    }),
    tap(() => {
      this.detailFinishedLoading();
    }),
    shareReplay(1),
    catchError(err => {
      this.closeDetail();
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  saveResult$ = this.save$.pipe(
    switchMap(saveType => {
      this.detailLoading$.next(true);
      this.detailForm.disable();
      const itemToSave: Detail = this.detailForm.value as Detail;
      return this.service.saveDetail(itemToSave, saveType);
    }),
    tap((saveResult) => {
      this.notify.success('save successful');
      this.closeDetail();
      this.refreshItems$.next(null);
      this.mySelection = [saveResult];
    }),
    shareReplay(1),
    catchError(err => {
      this.detailFinishedLoading();
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  deleteResult$ = this.delete$.pipe(
    switchMap(() => {
      this.detailLoading$.next(true);
      this.detailForm.disable();
      const itemToDelete: Detail = this.detailForm.getRawValue();
      return this.service.deleteDetail(itemToDelete.id);
    }),
    tap(() => {
      this.notify.success('delete successful');
      this.detailFinishedLoading();
      this.closeDetail();
      this.refreshItems$.next(null);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  addSubTypeResult$ = this.addSubType$.pipe(
    switchMap(() => {
      const businessTypeId = (this.detailForm.value as Detail).id;
      const newSubTypeName = (this.detailSubTypeForm.value as util.IdName).name;
      return this.service.addNewSubType(businessTypeId, newSubTypeName);
    }),
    tap(() => {
      this.detailSubTypeOpened$.next(false);
      const id = (this.detailForm.value as Detail).id;
      this.refreshDetail$.next(id);
      this.notify.success('add successful');
      this.refreshItems$.next(null);
    }),
    shareReplay(1),
    catchError(err => {
      this.detailSubTypeForm.enable();
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  renameSubTypeResult$ = this.renameSubType$.pipe(
    switchMap(() => {
      const subTypeId = (this.detailSubTypeForm.value as util.IdName).id;
      const newSubTypeName = (this.detailSubTypeForm.value as util.IdName).name;
      return this.service.renameSubType(subTypeId, newSubTypeName);
    }),
    tap(() => {
      this.detailSubTypeOpened$.next(false);
      const id = (this.detailForm.value as Detail).id;
      this.refreshDetail$.next(id);
      this.notify.success('add successful');
      this.refreshItems$.next(null);
    }),
    shareReplay(1),
    catchError(err => {
      this.detailSubTypeForm.enable();
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  deleteSubTypeResult$ = this.deleteSubType$.pipe(
    switchMap(() => {
      const subTypeId = this.selectedSubType.id;
      return this.service.deleteSubType(subTypeId);
    }),
    tap(() => {
      const id = (this.detailForm.value as Detail).id;
      this.refreshDetail$.next(id);
      this.notify.success('delete successful');
      this.refreshItems$.next(null);
    }),
    shareReplay(1),
    catchError(err => {
      this.detailSubTypeForm.enable();
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  openDetail(id: number): void {
    this.refreshDetail$.next(id);
    this.detailOpened$.next(true);
  }

  onDetailClosing() {
    util.onDetailChanging(this.detailForm, this.dialogService, this.closeDetail, this.save);
  }

  onDetailSubTypeClosing() {
    util.onDetailChanging(this.detailSubTypeForm, this.dialogService, this.closeDetailSubType, this.saveSubType);
  }

  closeDetail = () => {
    this.detailOpened$.next(false);
  }

  closeDetailSubType = () => {
    this.detailSubTypeOpened$.next(false);
  }

  detailFinishedLoading(): void {
    this.detailForm.enable();
    this.detailLoading$.next(false);
    util.focusInputTarget();
  }

  add(): void {
    this.openDetail(0);
  }

  edit(dataItem: Item): void {
    this.openDetail(dataItem.Id);
  }

  save = (saveType: util.SaveType) => {
    this.detailForm.markAllAsTouched();
    if (this.detailForm.valid)
      this.save$.next(saveType);
    else
      this.notify.error("validation failed");
  }

  delete(): void {
    const deleteConfirmSettings: DialogSettings = {
      title: "Please confirm",
      content: "Are you sure you want to delete this item?",
      actions: [{ text: 'No' }, { text: 'Yes', cssClass: 'k-primary' }],
      cssClass: 'utilPrompt'
    }

    this.dialogService.open(deleteConfirmSettings).result.pipe(take(1)).subscribe(result => {
      if (util.getDialogAction(result) === util.dialogAction.Yes)
        this.delete$.next(null);
    });
  }

  export(): void {
    this.exportClicked$.next(null);
  }

  dataStateChange(state: State): void {
    this.gridScrollPosition = { topPos: 0, leftPos: 0 };
    util.fixKendoQueryFilter(state.filter);
    this.state = state;
    this.refreshItems$.next(null);
  }

  refreshDropdowns() {
    this.refreshRequiredData$.next(util.RefreshType.SelfOnly);
  }

  openAddSubType(): void {
    const businessTypeId = (this.detailForm.value as Detail).id;
    if (businessTypeId === 0)
      this.notify.info('The main business type must be saved first');
    else if (this.hasModifyPermission) {
      this.detailSubTypeForm.enable();
      this.selectedSubType = null;
      const newSubType: util.IdName = { id: 0, name: null };
      this.detailSubTypeForm.setValue(newSubType);
      this.detailSubTypeForm.markAsUntouched();
      this.detailSubTypeOpened$.next(true);
      util.focusInputTarget()
    }
  }

  openRenameSubType(): void {
    if (this.hasModifyPermission && this.selectedSubType) {
      this.detailSubTypeForm.enable();
      this.detailSubTypeForm.setValue(this.selectedSubType);
      this.detailSubTypeOpened$.next(true);
      util.focusAndSelectInputTarget();
    }
  }

  openDeleteSubType(): void {
    if (this.hasModifyPermission) {
      const deleteConfirmSettings: DialogSettings = {
        title: "Please confirm",
        content: "Are you sure you want to delete this sub-type?",
        actions: [{ text: 'No' }, { text: 'Yes', cssClass: 'k-primary' }],
        cssClass: 'utilPrompt'
      }

      this.dialogService.open(deleteConfirmSettings).result.pipe(take(1)).subscribe(result => {
        if (util.getDialogAction(result) === util.dialogAction.Yes) {
          this.detailForm.disable();
          this.detailLoading$.next(true);
          this.deleteSubType$.next(null);
        }
      });
    }
  }

  saveSubType = () => {
    if (this.detailSubTypeForm.valid) {
      this.detailSubTypeForm.disable();
      if (this.selectedSubType)
        this.renameSubType$.next(null);
      else
        this.addSubType$.next(null);
    } else {
      this.notify.error("validation failed");
    }
  }
}
