import { ChangeDetectionStrategy, Component, inject, input, output, ViewChild, ViewEncapsulation } from '@angular/core';
import * as util from '../_shared/utils/util';
import { BehaviorSubject, catchError, filter, map, of, retry, shareReplay, Subject, switchMap, tap } from 'rxjs';
import { MessageService } from '../_shared/services/message.service';
import { ApprovalDocsService, Detail } from './approval-docs.service';
import { Title } from '@angular/platform-browser';
import { DialogService } from '@progress/kendo-angular-dialog';
import { NotifyService } from '../_shared/services/notify.service';
import { Validators } from '@angular/forms';
import { SuccessEvent, ErrorEvent, UploadComponent, UploadProgressEvent } from '@progress/kendo-angular-upload';
import { FAST_KENDO_COMMON, FAST_PAGE_COMMON } from '../app.config';
import { CustomFormBuilder } from '../_shared/services/custom-form-builder.service';

@Component({
  selector: 'app-approval-docs',
  changeDetection: ChangeDetectionStrategy.OnPush,
  templateUrl: './approval-docs.component.html',
  encapsulation: ViewEncapsulation.None,
  imports: [FAST_KENDO_COMMON, FAST_PAGE_COMMON]
})
export class ApprovalDocsComponent {
  @ViewChild('docRateUploadTarget') docFileUploadElem: UploadComponent;
  util = util;
  icons = util.icons;
  detailForm: util.FormModel<Detail>;
  creditLimitId = input<number>(0);
  closed = output();
  docRateSaveUrl = `${window.location.origin}/api/ApprovalDocs/UploadDoc`;

  detailLoading$ = new BehaviorSubject<boolean>(false)
  refreshDetail$ = new BehaviorSubject<void>(null)
  save$ = new Subject<util.SaveType>()
  downloadDoc$ = new Subject<[util.DocItem]>()


  service = inject(ApprovalDocsService);
  fb = inject(CustomFormBuilder);
  messageService = inject(MessageService);
  titleService = inject(Title);
  dialogService = inject(DialogService);
  notify = inject(NotifyService);

  constructor() {
    this.detailForm = this.getDetailForm();
  }

  title$ = of('Approval Docs').pipe(
    tap((title) => util.trySetTitle(this.titleService, title))
  )

  detail$ = this.refreshDetail$.pipe(
    switchMap(() => {
      this.detailLoading$.next(true);
      this.detailForm.reset();
      return this.service.getDetail(this.creditLimitId());
    }),
    map(result => {
      const detail: Detail = result;
      if (detail) {
        util.convertToDates(detail);
        this.detailForm.setValue(detail);
      }
      return detail;
    }),
    tap(() => {
      this.detailLoading$.next(false);
    }),
    shareReplay(1),
    catchError(err => {
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  saveResult$ = this.save$.pipe(
    switchMap(saveType => {
      this.detailLoading$.next(true);
      const itemToSave: Detail = this.detailForm.value as Detail;
      return this.service.saveDetail(itemToSave, saveType);
    }),
    tap(() => {
      this.detailLoading$.next(false);
      this.notify.success('save successful');
      this.closed.emit();
    }),
    shareReplay(1),
    catchError(err => {
      this.detailLoading$.next(false);
      return util.handleError(err, this.messageService)
    }), retry(10)
  )

  downloadDocResult$ = this.downloadDoc$.pipe(
    filter(([docItem]) => {
      return docItem && docItem.fileNameOnDisk !== null;
    }),
    switchMap(([docItem]) => {
      return this.service.downloadDoc(docItem.fileNameOriginal, docItem.fileNameOnDisk);
    }),
    tap(res => {
      util.openOrSaveFile(res.fileBlob, res.fileName);
    }),
    catchError(err => {
      return util.handleError(err, this.messageService);
    }), retry(10)
  )

  getDetailForm() {
    const fb = this.fb;
    const fg: util.FormModel<Detail> = this.fb.group({
      creditLimitId: fb.ctr(361, Validators.required),
      documents: fb.ctr([]),
      notes: fb.ctr("")
    });

    return fg;
  }

  save = (saveType: util.SaveType) => {
    this.detailForm.markAllAsTouched();
    if (this.detailForm.valid)
      this.save$.next(saveType);
    else
      this.notify.error("validation failed");
  }

  onDetailClosing() {
    util.onDetailChanging(this.detailForm, this.dialogService, this.closedClicked, this.save);
  }

  closedClicked = () => {
    this.closed.emit();
  }

  docFileUploadSuccess(value: SuccessEvent) {
    const newdocitems: util.DocItem[] = value.response.body;
    let docs: util.DocItem[] = this.detailForm.get('documents').value;
    if (docs)
      docs = docs.concat(newdocitems);
    else
      docs = newdocitems;
    this.detailForm.patchValue({ documents: docs });
    this.detailForm.markAsDirty();
  }

  downloadFileDoc(doc: util.DocItem) {
    this.downloadDoc$.next([doc]);
  }

  removeFileDoc(doc: util.DocItem) {
    const docs: util.DocItem[] = this.detailForm.get('documents').getRawValue();
    const indexToRemove = docs.findIndex(x => x.fileNameOnDisk === doc.fileNameOnDisk);
    docs.splice(indexToRemove, 1);
    this.detailForm.patchValue({ documents: docs });
    this.detailForm.markAsDirty();
  }

  docComplete() {
    if (this.docFileUploadElem)
      this.docFileUploadElem.clearFiles();
  }

  // eslint-disable-next-line @typescript-eslint/no-unused-vars
  uploadProgress(event: UploadProgressEvent) {
    //the kendo-upload uploadProgress must be registered for percentages to update
    //but we don't need to do anything with it
  }

  docUploadError(value: ErrorEvent) {
    util.handleError(value.response, this.messageService);
  }
}
