codeunit 50150 "Superior Invoice API Test"
{
    // This codeunit provides test functionality for the Superior Invoice API.
    // Use it to test with fake data and set breakpoints during debugging.
    //
    // HOW TO DEBUG:
    // 1. Set breakpoints in this codeunit or in "Superior Invoice API" codeunit
    // 2. Press F5 to start debugging (AL: Download symbols + F5)
    // 3. Navigate to any page, then use Tell Me (Alt+Q) and search for:
    //    "Run Superior Invoice API Test"
    // 4. Click the action to execute the test with sample data

    Permissions =
        tabledata "Superior Invoice Request" = RIMD,
        tabledata "Superior Invoice Request Line" = RIMD,
        tabledata "Sales Header" = RIMD,
        tabledata "Sales Line" = RIMD,
        tabledata "Sales Invoice Header" = RIMD,
        tabledata "Sales Invoice Line" = RIMD,
        tabledata "Sales Shipment Header" = RIMD,
        tabledata "Sales Shipment Line" = RIMD,
        tabledata "Reservation Entry" = RIMD,
        tabledata "G/L Entry" = RIMD,
        tabledata "Cust. Ledger Entry" = RIMD,
        tabledata "Detailed Cust. Ledg. Entry" = RIMD,
        tabledata "Value Entry" = RIMD,
        tabledata "Item Ledger Entry" = RIMD,
        tabledata "VAT Entry" = RIMD;

    /// Main test procedure - creates a complete invoice with sample data.
    procedure RunFullTest()
    var
        InvoiceRequest: Record "Superior Invoice Request";
        InvoiceRequestLine: Record "Superior Invoice Request Line";
        SuperiorInvoiceAPI: Codeunit "Superior Invoice API";
        ResultDocNo: Code[20];
        ErrorText: Text;
        Success: Boolean;
        TestInvoiceNo: Code[20];
    begin
        // Generate a unique invoice number to avoid duplicate errors
        TestInvoiceNo := 'TEST-' + Format(CurrentDateTime(), 0, '<Year4><Month,2><Day,2><Hours24,2><Minutes,2><Seconds,2>');

        Message('Starting Superior Invoice API Test\Invoice No: %1', TestInvoiceNo);

        // STEP 1: Create the request header
        // Set a breakpoint here to inspect the header creation
        CreateTestHeader(InvoiceRequest, TestInvoiceNo);

        // STEP 2: Create the request lines
        // Set a breakpoint here to inspect line creation
        CreateTestLines(InvoiceRequest."ID", InvoiceRequestLine);

        // STEP 3: Call the API
        // Set a breakpoint here to step into the API codeunit
        InvoiceRequestLine.SetRange("Document ID", InvoiceRequest."ID");
        Success := SuperiorInvoiceAPI.TryCreateInvoiceFromRecords(
            InvoiceRequest,
            InvoiceRequestLine,
            ResultDocNo,
            ErrorText
        );

        // STEP 4: Display results
        if Success then
            Message('SUCCESS!\\Sales Invoice created: %1\\You can find it in:\Sales Invoices (Page 9301)', ResultDocNo)
        else
            Message('FAILED!\\Error: %1', ErrorText);
    end;

    /// Test that intentionally triggers a duplicate error.
    /// Use this to verify error handling works correctly.
    procedure RunDuplicateTest()
    var
        InvoiceRequest: Record "Superior Invoice Request";
        InvoiceRequestLine: Record "Superior Invoice Request Line";
        SuperiorInvoiceAPI: Codeunit "Superior Invoice API";
        ResultDocNo: Code[20];
        ErrorText: Text;
        Success: Boolean;
        TestInvoiceNo: Code[20];
    begin
        // Use a fixed number that likely already exists to test duplicate detection
        TestInvoiceNo := 'DUP-TEST-001';

        Message('Running Duplicate Test\Will try to create invoice: %1 twice', TestInvoiceNo);

        // First attempt - should succeed if it doesn't exist
        CreateTestHeader(InvoiceRequest, TestInvoiceNo);
        CreateTestLines(InvoiceRequest."ID", InvoiceRequestLine);

        InvoiceRequestLine.SetRange("Document ID", InvoiceRequest."ID");
        Success := SuperiorInvoiceAPI.TryCreateInvoiceFromRecords(
            InvoiceRequest,
            InvoiceRequestLine,
            ResultDocNo,
            ErrorText
        );

        if not Success then begin
            Message('First attempt already failed (invoice may already exist):\%1', ErrorText);
            exit;
        end;

        // Clean up first request record (we'll create a new one)
        InvoiceRequest.Delete(true);

        // Second attempt - should fail with duplicate error
        CreateTestHeader(InvoiceRequest, TestInvoiceNo);
        CreateTestLines(InvoiceRequest."ID", InvoiceRequestLine);

        InvoiceRequestLine.SetRange("Document ID", InvoiceRequest."ID");
        Success := SuperiorInvoiceAPI.TryCreateInvoiceFromRecords(
            InvoiceRequest,
            InvoiceRequestLine,
            ResultDocNo,
            ErrorText
        );

        if Success then
            Message('UNEXPECTED: Second attempt succeeded (duplicate check may not be working)')
        else
            Message('EXPECTED BEHAVIOR:\Duplicate detection caught the error:\%1', ErrorText);
    end;

    /// Test with minimal data to verify basic flow.
    procedure RunMinimalTest()
    var
        InvoiceRequest: Record "Superior Invoice Request";
        InvoiceRequestLine: Record "Superior Invoice Request Line";
        SuperiorInvoiceAPI: Codeunit "Superior Invoice API";
        ResultDocNo: Code[20];
        ErrorText: Text;
        Success: Boolean;
        TestInvoiceNo: Code[20];
    begin
        TestInvoiceNo := 'MIN-' + Format(CurrentDateTime(), 0, '<Year4><Month,2><Day,2><Hours24,2><Minutes,2>');

        // Create minimal header
        InvoiceRequest.Init();
        InvoiceRequest."ID" := CreateGuid();
        InvoiceRequest."Customer Num" := '71763'; //CANTIUM, LLC
        InvoiceRequest."Number" := TestInvoiceNo;
        InvoiceRequest."Posting Date" := Today();
        InvoiceRequest."Document Date" := Today();
        InvoiceRequest.Insert(true);

        // Create single minimal line
        InvoiceRequestLine.Init();
        InvoiceRequestLine."Document ID" := InvoiceRequest."ID";
        InvoiceRequestLine."Line No." := 10000;
        InvoiceRequestLine."Line Type" := 'Item';
        InvoiceRequestLine."Line Type Number" := 'NATGAS';
        InvoiceRequestLine."Quantity" := 1;
        InvoiceRequestLine."Unit Price" := 100;
        InvoiceRequestLine.Insert(true);

        InvoiceRequestLine.SetRange("Document ID", InvoiceRequest."ID");
        Success := SuperiorInvoiceAPI.TryCreateInvoiceFromRecords(
            InvoiceRequest,
            InvoiceRequestLine,
            ResultDocNo,
            ErrorText
        );

        if Success then
            Message('Minimal test SUCCESS!\Created: %1', ResultDocNo)
        else
            Message('Minimal test FAILED!\%1', ErrorText);
    end;

    /// Creates a test header record with sample data.
    local procedure CreateTestHeader(var InvoiceRequest: Record "Superior Invoice Request"; var TestInvoiceNo: Code[20])
    begin
        InvoiceRequest.Init();
        InvoiceRequest."ID" := CreateGuid();
        InvoiceRequest."Customer Num" := '71547'; // MC...
        InvoiceRequest."Number" := TestInvoiceNo;
        InvoiceRequest."External Document Number" := 'EXT-' + TestInvoiceNo;
        InvoiceRequest."Document Date" := Today();
        InvoiceRequest."Posting Date" := Today();
        InvoiceRequest."Due Date" := CalcDate('<+30D>', Today());
        InvoiceRequest.Insert(true);

        // Breakpoint checkpoint: inspect the created header
        // InvoiceRequest now contains all the header data
    end;

    /// Creates test line records with sample data including dimensions.
    local procedure CreateTestLines(DocumentID: Guid; var InvoiceRequestLine: Record "Superior Invoice Request Line")
    var
        LineNo: Integer;
    begin
        LineNo := 0;

        // Line 1: G/L Account with all dimensions
        LineNo += 10000;
        CreateTestLine(
            InvoiceRequestLine,
            DocumentID,
            LineNo,
            'Item',
            'NATGAS',
            1000.00,
            1,
            'NONTAXABLE',
            Format(Today(), 0, '<Year4>-<Month,2>'),   // PROD MTH
            'PIPE-HPGG-220006',                        // METER
            'STATE-LA-OFF FEDERAL',                    // STATE LOC
            'SD-GAS163925-D27',                        // SDEAL#
            'PD-TESTDEAL-D01',                         // PDEAL#
            'NJD-HPGG-G-0013'                          // PIPE
        );

        // Line 2: G/L Account with partial dimensions
        LineNo += 10000;
        CreateTestLine(
            InvoiceRequestLine,
            DocumentID,
            LineNo,
            'Item',
            'NATGAS',
            500.00,
            2,
            'NONTAXABLE',
            Format(Today(), 0, '<Year4>-<Month,2>'),   // PROD MTH
            'XSDKBHJFGHJSDF',                        // METER
            'STATE-LA-OFF FEDERAL',                    // STATE LOC
            'SD-GAS163925-D27',                        // SDEAL#
            'PD-TESTDEAL-D01',                         // PDEAL#
            'NJD-HPGG-G-0013'                          // PIPE
        );
    end;

    local procedure CreateTestLine(var InvoiceRequestLine: Record "Superior Invoice Request Line"; DocumentID: Guid; LineNo: Integer; LineType: Text[30]; LineTypeNo: Code[20]; UnitPrice: Decimal; Qty: Decimal; TaxCode: Code[20]; ProdMonth: Text[50]; MeterCode: Text[50]; StateLocation: Text[50]; SdealNumber: Text[50]; PdealNumber: Text[50]; PipeCode: Text[50])
    begin
        InvoiceRequestLine.Init();
        InvoiceRequestLine."Document ID" := DocumentID;
        InvoiceRequestLine."Line No." := LineNo;
        InvoiceRequestLine."Line Type" := LineType;
        InvoiceRequestLine."Line Type Number" := LineTypeNo;
        InvoiceRequestLine."Unit Price" := UnitPrice;
        InvoiceRequestLine."Quantity" := Qty;
        InvoiceRequestLine."Tax Code" := TaxCode;
        InvoiceRequestLine."Prod Month" := ProdMonth;
        InvoiceRequestLine."Meter Code" := MeterCode;
        InvoiceRequestLine."State Location" := StateLocation;
        InvoiceRequestLine."Sdeal Number" := SdealNumber;
        InvoiceRequestLine."Pdeal Number" := PdealNumber;
        InvoiceRequestLine."Pipe Code" := PipeCode;
        InvoiceRequestLine.Insert(true);
    end;

    /// Test that simulates an API deep insert by parsing JSON and inserting records.
    /// This mirrors exactly what happens when a client POSTs JSON to the API.
    /// Useful to copy/paste the JSON from the C# app and debug it here.
    procedure RunJsonTest()
    var
        InvoiceRequest: Record "Superior Invoice Request";
        InvoiceRequestLine: Record "Superior Invoice Request Line";
        SuperiorInvoiceAPI: Codeunit "Superior Invoice API";
        JsonObject: JsonObject;
        JsonToken: JsonToken;
        LinesArray: JsonArray;
        LineToken: JsonToken;
        LineObject: JsonObject;
        ResultDocNo: Code[20];
        ErrorText: Text;
        Success: Boolean;
        TestInvoiceNo: Code[20];
        JsonString: Text;
        DocumentID: Guid;
        LineNo: Integer;
        i: Integer;
    begin
        // Generate unique invoice number
        // Generate unique invoice number
        // Reverted to static "TEST-75" to verify Overwrite/Cleanup logic.
        TestInvoiceNo := 'TEST-75';

        // Example JSON payload
        // copy/paste into JsonString from the C# app and then debug from here
        JsonString := '{"number":"TEST-75","externalDocumentNumber":"","customerNum":"70086 ","documentDate":"2025-12-09","postingDate":"2025-11-30","dueDate":"2025-12-25","invoiceLines":[{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D01","sdealNumber":"SD-GAS166647-D01","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D02","sdealNumber":"SD-GAS166647-D02","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D03","sdealNumber":"SD-GAS166647-D03","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D04","sdealNumber":"SD-GAS166647-D04","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D05","sdealNumber":"SD-GAS166647-D05","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D06","sdealNumber":"SD-GAS166647-D06","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D07","sdealNumber":"SD-GAS166647-D07","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D08","sdealNumber":"SD-GAS166647-D08","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D09","sdealNumber":"SD-GAS166647-D09","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D10","sdealNumber":"SD-GAS166647-D10","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D11","sdealNumber":"SD-GAS166647-D11","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D12","sdealNumber":"SD-GAS166647-D12","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D13","sdealNumber":"SD-GAS166647-D13","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D14","sdealNumber":"SD-GAS166647-D14","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D15","sdealNumber":"SD-GAS166647-D15","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D16","sdealNumber":"SD-GAS166647-D16","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D17","sdealNumber":"SD-GAS166647-D17","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D18","sdealNumber":"SD-GAS166647-D18","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D19","sdealNumber":"SD-GAS166647-D19","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D20","sdealNumber":"SD-GAS166647-D20","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D21","sdealNumber":"SD-GAS166647-D21","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D22","sdealNumber":"SD-GAS166647-D22","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D23","sdealNumber":"SD-GAS166647-D23","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D24","sdealNumber":"SD-GAS166647-D24","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D25","sdealNumber":"SD-GAS166647-D25","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D26","sdealNumber":"SD-GAS166647-D26","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D27","sdealNumber":"SD-GAS166647-D27","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D28","sdealNumber":"SD-GAS166647-D28","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D29","sdealNumber":"SD-GAS166647-D29","pipeCode":"INTER-TGP-80032"},{"lineType":"Item","lineTypeNumber":"NATGAS","unitPrice":3.17,"taxCode":"NONTAXABLE","quantity":5000,"prodMonth":"2025-11","meterCode":"PIPE-TGP-405345","stateLocation":"STATE-TX-ONSHORE","pdealNumber":"PD-GAS166642-D30","sdealNumber":"SD-GAS166647-D30","pipeCode":"INTER-TGP-80032"}]}';

        // Note: Format(Date, 0, 9) produces XML format (YYYY-MM-DD) which ReadFrom understands

        Message('Starting JSON Deep Insert Test\Parsing JSON payload for invoice: %1', TestInvoiceNo);

        // Parse the JSON
        if not JsonObject.ReadFrom(JsonString) then
            Error('Failed to parse JSON payload');

        // Create the header record from JSON (simulating what the API page does)
        DocumentID := CreateGuid();
        InvoiceRequest.Init();
        InvoiceRequest."ID" := DocumentID;

        if JsonObject.Get('customerNum', JsonToken) then
            InvoiceRequest."Customer Num" := CopyStr(JsonToken.AsValue().AsText(), 1, MaxStrLen(InvoiceRequest."Customer Num"));

        if JsonObject.Get('number', JsonToken) then
            InvoiceRequest."Number" := CopyStr(JsonToken.AsValue().AsText(), 1, MaxStrLen(InvoiceRequest."Number"));

        if JsonObject.Get('externalDocumentNumber', JsonToken) then
            InvoiceRequest."External Document Number" := CopyStr(JsonToken.AsValue().AsText(), 1, MaxStrLen(InvoiceRequest."External Document Number"));

        if JsonObject.Get('documentDate', JsonToken) then
            Evaluate(InvoiceRequest."Document Date", JsonToken.AsValue().AsText());

        if JsonObject.Get('postingDate', JsonToken) then
            Evaluate(InvoiceRequest."Posting Date", JsonToken.AsValue().AsText());

        if JsonObject.Get('dueDate', JsonToken) then
            Evaluate(InvoiceRequest."Due Date", JsonToken.AsValue().AsText());

        InvoiceRequest.Insert(true);

        // Parse and insert lines (simulating deep insert)
        if JsonObject.Get('invoiceLines', JsonToken) then begin
            LinesArray := JsonToken.AsArray();
            LineNo := 0;

            for i := 0 to LinesArray.Count() - 1 do begin
                LinesArray.Get(i, LineToken);
                LineObject := LineToken.AsObject();
                LineNo += 10000;

                InvoiceRequestLine.Init();
                InvoiceRequestLine."Document ID" := DocumentID;
                InvoiceRequestLine."Line No." := LineNo;

                if LineObject.Get('lineType', JsonToken) then
                    InvoiceRequestLine."Line Type" := CopyStr(JsonToken.AsValue().AsText(), 1, MaxStrLen(InvoiceRequestLine."Line Type"));

                if LineObject.Get('lineTypeNumber', JsonToken) then
                    InvoiceRequestLine."Line Type Number" := CopyStr(JsonToken.AsValue().AsText(), 1, MaxStrLen(InvoiceRequestLine."Line Type Number"));

                if LineObject.Get('unitPrice', JsonToken) then
                    InvoiceRequestLine."Unit Price" := JsonToken.AsValue().AsDecimal();

                if LineObject.Get('quantity', JsonToken) then
                    InvoiceRequestLine."Quantity" := JsonToken.AsValue().AsDecimal();

                if LineObject.Get('taxCode', JsonToken) then
                    InvoiceRequestLine."Tax Code" := CopyStr(JsonToken.AsValue().AsText(), 1, MaxStrLen(InvoiceRequestLine."Tax Code"));

                if LineObject.Get('prodMonth', JsonToken) then
                    InvoiceRequestLine."Prod Month" := CopyStr(JsonToken.AsValue().AsText(), 1, MaxStrLen(InvoiceRequestLine."Prod Month"));

                if LineObject.Get('meterCode', JsonToken) then
                    InvoiceRequestLine."Meter Code" := CopyStr(JsonToken.AsValue().AsText(), 1, MaxStrLen(InvoiceRequestLine."Meter Code"));

                if LineObject.Get('stateLocation', JsonToken) then
                    if not JsonToken.AsValue().IsNull() then
                        InvoiceRequestLine."State Location" := CopyStr(JsonToken.AsValue().AsText(), 1, MaxStrLen(InvoiceRequestLine."State Location"));

                if LineObject.Get('sdealNumber', JsonToken) then
                    InvoiceRequestLine."Sdeal Number" := CopyStr(JsonToken.AsValue().AsText(), 1, MaxStrLen(InvoiceRequestLine."Sdeal Number"));

                if LineObject.Get('pdealNumber', JsonToken) then
                    InvoiceRequestLine."Pdeal Number" := CopyStr(JsonToken.AsValue().AsText(), 1, MaxStrLen(InvoiceRequestLine."Pdeal Number"));

                if LineObject.Get('pipeCode', JsonToken) then
                    InvoiceRequestLine."Pipe Code" := CopyStr(JsonToken.AsValue().AsText(), 1, MaxStrLen(InvoiceRequestLine."Pipe Code"));

                InvoiceRequestLine.Insert(true);
            end;
        end;

        // Now call the API exactly as the API page would
        InvoiceRequestLine.SetRange("Document ID", DocumentID);
        Success := SuperiorInvoiceAPI.TryCreateInvoiceFromRecords(
            InvoiceRequest,
            InvoiceRequestLine,
            ResultDocNo,
            ErrorText
        );

        if Success then
            Message('JSON Test SUCCESS!\\Parsed JSON and created invoice: %1\\Lines created: %2', ResultDocNo, LineNo / 10000)
        else
            Message('JSON Test FAILED!\\Error: %1', ErrorText);
    end;

    procedure CleanupTestData()
    var
        SalesHeader: Record "Sales Header";
        SalesLine: Record "Sales Line";
        SalesInvoiceHeader: Record "Sales Invoice Header";
        SalesInvoiceLine: Record "Sales Invoice Line";
        SalesShipmentHeader: Record "Sales Shipment Header";
        SalesShipmentLine: Record "Sales Shipment Line";
        InvoiceRequest: Record "Superior Invoice Request";
        GLEntry: Record "G/L Entry";
        CustLedgEntry: Record "Cust. Ledger Entry";
        DetCustLedgEntry: Record "Detailed Cust. Ledg. Entry";
        ValueEntry: Record "Value Entry";
        ItemLedgEntry: Record "Item Ledger Entry";
        VATEntry: Record "VAT Entry";
        ReservEntry: Record "Reservation Entry";
        DeletedCount: Integer;
        TestFilter: Label 'TEST-*|MIN-*|DUP-*|JSON-*';
    begin
        // Delete unposted sales invoices that match TestFilter
        // First delete the lines, then the headers (skip triggers to avoid shipment issues)
        SalesHeader.SetRange("Document Type", SalesHeader."Document Type"::Invoice);
        SalesHeader.SetFilter("No.", TestFilter);
        DeletedCount := SalesHeader.Count();
        if DeletedCount > 0 then begin
            if SalesHeader.FindSet() then
                repeat
                    // Delete related sales lines first
                    SalesLine.SetRange("Document Type", SalesHeader."Document Type");
                    SalesLine.SetRange("Document No.", SalesHeader."No.");
                    SalesLine.DeleteAll(false);

                    // CLEANUP ORPHANED RESERVATIONS (caused by skipping triggers)
                    ReservEntry.SetRange("Source Type", 37); // Sales Line
                    ReservEntry.SetRange("Source ID", SalesHeader."No.");
                    ReservEntry.DeleteAll(false);
                until SalesHeader.Next() = 0;
            // Now delete headers without triggers to avoid shipment creation
            SalesHeader.DeleteAll(false);
        end;

        // Delete sales shipment headers that match TestFilter
        SalesShipmentHeader.SetFilter("Order No.", TestFilter);
        DeletedCount += SalesShipmentHeader.Count();
        if not SalesShipmentHeader.IsEmpty() then begin
            if SalesShipmentHeader.FindSet() then
                repeat
                    SalesShipmentLine.SetRange("Document No.", SalesShipmentHeader."No.");
                    SalesShipmentLine.DeleteAll(false);
                until SalesShipmentHeader.Next() = 0;
            SalesShipmentHeader.DeleteAll(false);
        end;

        // Delete posted sales invoices that match TestFilter
        SalesInvoiceHeader.SetFilter("Pre-Assigned No.", TestFilter);
        DeletedCount += SalesInvoiceHeader.Count();
        if not SalesInvoiceHeader.IsEmpty() then begin
            if SalesInvoiceHeader.FindSet() then
                repeat
                    SalesInvoiceLine.SetRange("Document No.", SalesInvoiceHeader."No.");
                    SalesInvoiceLine.DeleteAll(false);
                until SalesInvoiceHeader.Next() = 0;
            SalesInvoiceHeader.DeleteAll(false); // Skip triggers to avoid validation errors
        end;

        // Delete request records that match TestFilter
        InvoiceRequest.SetFilter("Number", TestFilter);
        DeletedCount += InvoiceRequest.Count();
        if not InvoiceRequest.IsEmpty() then
            InvoiceRequest.DeleteAll(true);

        // SCORCHED EARTH: Delete Ledger Entries to allow reuse of the same Posted Invoice No.
        GLEntry.SetFilter("Document No.", TestFilter);
        if not GLEntry.IsEmpty() then GLEntry.DeleteAll(false);

        CustLedgEntry.SetFilter("Document No.", TestFilter);
        if not CustLedgEntry.IsEmpty() then CustLedgEntry.DeleteAll(false);

        DetCustLedgEntry.SetFilter("Document No.", TestFilter);
        if not DetCustLedgEntry.IsEmpty() then DetCustLedgEntry.DeleteAll(false);

        ValueEntry.SetFilter("Document No.", TestFilter);
        if not ValueEntry.IsEmpty() then ValueEntry.DeleteAll(false);

        ItemLedgEntry.SetFilter("Document No.", TestFilter);
        if not ItemLedgEntry.IsEmpty() then ItemLedgEntry.DeleteAll(false);

        VATEntry.SetFilter("Document No.", TestFilter);
        if not VATEntry.IsEmpty() then VATEntry.DeleteAll(false);

        Message('Cleaned up %1 test records + Ledger Entries.', DeletedCount);
    end;
}
