page 50148 "Superior Invoice API"
{
    PageType = API;
    APIPublisher = 'SuperiorNaturalGasCorporation';
    APIGroup = 'sales';
    APIVersion = 'v2.0';
    EntityName = 'superiorInvoiceRequest';
    EntitySetName = 'superiorInvoiceRequests';
    SourceTable = "Superior Invoice Request";
    DelayedInsert = true;
    InsertAllowed = true;
    ModifyAllowed = false;
    DeleteAllowed = false;
    ODataKeyFields = "ID";
    Permissions =
        tabledata "Superior Invoice Request" = RIMD,
        tabledata "Superior Invoice Request Line" = RIMD;

    layout
    {
        area(Content)
        {
            repeater(Group)
            {
                field(id; Rec."ID")
                {
                    Caption = 'ID';
                    Editable = false;
                }
                // Input Fields
                field(customerNum; Rec."Customer Num")
                {
                    Caption = 'Customer Num';
                }
                field(number; Rec."Number")
                {
                    Caption = 'Number';
                }
                field(externalDocumentNumber; Rec."External Document Number")
                {
                    Caption = 'External Document Number';
                }
                field(documentDate; Rec."Document Date")
                {
                    Caption = 'Document Date';
                }
                field(postingDate; Rec."Posting Date")
                {
                    Caption = 'Posting Date';
                }
                field(dueDate; Rec."Due Date")
                {
                    Caption = 'Due Date';
                }
                // Output Fields
                field(resultDocumentNo; Rec."Result Document No.")
                {
                    Caption = 'Result Document No.';
                    Editable = false;
                }
                field(status; Rec."Status")
                {
                    Caption = 'Status';
                    Editable = false;
                }
                field(errorMessage; Rec."Error Message")
                {
                    Caption = 'Error Message';
                    Editable = false;
                }
            }
            // CRITICAL: Part control MUST be OUTSIDE the repeater for deep inserts
            part(invoiceLines; "Superior Invoice API Lines")
            {
                Caption = 'Invoice Lines';
                EntityName = 'invoiceLine';
                EntitySetName = 'invoiceLines';
                SubPageLink = "Document ID" = field("ID");
            }
        }
    }

    trigger OnInsertRecord(BelowxRec: Boolean): Boolean
    var
        Customer: Record "Customer";
    begin
        // Validate required fields
        if Rec."Customer Num" = '' then
            Error('Customer Num is required.');
        if Rec."Number" = '' then
            Error('Number is required.');

        // Validate customer exists (throws HTTP error that C# can catch)
        if not Customer.Get(Rec."Customer Num") then
            Error('Customer ''%1'' does not exist.', Rec."Customer Num");

        // Insert the header record (lines will be inserted after this trigger)
        Rec.Insert(true);

        exit(false); // We already inserted
    end;

    trigger OnAfterGetRecord()
    var
        SuperiorInvoiceAPICU: Codeunit "Superior Invoice API";
        InvoiceRequestLine: Record "Superior Invoice Request Line";
        ResultDocNo: Code[20];
        ErrorText: Text;
    begin
        // Only process newly inserted, unprocessed records
        // OnAfterGetRecord fires AFTER all deep-inserted lines are committed
        if Rec."Processed" then
            exit;

        // Get lines that were inserted via deep insert (now available!)
        InvoiceRequestLine.SetRange("Document ID", Rec."ID");

        // Try to create the Sales Invoice
        if SuperiorInvoiceAPICU.TryCreateInvoiceFromRecords(Rec, InvoiceRequestLine, ResultDocNo, ErrorText) then begin
            Rec."Result Document No." := ResultDocNo;
            Rec."Status" := Rec."Status"::Success;
            Rec."Error Message" := '';
        end else begin
            Rec."Result Document No." := '';
            Rec."Status" := Rec."Status"::Failed;
            Rec."Error Message" := CopyStr(ErrorText, 1, MaxStrLen(Rec."Error Message"));
            SuperiorInvoiceAPICU.RemoveInvoiceOnError(Rec.Number);
            Commit(); // Commit the cleanup before raising error
            Error(ErrorText);
        end;

        // Mark as processed and update
        Rec."Processed" := true;
        Rec.Modify(true);

        // Clean up old processed records (older than 7 days)
        CleanupOldRecords();
    end;

    local procedure CleanupOldRecords()
    var
        InvoiceRequest: Record "Superior Invoice Request";
        CutoffDate: DateTime;
    begin
        // Delete records older than 7 days that have been processed
        CutoffDate := CreateDateTime(CalcDate('<-7D>', Today()), 0T);

        InvoiceRequest.SetRange("Processed", true);
        InvoiceRequest.SetFilter("Created At", '<%1', CutoffDate);

        if not InvoiceRequest.IsEmpty() then
            InvoiceRequest.DeleteAll(true); // Cascade deletes lines via OnDelete trigger
    end;
}
