codeunit 50149 "Superior Invoice API"
{
    Permissions =
        tabledata "Superior Invoice Request" = RIMD,
        tabledata "Superior Invoice Request Line" = RIMD,
        tabledata "Sales Header" = RIMD,
        tabledata "Sales Line" = RIMD,
        tabledata "Sales Invoice Header" = RIMD,
        tabledata "Sales Invoice Line" = RIMD,
        tabledata "Dimension Value" = RIMD,
        tabledata "Dimension Set Entry" = RIMD,
        tabledata "G/L Entry" = RIMD,
        tabledata "Cust. Ledger Entry" = RIMD,
        tabledata "Detailed Cust. Ledg. Entry" = RIMD,
        tabledata "Value Entry" = RIMD,
        tabledata "Item Ledger Entry" = RIMD,
        tabledata "Sales Shipment Header" = RIMD,
        tabledata "Sales Shipment Line" = RIMD,
        tabledata "Reservation Entry" = RIMD,
        tabledata "VAT Entry" = RIMD;

    /// <summary>
    /// Creates an invoice from typed record structures (deep insert pattern).
    /// Returns true on success, false on failure with error details.
    /// </summary>
    procedure TryCreateInvoiceFromRecords(InvoiceRequest: Record "Superior Invoice Request"; var InvoiceRequestLine: Record "Superior Invoice Request Line"; var ResultDocNo: Code[20]; var ErrorText: Text): Boolean
    var
        CapturedError: Text;
    begin
        ClearLastError();
        ErrorText := '';
        ResultDocNo := '';
        CapturedError := '';

        if not TryCreateInvoiceFromRecordsInternal(InvoiceRequest, InvoiceRequestLine, CapturedError) then begin
            ErrorText := GetLastErrorText();
            exit(false);
        end;

        if (LastCreatedDocNo = '') and (CapturedError <> '') then begin
            Commit(); // Commit the cleanup (Delete) before returning the error which might cause a rollback
            ErrorText := CapturedError;
            exit(false);
        end;

        if LastCreatedDocNo = '' then begin
            ErrorText := 'Unknown error occurred: No document number returned.';
            exit(false);
        end;

        ResultDocNo := LastCreatedDocNo;
        exit(true);
    end;

    [TryFunction]
    local procedure TryCreateInvoiceFromRecordsInternal(InvoiceRequest: Record "Superior Invoice Request"; var InvoiceRequestLine: Record "Superior Invoice Request Line"; var OutError: Text)
    begin
        LastCreatedDocNo := CreateInvoiceFromRecords(InvoiceRequest, InvoiceRequestLine, OutError);
    end;

    var
        LastCreatedDocNo: Code[20];

    /// <summary>
    /// Creates a Sales Invoice from typed request records.
    /// </summary>
    procedure CreateInvoiceFromRecords(InvoiceRequest: Record "Superior Invoice Request"; var InvoiceRequestLine: Record "Superior Invoice Request Line"; var OutError: Text): Code[20]
    var
        SalesHeader: Record "Sales Header";
        LineNo: Integer;
    begin
        // 1. Customer Validation
        CheckCustomerExists(InvoiceRequest."Customer Num");

        // 2. Delete unposted invoice if exists
        DeleteUnpostedInvoiceIfExists(InvoiceRequest."Number");
        Commit(); // Persist the deletion of the old invoice so we start with a clean slate even if creation fails


        // 3. Duplicate Check
        if IsDuplicatePostedInvoice(InvoiceRequest."Number") then
            exit('DUPLICATE INVOICE');

        // 4. Create Header
        CreateSalesHeader(
            SalesHeader,
            InvoiceRequest."Customer Num",
            InvoiceRequest."Number",
            InvoiceRequest."External Document Number",
            InvoiceRequest."Document Date",
            InvoiceRequest."Posting Date",
            InvoiceRequest."Due Date"
        );

        // 5. Process Lines
        LineNo := 0;
        InvoiceRequestLine.SetRange("Document ID", InvoiceRequest."ID");
        if InvoiceRequestLine.FindSet() then
            repeat
                LineNo += 10000;
                CreateSalesLineFromRecord(SalesHeader, InvoiceRequestLine, LineNo);
            until InvoiceRequestLine.Next() = 0;

        // 6. Check Accounting Period
        if not CheckAccountingPeriod(SalesHeader) then begin
            RemoveInvoiceOnError(SalesHeader."No.");
            OutError := StrSubstNo('Posting Date %1 is in a closed or date-locked accounting period.', SalesHeader."Posting Date");
            exit('');
        end;

        // 7. Post Invoice once created / processed
        Commit(); // COMMIT CREATION TRANSACTION: Save Unposted Invoice state.
                  // If Posting fails (Consistency Error), it rolls back to HERE.

        if not TryPostSalesInvoice(SalesHeader) then begin
            OutError := GetLastErrorText();
            RemoveInvoiceOnError(SalesHeader."No.");
            Commit(); // COMMIT CLEANUP: Persist deletion immediately.
            exit('');
        end;

        exit(SalesHeader."No.");
    end;

    local procedure CheckCustomerExists(CustNo: Code[20])
    var
        Customer: Record "Customer";
    begin
        if CustNo = '' then
            Error('Customer Num is required.');
        if not Customer.Get(CustNo) then
            Error('Customer ''%1'' does not exist.', CustNo);
    end;

    local procedure IsDuplicatePostedInvoice(DocNo: Code[20]): Boolean
    var
        PostedSalesHeader: Record "Sales Invoice Header";
    begin
        if DocNo = '' then exit(true);

        // Check if posted invoice exists with this number directly
        if PostedSalesHeader.Get(DocNo) then
            exit(true);

        // Also check Pre-Assigned No. (original unposted invoice number)
        PostedSalesHeader.Reset();
        PostedSalesHeader.SetRange("Pre-Assigned No.", DocNo);
        if not PostedSalesHeader.IsEmpty() then
            exit(true);

        exit(false);
    end;

    local procedure DeleteUnpostedInvoiceIfExists(DocNo: Code[20])
    var
        SalesHeader: Record "Sales Header";
        SalesLine: Record "Sales Line";
        SalesShipmentHeader: Record "Sales Shipment Header";
        SalesShipmentLine: Record "Sales Shipment Line";
        ReservEntry: Record "Reservation Entry";
        PostedInvoiceHeader: Record "Sales Invoice Header";
        PostedInvoiceLine: Record "Sales Invoice Line";

    begin
        if DocNo = '' then exit;
        SalesHeader.SetRange("Document Type", SalesHeader."Document Type"::Invoice);
        SalesHeader.SetRange("No.", DocNo);

        if not SalesHeader.FindFirst() then
            exit;

        if not TryDeleteSalesHeader(SalesHeader) then begin
            // Force delete lines first (Orphans prevention)
            SalesLine.SetRange("Document Type", SalesHeader."Document Type");
            SalesLine.SetRange("Document No.", SalesHeader."No.");
            SalesLine.DeleteAll(false);

            // CLEANUP ORPHANED RESERVATIONS (caused by skipping triggers)
            ReservEntry.SetRange("Source Type", 37); // Sales Line
            ReservEntry.SetRange("Source ID", SalesHeader."No.");
            ReservEntry.DeleteAll(false);

            SalesHeader.Delete(false); // Force delete if standard delete fails
        end;

        // Clean up any lingering Posted Invoice Headers (from previous Posts)
        if PostedInvoiceHeader.Get(DocNo) then begin
            if not TryDeletePostedInvoice(PostedInvoiceHeader) then begin
                PostedInvoiceLine.SetRange("Document No.", PostedInvoiceHeader."No.");
                PostedInvoiceLine.DeleteAll(false);
                PostedInvoiceHeader.Delete(false);
            end;
        end;

        // Also check by Pre-Assigned No in case the posted number was different but linked
        PostedInvoiceHeader.Reset();
        PostedInvoiceHeader.SetRange("Pre-Assigned No.", DocNo);
        if PostedInvoiceHeader.FindSet() then begin
            repeat
                PostedInvoiceLine.SetRange("Document No.", PostedInvoiceHeader."No.");
                PostedInvoiceLine.DeleteAll(false);
            until PostedInvoiceHeader.Next() = 0;
            PostedInvoiceHeader.DeleteAll(false);
        end;

        // Clean up any lingering Sales Shipment Headers (from previous Posts)
        // Check by Order No. (which usually matches Invoice No. in this flow) OR Pre-Assigned No.
        if SalesShipmentHeader.Get(DocNo) then begin
            SalesShipmentLine.SetRange("Document No.", SalesShipmentHeader."No.");
            SalesShipmentLine.DeleteAll(false);
            SalesShipmentHeader.Delete(false);
        end else begin
            // Also check if DocNo was the Order No.
            SalesShipmentHeader.Reset();
            SalesShipmentHeader.SetRange("Order No.", DocNo);
            if SalesShipmentHeader.FindSet() then
                repeat
                    SalesShipmentLine.SetRange("Document No.", SalesShipmentHeader."No.");
                    SalesShipmentLine.DeleteAll(false);
                until SalesShipmentHeader.Next() = 0;
            SalesShipmentHeader.DeleteAll(false);
        end;
    end;

    local procedure CreateSalesHeader(var SalesHeader: Record "Sales Header"; CustNo: Code[20]; DocNo: Code[20]; ExtDocNo: Code[35]; DocDate: Date; PostDate: Date; DueDate: Date)
    begin
        SalesHeader.Init();
        SalesHeader."Document Type" := SalesHeader."Document Type"::Invoice;
        SalesHeader."No." := DocNo;
        SalesHeader."Posting No." := DocNo;
        SalesHeader.SetHideValidationDialog(true); // Suppress "Posting Date Changed" dialogs
        SalesHeader.Insert(true);

        SalesHeader.Validate("Sell-to Customer No.", CustNo);
        if ExtDocNo <> '' then
            SalesHeader.Validate("External Document No.", ExtDocNo);
        // Set Posting Date BEFORE Document Date - BC's validation logic
        // resets Document Date when Posting Date is validated
        if PostDate <> 0D then
            SalesHeader.Validate("Posting Date", PostDate);
        if DocDate <> 0D then
            SalesHeader.Validate("Document Date", DocDate);
        if DueDate <> 0D then
            SalesHeader.Validate("Due Date", DueDate);

        SalesHeader.Modify(true);
    end;

    /// <summary>
    /// Creates a Sales Line from a typed request line record.
    /// </summary>
    local procedure CreateSalesLineFromRecord(SalesHeader: Record "Sales Header"; InvoiceRequestLine: Record "Superior Invoice Request Line"; LineNo: Integer)
    var
        SalesLine: Record "Sales Line";
    begin
        SalesLine.Init();
        SalesLine."Document Type" := SalesHeader."Document Type";
        SalesLine."Document No." := SalesHeader."No.";
        SalesLine."Line No." := LineNo;

        EvaluateLineType(SalesLine, InvoiceRequestLine."Line Type");

        SalesLine.Insert(true);

        if InvoiceRequestLine."Line Type Number" <> '' then
            SalesLine.Validate("No.", InvoiceRequestLine."Line Type Number");

        if InvoiceRequestLine."Quantity" <> 0 then
            SalesLine.Validate(Quantity, InvoiceRequestLine."Quantity");

        if InvoiceRequestLine."Unit Price" <> 0 then
            SalesLine.Validate("Unit Price", InvoiceRequestLine."Unit Price");

        if InvoiceRequestLine."Tax Code" <> '' then
            SalesLine.Validate("Tax Group Code", InvoiceRequestLine."Tax Code");

        // Handle Dimensions
        HandleDimensionsFromRecord(SalesLine, InvoiceRequestLine);

        SalesLine.Modify(true);
    end;

    /// <summary>
    /// Handles dimension assignment from typed record fields.
    /// </summary>
    local procedure HandleDimensionsFromRecord(var SalesLine: Record "Sales Line"; InvoiceRequestLine: Record "Superior Invoice Request Line")
    var
        TempDimSetEntry: Record "Dimension Set Entry" temporary;
    begin
        AddDimension(TempDimSetEntry, 'PROD MTH', InvoiceRequestLine."Prod Month");
        AddDimension(TempDimSetEntry, 'METER', InvoiceRequestLine."Meter Code");
        AddDimension(TempDimSetEntry, 'STATE LOC', InvoiceRequestLine."State Location");
        AddDimension(TempDimSetEntry, 'SDEAL#', InvoiceRequestLine."Sdeal Number");
        AddDimension(TempDimSetEntry, 'PDEAL#', InvoiceRequestLine."Pdeal Number");
        AddDimension(TempDimSetEntry, 'PIPE', InvoiceRequestLine."Pipe Code");

        if not TempDimSetEntry.IsEmpty() then
            UpdateLineDimensions(SalesLine, TempDimSetEntry);
    end;

    local procedure ProcessLines(SalesHeader: Record "Sales Header"; LinesArray: JsonArray)
    var
        LineToken: JsonToken;
        JLine: JsonObject;
        SalesLine: Record "Sales Line";
        LineTypeStr: Text;
        LineNo: Integer;
    begin
        LineNo := 0;
        foreach LineToken in LinesArray do begin
            JLine := LineToken.AsObject();
            LineNo += 10000;
            CreateSalesLine(SalesHeader, JLine, LineNo);
        end;
    end;

    local procedure CreateSalesLine(SalesHeader: Record "Sales Header"; JLine: JsonObject; LineNo: Integer)
    var
        SalesLine: Record "Sales Line";
        LineTypeStr: Text;
        LineTypeNo: Code[20];
        UnitPrice: Decimal;
        Quantity: Decimal;
        TaxCode: Code[20];
        DimSetID: Integer;
    begin
        SalesLine.Init();
        SalesLine."Document Type" := SalesHeader."Document Type";
        SalesLine."Document No." := SalesHeader."No.";
        SalesLine."Line No." := LineNo;

        LineTypeStr := GetJsonValue(JLine, 'LineType').AsText();
        EvaluateLineType(SalesLine, LineTypeStr);

        SalesLine.Insert(true);

        LineTypeNo := GetJsonValue(JLine, 'LineTypeNumber').AsCode();
        if LineTypeNo <> '' then
            SalesLine.Validate("No.", LineTypeNo);

        Quantity := GetJsonDecimal(JLine, 'Quantity');
        if Quantity <> 0 then
            SalesLine.Validate(Quantity, Quantity);

        UnitPrice := GetJsonDecimal(JLine, 'UnitPrice');
        if UnitPrice <> 0 then
            SalesLine.Validate("Unit Price", UnitPrice);

        TaxCode := GetJsonValue(JLine, 'TaxCode').AsCode();
        if TaxCode <> '' then
            SalesLine.Validate("Tax Group Code", TaxCode);

        // Handle Dimensions
        HandleDimensions(SalesLine, JLine);

        SalesLine.Modify(true);
    end;

    local procedure EvaluateLineType(var SalesLine: Record "Sales Line"; TypeStr: Text)
    begin
        case TypeStr of
            'Comment', '':
                SalesLine.Validate(Type, SalesLine.Type::" ");
            'G/L Account':
                SalesLine.Validate(Type, SalesLine.Type::"G/L Account");
            'Item':
                SalesLine.Validate(Type, SalesLine.Type::Item);
            'Resource':
                SalesLine.Validate(Type, SalesLine.Type::Resource);
            'Fixed Asset':
                SalesLine.Validate(Type, SalesLine.Type::"Fixed Asset");
            'Charge (Item)':
                SalesLine.Validate(Type, SalesLine.Type::"Charge (Item)");
            else
                Error('Invalid Line Type: %1', TypeStr);
        end;
    end;

    local procedure HandleDimensions(var SalesLine: Record "Sales Line"; JLine: JsonObject)
    var
        TempDimSetEntry: Record "Dimension Set Entry" temporary;
        DimMgt: Codeunit DimensionManagement;
        NewDimSetID: Integer;
    begin
        // Add dimensions to temp table
        AddDimension(TempDimSetEntry, 'PROD MTH', GetJsonValue(JLine, 'ProdMonth').AsText());
        AddDimension(TempDimSetEntry, 'METER', GetJsonValue(JLine, 'MeterCode').AsText());
        AddDimension(TempDimSetEntry, 'STATE LOC', GetJsonValue(JLine, 'StateLocation').AsText());
        AddDimension(TempDimSetEntry, 'SDEAL#', GetJsonValue(JLine, 'SdealNumber').AsText());
        AddDimension(TempDimSetEntry, 'PDEAL#', GetJsonValue(JLine, 'PdealNumber').AsText());
        AddDimension(TempDimSetEntry, 'PIPE', GetJsonValue(JLine, 'PipeCode').AsText());

        if not TempDimSetEntry.IsEmpty() then
            UpdateLineDimensions(SalesLine, TempDimSetEntry);
    end;

    local procedure AddDimension(var TempDimSetEntry: Record "Dimension Set Entry" temporary; DimCode: Code[20]; DimName: Text[50])
    var
        DimValueCode: Code[20];

    begin
        if DimName = '' then exit;

        // Check/Create Dimension Value
        DimValueCode := EnsureDimensionValueExists(DimCode, DimName);

        if TempDimSetEntry.Get(0, DimCode) then begin
            TempDimSetEntry."Dimension Value Code" := DimValueCode;
            TempDimSetEntry.Modify();
        end else begin
            TempDimSetEntry.Init();
            TempDimSetEntry."Dimension Set ID" := 0;
            TempDimSetEntry."Dimension Code" := DimCode;
            TempDimSetEntry."Dimension Value Code" := DimValueCode;
            TempDimSetEntry.Insert();
        end;
    end;

    procedure GetDimensionValueCode(DimCode: Code[20]; DimensionValueName: Text[50]): Code[20]
    var
        DimValue: Record "Dimension Value";
    begin
        DimValue.SetRange("Dimension Code", DimCode);
        DimValue.SetRange(Name, DimensionValueName);

        if DimValue.FindFirst() then
            exit(DimValue.Code);

        exit('');
    end;

    local procedure EnsureDimensionValueExists(DimCode: Code[20]; DimensionValueName: Text[50]): Code[20]
    var
        DimValue: Record "Dimension Value";
        DimCodeResult: Code[20];
    begin
        DimCodeResult := GetDimensionValueCode(DimCode, DimensionValueName);

        if DimCodeResult <> '' then begin
            exit(DimCodeResult);
        end else if StrLen(DimensionValueName) > 20 then begin
            Error(
                'Please enter a %1 dimension value with the name "%2" and make up your own unique code for it',
                DimCode,
                DimensionValueName
            );
        end else begin
            DimCodeResult := DimensionValueName;
            DimValue.Init();
            DimValue.Validate("Dimension Code", DimCode);
            DimValue.Validate(Code, DimCodeResult);
            DimValue.Validate(Name, DimensionValueName);
            DimValue.Insert(true);
            exit(DimCodeResult);
        end;
    end;

    local procedure UpdateLineDimensions(var SalesLine: Record "Sales Line"; var NewDims: Record "Dimension Set Entry" temporary)
    var
        TempDimSetEntry: Record "Dimension Set Entry" temporary;
        DimValue: Record "Dimension Value";
        DimMgt: Codeunit DimensionManagement;
        NewDimSetID: Integer;
    begin
        // 1. Load existing dimensions into a new temp table
        DimMgt.GetDimensionSet(TempDimSetEntry, SalesLine."Dimension Set ID");

        // 2. Update/Insert with new dimensions
        if NewDims.FindSet() then
            repeat
                // Get the Dimension Value ID from the Dimension Value table
                if DimValue.Get(NewDims."Dimension Code", NewDims."Dimension Value Code") then begin
                    if TempDimSetEntry.Get(TempDimSetEntry."Dimension Set ID", NewDims."Dimension Code") then begin
                        TempDimSetEntry."Dimension Value Code" := NewDims."Dimension Value Code";
                        TempDimSetEntry."Dimension Value ID" := DimValue."Dimension Value ID";
                        TempDimSetEntry.Modify();
                    end else begin
                        TempDimSetEntry.Init();
                        TempDimSetEntry."Dimension Set ID" := 0;
                        TempDimSetEntry."Dimension Code" := NewDims."Dimension Code";
                        TempDimSetEntry."Dimension Value Code" := NewDims."Dimension Value Code";
                        TempDimSetEntry."Dimension Value ID" := DimValue."Dimension Value ID";
                        TempDimSetEntry.Insert();
                    end;
                end;
            until NewDims.Next() = 0;

        // 3. Get new ID
        NewDimSetID := DimMgt.GetDimensionSetID(TempDimSetEntry);
        SalesLine.Validate("Dimension Set ID", NewDimSetID);
    end;

    // Helper Functions
    local procedure GetJsonValue(JObject: JsonObject; KeyName: Text): JsonValue
    var
        JToken: JsonToken;
    begin
        if JObject.Get(KeyName, JToken) then
            if JToken.IsValue() then
                exit(JToken.AsValue());
        exit(JToken.AsValue()); // Return null/empty value
    end;

    local procedure GetJsonDate(JObject: JsonObject; KeyName: Text): Date
    var
        Val: JsonValue;
        Dt: Date;
    begin
        Val := GetJsonValue(JObject, KeyName);
        if not Val.IsNull() then
            if Val.AsDate() <> 0D then
                exit(Val.AsDate());
        exit(0D);
    end;

    local procedure GetJsonDecimal(JObject: JsonObject; KeyName: Text): Decimal
    var
        Val: JsonValue;
    begin
        Val := GetJsonValue(JObject, KeyName);
        if not Val.IsNull() then
            exit(Val.AsDecimal());
        exit(0);
    end;

    procedure RemoveInvoiceOnError(InvoiceNo: Code[20])
    var
        SalesHeader: Record "Sales Header";
        SalesLine: Record "Sales Line";
        PostedInvoiceHeader: Record "Sales Invoice Header";
        PostedInvoiceLine: Record "Sales Invoice Line";
        SalesShipmentHeader: Record "Sales Shipment Header";
        SalesShipmentLine: Record "Sales Shipment Line";
        ReservEntry: Record "Reservation Entry";
    begin
        // Delete unposted invoice if it exists
        SalesHeader.SetRange("Document Type", SalesHeader."Document Type"::Invoice);
        SalesHeader.SetRange("No.", InvoiceNo);
        if SalesHeader.FindFirst() then begin
            if not TryDeleteSalesHeader(SalesHeader) then begin
                // Force delete lines first
                SalesLine.SetRange("Document Type", SalesHeader."Document Type");
                SalesLine.SetRange("Document No.", SalesHeader."No.");
                SalesLine.DeleteAll(false);

                // CLEANUP ORPHANED RESERVATIONS (caused by skipping triggers)
                ReservEntry.SetRange("Source Type", 37); // Sales Line
                ReservEntry.SetRange("Source ID", SalesHeader."No.");
                ReservEntry.DeleteAll(false);

                // If soft delete fails, try hard delete
                SalesHeader.Delete(false);
            end;
        end;

        // Also delete any orphaned posted invoice (in case Sales-Post partially succeeded)
        // Check by direct No. match
        if PostedInvoiceHeader.Get(InvoiceNo) then begin
            if not TryDeletePostedInvoice(PostedInvoiceHeader) then begin
                // Force delete lines first
                PostedInvoiceLine.SetRange("Document No.", PostedInvoiceHeader."No.");
                PostedInvoiceLine.DeleteAll(false);

                PostedInvoiceHeader.Delete(false); // Force delete if trigger blocked
            end;
        end;

        // Check by Pre-Assigned No. (original invoice number before posting)
        PostedInvoiceHeader.Reset();
        PostedInvoiceHeader.SetRange("Pre-Assigned No.", InvoiceNo);
        if not PostedInvoiceHeader.IsEmpty() then begin
            // Iterate to delete lines for each found header
            if PostedInvoiceHeader.FindSet() then
                repeat
                    PostedInvoiceLine.SetRange("Document No.", PostedInvoiceHeader."No.");
                    PostedInvoiceLine.DeleteAll(false);
                until PostedInvoiceHeader.Next() = 0;

            PostedInvoiceHeader.DeleteAll(false); // Direct force delete for set to be safe
        end;

        // Final Safety Net: Delete any truly orphaned Posted Invoice Lines for this ID
        // This handles cases where the Header is already gone but lines remain (Corruption)
        PostedInvoiceLine.Reset();
        PostedInvoiceLine.SetRange("Document No.", InvoiceNo);
        if not PostedInvoiceLine.IsEmpty() then
            PostedInvoiceLine.DeleteAll(false);

        // CLEANUP SALES SHIPMENTS (New Addition)
        if SalesShipmentHeader.Get(InvoiceNo) then begin
            SalesShipmentLine.SetRange("Document No.", SalesShipmentHeader."No.");
            SalesShipmentLine.DeleteAll(false);
            SalesShipmentHeader.Delete(false);
        end;

        SalesShipmentHeader.Reset();
        SalesShipmentHeader.SetRange("Order No.", InvoiceNo);
        if SalesShipmentHeader.FindSet() then begin
            repeat
                SalesShipmentLine.SetRange("Document No.", SalesShipmentHeader."No.");
                SalesShipmentLine.DeleteAll(false);
            until SalesShipmentHeader.Next() = 0;
            SalesShipmentHeader.DeleteAll(false);
        end;
    end;

    [TryFunction]
    local procedure TryDeleteSalesHeader(var SalesHeader: Record "Sales Header")
    begin
        SalesHeader.Delete(true);
    end;

    [TryFunction]
    local procedure TryDeletePostedInvoice(var SalesInvoiceHeader: Record "Sales Invoice Header")
    begin
        SalesInvoiceHeader.Delete(true);
    end;

    procedure CheckAccountingPeriod(var SalesHeader: Record "Sales Header"): Boolean
    var
        AccountingPeriod: Record "Accounting Period";
        PostingDate: Date;
        FirstOfMonth: Date;
        LastOfMonth: Date;
    begin
        PostingDate := SalesHeader."Posting Date";

        if PostingDate = 0D then
            exit(false);

        FirstOfMonth := CalcDate('<-CM>', PostingDate);
        // set to find earliest possible posting date
        AccountingPeriod.Reset();
        AccountingPeriod.SetRange(Closed, false);
        AccountingPeriod.SetFilter("Starting Date", '>=%1', FirstOfMonth);

        if AccountingPeriod.FindFirst() then begin
            LastOfMonth := CalcDate('<CM>', AccountingPeriod."Starting Date");
            SalesHeader.Validate("Posting Date", LastOfMonth);
            exit(true);
        end;
        exit(false);
    end;

    [TryFunction]
    local procedure TryPostSalesInvoice(SalesHeader: Record "Sales Header")
    var
        SalesPost: Codeunit "Sales-Post";
    begin
        SalesPost.Run(SalesHeader);
    end;
}