using System.Diagnostics;
using System.Security.Cryptography;
using System.Text.Json;
using Fast.Shared.Logic;
using Fast.Shared.Logic.FileService;
using Fast.Shared.Logic.Valuation;
using Fast.Shared.Logic.ValuationByPath;
using Fast.Shared.Models;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.OData.Query;

namespace Test.Shared;

public class GeneralTest
{
    [Test]
    public async Task AuthSecretGenerator()
    {
        RandomNumberGenerator cryptoRandomDataGenerator = RandomNumberGenerator.Create();
        byte[] buffer = new byte[16];
        cryptoRandomDataGenerator.GetBytes(buffer);
        string uniq = Convert.ToHexString(buffer);
        await Assert.That(uniq.Length).IsGreaterThan(0);
    }

    [Test]
    public async Task ValTest()
    {
        var valuater = new Valuater();
        var x = (await valuater.GetValuationValues("CRO2558")).ToList();
        var f = x.First();
        var p = f.CrudeRollPrice;
        await Assert.That(p).IsNotEqualTo(0);
    }

    [Test]
    public async Task PathValTest()
    {
        var valParams = new ValParams();
        valParams.TransactionTypeIds.Add((int)Enums.TransactionType.PhysicalGas);
        var startDate = DateOnly.Parse("6/1/2025");
        var endDate = DateOnly.Parse("8/31/2025");
        valParams.PositionDateRanges.Add(new DateRange(Enums.DateStyle.DateRange, startDate, endDate));
        valParams.IncludeBasisInContractPrice = false;

        var sw = Stopwatch.StartNew();
        var valuater = new PathValuaterGas();
        var vals = await valuater.GetValuationValues(valParams);
        sw.Stop();

        // Log timing and result summary
        GS.LogInfo($"PathValTest: {sw.Elapsed}");
        GS.LogInfo($"PathValTest results count: {vals.Count}");
        System.Threading.Thread.Sleep(1000);

        await Assert.That(vals.Count).IsGreaterThan(0);
    }

    internal static ODataQueryOptions<T> GetOdataOptions<T>(string routeUrl)
    {
        string path = "../.vscode/settings.json";
        string settingsText = File.ReadAllText(path);
        var settings = JsonSerializer.Deserialize<Dictionary<string, object>>(settingsText);
        string? port = "";
        if (settings != null)
            port = settings["fast.module.name"].ToString();

        var baseOdataStr = "https://127.0.0.1:443" + port + @"/odata";
        var uri = new Uri($"{baseOdataStr}/{routeUrl}");
        var httpContext = new DefaultHttpContext();

        HttpRequest request = httpContext.Request;
        request.Method = "GET";
        request.Host = new HostString(uri.Host, uri.Port);
        request.Path = uri.LocalPath;
        request.QueryString = new QueryString(uri.Query);

        var edmModel = Fast.Web.Helper.GetCustomEdmModel();
        ODataQueryContext context = new(edmModel, typeof(T), new Microsoft.OData.UriParser.ODataPath());
        ODataQueryOptions<T> options = new(context, request);

        return options;
    }
}

