import { inject, Injectable } from '@angular/core';
import { HttpClient } from '@angular/common/http';
import { OdataGridHelper } from '../_shared/utils/odata-grid-helper';
import { Observable } from 'rxjs';
import { State } from '@progress/kendo-data-query';
import { GridDataResult } from '@progress/kendo-angular-grid';
import { IdName, convertDatesToDateOnlyStrings } from '../_shared/utils/util';

export interface Item {
  SalesTaxRateId: number;
  StateName: string;
  CountyName: string;
  Rate: number;
  InactiveDate: Date;
}

export interface Detail {
  salesTaxRateId: number;
  countyId: number;
  stateId: number;
  rate: number;
  inactiveDate: Date;
}

export enum SaveType {
  New = 1,
  Normal = 2
}

export interface countyItem {
  countyId: number;
  countyName: string;
  stateId: number;
  stateName: string;
}

export interface RequiredData {
  hasModifyPermission: boolean;
  counties: countyItem[];
  states: IdName[];
}

@Injectable({
  providedIn: 'root'
})
export class SalesTaxRateService extends OdataGridHelper {
  private baseUrl = `${window.location.origin}/api/SalesTaxRate`;
  http = inject(HttpClient);

  requiredData$ = this.http.get<RequiredData>(`${this.baseUrl}/getRequiredData`)

  getItems(state: State): Observable<GridDataResult> {
    return this.fetch(`GetSalesTaxRateItems`, state, null, false)
  }

  getDetail(id: number): Observable<Detail> {
    const url = `${this.baseUrl}/GetDetail?id=${id}`;
    return this.http.get<Detail>(url);
  }

  saveDetail(detail: Detail, saveType: SaveType): Observable<number> {
    const url = `${this.baseUrl}/SaveDetail?saveType=${saveType}`;
    detail = convertDatesToDateOnlyStrings(detail, ['inactiveDate']);
    return this.http.put<number>(url, detail);
  }

  deleteDetail(id: number) {
    const url = `${this.baseUrl}/DeleteDetail/${id}`;
    return this.http.delete(url);
  }

  exportItems(state: State, fileNameOriginal: string) {
    return this.fetchExport(`GetSalesTaxRateItems`, state, null, fileNameOriginal);
  }
}
