import { Component, computed, input } from "@angular/core";

interface PipeLine {
  y: number;
  leftClass: string;
  rightClass: string;
}

@Component({
  selector: 'pipe-icon',
  standalone: true,
  template: `
    <svg
      [attr.width]="size()"
      [attr.height]="size()"
      viewBox="0 0 24 24"
      fill="none"
      stroke="currentColor"
      stroke-width="1"
    >
      @for (line of lines(); track line.y) {
        <line x1="5.4" [attr.y1]="line.y" x2="18.7" [attr.y2]="line.y" />
        <circle [class]="line.leftClass" cx="4" [attr.cy]="line.y" r="1.5" />
        <circle [class]="line.rightClass" cx="20" [attr.cy]="line.y" r="1.5" />
      }
    </svg>
  `,
  styles: `
    :host {
      display: inline-flex;
      align-items: center;
      justify-content: center;
    }
    svg {
      display: block;
    }
  `,
})
export class PipeIconComponent {
  pipeCount = input<number>(1);
  highlightLeg = input<number>(0);
  size = input<number>(34);

  // Y positions for each pipe configuration
  private readonly pipeYPositions: Record<number, number[]> = {
    1: [12],
    2: [9.5, 14.5],
    3: [8, 12, 16],
    4: [6, 10, 14, 18],
    5: [4, 8, 12, 16, 20],
  };

  lines = computed<PipeLine[]>(() => {
    const count = Math.min(Math.max(this.pipeCount(), 1), 5);
    const highlight = this.highlightLeg();
    const yPositions = this.pipeYPositions[count] || this.pipeYPositions[1];

    return yPositions.map((y, index) => {
      const lineNum = index + 1;
      let leftClass: string;
      let rightClass: string;

      if (highlight > 0 && lineNum === highlight) {
        leftClass = `
          fill-base-blue-1000 dark:fill-base-yellow-250
          stroke-base-blue-1000 dark:stroke-base-yellow-250
          stroke-1
        `;
        rightClass = 'fill-transparent stroke-[0.4]';
      } else if (highlight > 0 && lineNum > highlight) {
        leftClass = 'fill-transparent stroke-[0.4]';
        rightClass = 'fill-transparent stroke-[0.4]';
      } else {
        leftClass = 'fill-current stroke-[0.4]';
        rightClass = 'fill-current stroke-[0.4]';
      }

      return { y, leftClass, rightClass };
    });
  });
}
